/*  -*- c++ -*-  (for Emacs)
 *
 *  svm_recogniser.h
 *  Digest
 * 
 *  Created by Adrian Bickerstaffe on Wed Jan 18 2006.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef SVM_RECOGNISER_H
#define SVM_RECOGNISER_H

#include <iostream>

#ifdef DEBUG_RECOGNIZER	
	#include <fstream>
#endif

#include "stlrecogniserinterface.h"
#include "multiclass_svm.h"

using namespace std;

////////////////////////

// Template class to represent a generic SVM gesture recogniser
// for GestureLab.
template<class T>
class svm_recogniser : public StlRecogniserInterface
{
public:
	enum StrokeType{closed, horizontal, vertical, diagonal};
	
	svm_recogniser(void);
	~svm_recogniser(void);

	const map<string, string> & defaultParams(void) const;

	bool initTraining(const list<string> &, const map<string, string> &);
	bool examineSample(const StlFeatureVec &, const set<int> &);
	bool finaliseTraining(void);

	bool writeModelFile(const string &);
	bool readModelFile(const string &);

	StlClassProbabilities classify(const StlFeatureVec &);
	
	int classifyWithArray(const float*, int);
	float * classifyProbWithArray(const float *, int, int *);
	StlStroke flatten(const StlStrokeList & strokes);
	
private:
	T svm_model;					// the model to train and save
	svm_dataset training_dataset;	// the set of training data

#ifdef DEBUG_RECOGNIZER
	unsigned long int gesture_num;
	ofstream debug_file;
	
	int find_mostlikely_class(const StlClassProbabilities &);
#endif
	
	// in-recogniser feature related data
	int f_num_strokes;
	double f_endpoint_distance, f_endpoint_angle, f_penup_time;
	
	StlFeatureVec add_features(const StlFeatureVec &);
	double average_endpoint_distance(const StlStrokeList &);
	double average_endpoint_angle(const StlStrokeList &);
	double portion_pen_up(const StlStrokeList &);

	// stroke pre-processing functions
	StlStroke unit_resize(const StlStroke &);
	
	StlStrokeList arrayToStrokes(const float *, int);
	StlStroke arrayToStroke(int &, const float *, int);
	float * probMapToArray(const StlClassProbabilities &, int *);
	
	StlStroke sub_sample(const StlStroke &);
	StlStroke resolution_resample(const StlStroke & stroke);	
	StlStrokePoint interpolate_by_time(const StlStrokePoint &,
									   const StlStrokePoint &, double);	

#ifdef ESTIMATE_GESTURELAB_RESOLUTION
	void estimate_sampling_rate(const StlStrokeList & strokes);
#endif
		
	StlStroke reverse(const StlStroke &, const long double &);
	StlStroke reverse_points(const StlStroke &);
	void swap_coordinates(StlStrokePoint &, StlStrokePoint &);
	StrokeType find_stroke_type(const long double &, const long double &,
								const long double &);
										 
	StlStrokeList reorder(const StlStrokeList &);
	StlStroke remove_duplicates(const StlStroke &);
	StlStroke remove_jitter(const StlStroke &);
	StlStroke smooth(const StlStroke &);
	StlStroke equidistant_resample(const StlStroke &);
	StlStroke remove_hooklet(const StlStroke &);
	bool direction_change_hook(const StlStroke &,
							   const StlStroke::const_iterator &);
	
	StlStrokePoint interpolate_by_distance(const StlStrokePoint &,
										   const StlStrokePoint &, double);

	
	void set_head_tail(const StlStroke &, StlStroke::const_iterator &,
					   StlStroke::const_iterator &);
};

/////////////////

#include "svm_recogniser.ipp"

#endif  // !SVM_RECOGNISER_H
