/*  -*- c++ -*-  (for Emacs)
 *
 *  common.cpp
 *  Digest
 * 
 *  Created by Adrian Bickerstaffe on 17/5/2006.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <algorithm>
#include <iostream>
#include <fstream>
#include <vector>
#include <cmath>
#include <iomanip>		// for formatted output

#include "common.h"
#include "standardfeatures.h"	

using namespace std;

///// Operator functions /////

// Operator used for sorting strokes according to their pen-down time.
bool operator<(const StlStroke & a, const StlStroke & b)
{
	// if b has no points, b goes first
	if(b.empty())
		return false;
	// if a has no points, a goes first
	else if(a.empty())
		return true;
	// return whichever stroke has the "earliest" first point
	else
		return a.front().milliTime < b.front().milliTime;
}

// Operator used for sorting stroke points according to their
// x-coordinates.
bool x_compare(const StlStrokePoint & a, const StlStrokePoint & b)
{
	return a.x < b.x;
}

// Operator used for sorting stroke points according to their
// y-coordinates.
bool y_compare(const StlStrokePoint & a, const StlStrokePoint & b)
{
	return a.y < b.y;
}

//////////////////////////

// Function to extract a default set of features from a stroke.
StlFeatureVec extract_features(const StlStroke & the_stroke)
{
	StlFeatureVec feature_set;
	bool ok;
	
	// feature #1
	SinFirstToLastAngleFeature SinFirstToLastAngle;
	feature_set.push_back(SinFirstToLastAngle.classCalcValue(the_stroke, &ok));	
	// feature #2
	DurationFeature Duration;
	feature_set.push_back(Duration.classCalcValue(the_stroke, &ok));
	// feature #3
	TotalSqrdAngleTraversedFeature TotalSqrdAngleTraversed;
	feature_set.push_back(TotalSqrdAngleTraversed.classCalcValue(the_stroke, &ok));	
	// feature #4	
	MaxSqrdSpeedFeature MaxSqrdSpeed;
	feature_set.push_back(MaxSqrdSpeed.classCalcValue(the_stroke, &ok));	
	// feature #5	
	BBoxDiagAngleFeature BBoxDiagAngle;
	feature_set.push_back(BBoxDiagAngle.classCalcValue(the_stroke, &ok));
	// feature #6	
	CosInitAngleFeature CosInitAngle;
	feature_set.push_back(CosInitAngle.classCalcValue(the_stroke, &ok));
	// feature #7
	SinInitAngleFeature SinInitAngle;
	feature_set.push_back(SinInitAngle.classCalcValue(the_stroke, &ok));
	// feature #8
	TotalAngleTraversedFeature TotalAngleTraversed;
	feature_set.push_back(TotalAngleTraversed.classCalcValue(the_stroke, &ok));
	// feature #9
	BBoxDiagLengthFeature BBoxDiagLength;
	feature_set.push_back(BBoxDiagLength.classCalcValue(the_stroke, &ok));
	// feature #10
	TotalGestureLengthFeature TotalGestureLength;
	feature_set.push_back(TotalGestureLength.classCalcValue(the_stroke, &ok));	
	// feature #11
	TotalAbsAngleTraversedFeature TotalAbsAngleTraversed;
	feature_set.push_back(TotalAbsAngleTraversed.classCalcValue(the_stroke, &ok));
	// feature #12	
	FirstToLastDistFeature FirstToLastDist;
	feature_set.push_back(FirstToLastDist.classCalcValue(the_stroke, &ok));
	// feature #13
	CosFirstToLastAngleFeature CosFirstToLastAngle;
	feature_set.push_back(CosFirstToLastAngle.classCalcValue(the_stroke, &ok));
	
	return feature_set;
}

// Function to find the mean x and y coordinate values of a stroke.
void find_coordinate_means(const StlStroke & the_stroke,
						   double & mean_x, double & mean_y)
{
	// cumulative x and y sums
	double x_sum = 0.0, y_sum = 0.0;
	StlStroke::const_iterator iter;	
	// for each point in the stroke
	for(iter = the_stroke.begin(); iter != the_stroke.end(); iter++)
	{
		// accumulate the x and y values
		x_sum += (*iter).x;
		y_sum += (*iter).y;
	}
	// calculate the mean
	mean_x = x_sum / the_stroke.size();
	mean_y = y_sum / the_stroke.size();
}

// Function to calculate the length of a stroke.
double calc_length(const StlStroke & stroke)
{
	double length = 0.0;
	StlStroke::const_iterator iter = stroke.begin();
	StlStrokePoint prev_point = *iter, curr_point;

	// for each stroke point
	for(iter++; iter != stroke.end(); iter++)
	{
		curr_point = *iter;	
		// accumulate the absolute difference in each direction	
		length += hypot(curr_point.x - prev_point.x,
						curr_point.y - prev_point.y);

		prev_point = curr_point;
	}
	
	return length;
}

// Function calculate the relative length of the stroke; relative length
// is defined as the ratio of distance travelled to distance between
// stroke endpoints.
double calc_relative_length(const StlStroke & stroke)
{
	// calculate the length of the stroke
	double distance_travelled = calc_length(stroke);
	// calculate the distance between the stroke endpoints
	double endpoint_distance = hypot(stroke.back().x - stroke.front().x,
									 stroke.back().y - stroke.front().y);

	// avoid divide by zero error when calculating the ratio
	if(endpoint_distance != 0)
		return distance_travelled / endpoint_distance;
	else
		return 0;
}

// Function returns the angle between p0, p1, and p2.  The angle is returned
// in degrees and is calculated using the cosine rule.
double inside_angle(const StlStrokePoint & p0, const StlStrokePoint & p1,
					const StlStrokePoint & p2)
{
	// edge lengths and result
	double a, b, c, angle;
	// calculate the triangle edge lengths
	a = hypot(p2.x - p0.x, p2.y - p0.y);
	b = hypot(p2.x - p1.x, p2.y - p1.y);
	c = hypot(p1.x - p0.x, p1.y - p0.y);
	// use the cosine rule to calculate the angle in degrees
	angle = (b * b + c * c - a * a) / (2.0 * b * c);
	angle = acos(angle) * (180.0 / M_PI);

	return angle;
}

// Function to output gesture x, y, and time data.  A gesture may be one or more strokes.
void print_gesture(const StlStrokeList & strokes, ostream & outputStream)
{
	int stroke_num = 1;
	StlStrokeList::const_iterator strokelist_iter;
	
	// For each stroke in the gesture
	for(strokelist_iter = strokes.begin(); strokelist_iter != strokes.end(); 
		strokelist_iter++, stroke_num++)
	{
		outputStream << "Stroke number " << stroke_num << " of " 
					 << static_cast<int>(strokes.size()) << endl;
		print_stroke(*strokelist_iter, outputStream);
	}
}

// Function to output stroke x, y, and time data.
void print_stroke(const StlStroke & stroke, ostream & outputStream)
{
	StlStroke::const_iterator stroke_iter;
	int point_num = 1;
	
	outputStream << "Num.\tTime\tx\ty" << endl
				 << "----------------------------" << endl;
	
	// output the x, y and time information about the stroke
	for(stroke_iter = stroke.begin(); stroke_iter != stroke.end(); stroke_iter++, point_num++)
		outputStream << point_num << "\t" << (*stroke_iter).milliTime << "\t" << (*stroke_iter).x  
					 << "\t" <<  (*stroke_iter).y << endl;
	
	outputStream << endl;
}

// Function to output stroke x, y, and time data.
void print_features(const StlFeatureVec & features, ostream & outputStream)
{
	StlFeatureVec::const_iterator feature_iter;
	int feature_num = 1;	
	
	// setup the output stream for pretty formatting
	outputStream << setprecision(6) << fixed;
	
	outputStream << "Features extracted:" << endl;
	// for each feature
	for(feature_iter = features.begin(); feature_iter != features.end(); feature_iter++, feature_num++)
	{
		outputStream << noshowpos << feature_num << ": ";
		outputStream << showpos << *feature_iter << "\t";
		
		// print 5 to a line
		if(feature_num % 5 == 0)
			outputStream << endl;
	}

	outputStream << endl << endl;
	
	// reset the output stream
	outputStream.unsetf(ios::fixed | ios::scientific);
	outputStream << noshowpos;
}

// Function to print a table of classification probabilities
void print_classification_probs(const StlClassProbabilities & class_probs, ostream & outputStream)
{
	int feature_num = 1;
	StlClassProbabilities::const_iterator iter;
	
	// setup the output stream for pretty formatting
	outputStream << setprecision(6) << fixed;
	
	outputStream << "Classfication probabilities:" << endl; 
	for(iter = class_probs.begin(); iter != class_probs.end(); iter++, feature_num++)
	{
		outputStream << iter->first << ": " << iter->second << "\t";
		
		if(feature_num % 5 == 0)
			outputStream << endl;
	}
	
	outputStream << endl << endl;
	
	// reset the output stream
	outputStream.unsetf(ios::fixed | ios::scientific);
}

//////////////////////////


// Default bounding box constructor sets the default box
// sets the box limits to zero.
bounding_box::bounding_box(void)
{
	x_low = y_low = x_high = y_high = 0.0;
}

// Constructor sets the limits of the bounding box given
// a stroke.
bounding_box::bounding_box(const StlStroke & stroke)
{
	find_limits(stroke);
}

// Constructor sets the limits of the bounding box given
// each limit point.
bounding_box::bounding_box(const StlStrokeCoordT & the_x_low,
						   const StlStrokeCoordT & the_y_low, 
						   const StlStrokeCoordT & the_x_high,
						   const StlStrokeCoordT & the_y_high)
{
	// if the boundary vertices are sane
	if(check_vertices(the_x_low, the_y_low, the_x_high, the_y_high))
	{
		x_low = the_x_low;
		y_low = the_y_low;
		x_high = the_x_high;
		y_high = the_y_high;
	}
	// else dont set the vertices and issue a warning
	else
		cout << "Warning:- bounding box vertices did not pass sanity checks, "
			 << "ignoring constructor parameters..." << endl;
}

// Function to sanity check bounding box defining vertices.
bool bounding_box::check_vertices(const StlStrokeCoordT & the_x_low,  
								  const StlStrokeCoordT & the_y_low, 
								  const StlStrokeCoordT & the_x_high,
								  const StlStrokeCoordT & the_y_high) const

{
	// if the limits are sane, return true
	if(the_x_high > the_x_low && the_y_high > the_y_low)
		return true;
	else
		return false;
}

// Function to find the bounding box defining vertices of a given stroke.
// The vertices and edges on the bounding box never overlap the stroke.
void bounding_box::find_limits(const StlStroke & stroke)
{
	// cannot sort a list, so create a vector typed copy
	vector<StlStrokePoint> temp_stroke(stroke.size());
	
	// copy the stroke points into the vector
	copy(stroke.begin(), stroke.end(), temp_stroke.begin());
	// sort the vector typed stroke by its x-coordinates
	sort(temp_stroke.begin(), temp_stroke.end(), x_compare);
	// set the low and high points (make the bbox non-inclusive)
	x_low = temp_stroke.front().x - 1;
	x_high = temp_stroke.back().x + 1;
	// sort the vector typed stroke by its y-coordinates
	sort(temp_stroke.begin(), temp_stroke.end(), y_compare);
	// set the low and high points (make the bbox non-inclusive)
	y_low = temp_stroke.front().y - 1;
	y_high = temp_stroke.back().y + 1;	
}

// Function returns the area of the bounding box.
 double bounding_box::area(void) const
{
	return (x_high - x_low) * (y_high - y_low);
}

// Function to calculate the diagonal length of a bounding box.
long double bounding_box::diagonal_length(void) const
{
	// return the Euclidian distance between the defining vertices
	return hypot(x_high - x_low, y_high - y_low);
}

// Function to calculate the angle between the top edge of the bounding box
// and the last point of the stroke.
long double bounding_box::top_edge_angle(const StlStroke & stroke) const
{
	long double result;
	long double left_corner_dist, right_corner_dist, top_edge_dist; 
	// extract the coordinates of the last point in the stroke
	
//	cout << "Bounding box of this stroke has edges: ("
//		 << x_low << ", " << y_low << ") and ("
//		 << x_high << ", " << y_high << ")" << endl;
		 
	StlStrokeCoordT endpoint_x = stroke.back().x;
	StlStrokeCoordT endpoint_y = stroke.back().y;
	
//	cout << "Last point of this stroke has coordinates: ("
//		 << endpoint_x << ", " << endpoint_y << ")" << endl;
		 
	// calculate the length of the bounding box's top edge
	top_edge_dist = x_high - x_low;
//	cout << "Top edge length: " << top_edge_dist << endl;
	// calculate the distance to this point from the top-left corner
	left_corner_dist = hypot(x_low - endpoint_x, y_low - endpoint_y);
//	cout << "Left corner distance: " << left_corner_dist << endl;
	// calculate the distance to this point from the top-right corner							   
	right_corner_dist = hypot(x_high - endpoint_x, y_low - endpoint_y);
//	cout << "Right corner distance: " << right_corner_dist << endl;
	
	// calculate the angle using the cosine rule
	result = ((left_corner_dist * left_corner_dist) + 
			  (top_edge_dist * top_edge_dist) -
			  (right_corner_dist * right_corner_dist)) / 
			  (2.0 * left_corner_dist * top_edge_dist);
	
//	cout << "Result before acos() is: " << result << endl;
	result = acos(result);	

//	cout << "Top edge angle is " << result << " radians ("
//		 << result * (180.0 / M_PI) << " degrees)." 
//		 << endl << endl;

	return result;
}

// Function to find which quadrant within the bounding box a given
// point is located.
int bounding_box::find_quadrant(const StlStrokePoint & point) const
{
	// calculate the side lengths of each quadrant
	double x_quadrant_len = (x_high - x_low) / 2.0;
	double y_quadrant_len = (y_high - y_low) / 2.0;
	// detect the north-west quadrant
	if(point.x <= x_low + x_quadrant_len &&
	   point.y <= y_low + y_quadrant_len)
	   return NW;
	// detect the north-east quadrant
	else if(point.x > x_low + x_quadrant_len && 
			point.y <= y_low + y_quadrant_len)
		return NE;
	// detect the south-west quadrant
	else if(point.x <= x_low + x_quadrant_len &&
			point.y > y_low + y_quadrant_len)
		return SW;
	// detect the south-east quadrant
	else if(point.x > x_low + x_quadrant_len &&
			point.y > y_low + y_quadrant_len)
		return SE;
	// catch-all, should never reach this point if the bounding box
	// is created properly
	else
		return NW;
}

// Function to find which quadrant within the bounding box a given
// point is located.
int bounding_box::classify_point_density(const StlStroke & the_stroke) const
{
	StlStroke::const_iterator iter;
	// vertical length of the bounding box
	double vertical_length = y_high - y_low;
	// end of the top segment and middle segment
	double top_segment_end, middle_segment_end;
	// frequency counts of stroke points in each segment
	map<string, int> frequency_counts;
	// assign 40% of the vertical length to the top and bottom segments
	top_segment_end = y_low + 0.4 * vertical_length;
	// assign 20% of the vertical length to the middle segment
	middle_segment_end = y_low + 0.6 * vertical_length;
	// initialise frequency counts
	frequency_counts["upper"] = 0;
	frequency_counts["middle"] = 0;
	frequency_counts["lower"] = 0;
	// for each point of the stroke
	for(iter = the_stroke.begin(); iter != the_stroke.end(); iter++)
	{
		// extract the y-coordinate
		double curr_y = (*iter).y;
		// if the point falls in the top segment, record the event
		if(curr_y <= top_segment_end)
			frequency_counts["upper"]++;
		// if the point falls in the middle segment
		else if(curr_y > top_segment_end && curr_y <= middle_segment_end)
			frequency_counts["middle"]++;
		// if the point falls in the bottom segment
		else
			frequency_counts["lower"]++;
	}
	// if the density is equal across segments
	if(frequency_counts["upper"] == frequency_counts["middle"] &&
	   frequency_counts["middle"] == frequency_counts["lower"])
	   return equal;
	// if the upper segment is the most dense
	else if(frequency_counts["upper"] > frequency_counts["lower"])
		return upper;
	// if the lower segment is the most dense
	else if(frequency_counts["upper"] < frequency_counts["lower"])
		return lower;
	// catch-all case, should not reach here in the bounding box is
	// correctly setup
	else
		return equal;
}
