/*  -*- c++ -*-  (for Emacs)
 *
 *  standardfeatures.cpp
 *  Digest
 * 
 *  Created by Aidan Lane on Mon Jul 11 2005.
 *  Copyright (c) 2005-2006 Optimisation and Constraint Solving Group,
 *  Monash University. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

// TODO: CHECK AND OPTIMISE FEATURES!!!!

#include "standardfeatures.h"

#include <assert.h>
#include <cmath>

using namespace std;


typedef StlFeatureResultT FeatureCalcT;


// U T I L I T I E S //
inline void getCoordMinAndMaxs( const StlStroke& stroke,
				StlStrokeCoordT& xmin, StlStrokeCoordT& xmax,
				StlStrokeCoordT& ymin, StlStrokeCoordT& ymax )
{
  StlStroke::const_iterator it = stroke.begin();
  // Step 1. init max and mins with first point
  //         - prevent checks of point being 1st in the loop
  if ( it != stroke.end() ) {
    xmin = xmax = (*it).x;
    ymin = ymax = (*it).y;
  }
  // Step 2. find max and mins across all points
  while ( it != stroke.end() ) {
    const StlStrokePoint& pt = *it;
    if ( pt.x < xmin ) xmin = pt.x;
    else if ( pt.x > xmax ) xmax = pt.x;
    if ( pt.y < ymin ) ymin = pt.y;
    else if ( pt.y > ymax ) ymax = pt.y;
    ++it;
  }
}


/*!
  \b Note: For the sake of performance, it's asserted that \em ok is non-null
           (i.e. the null-check will be stripped when compiled in release mode).
*/
inline double safe_div( double x, double y, bool* ok )
{
  assert( ok );
  if ( y == 0.0 ) {
    *ok = false;
    return 0.0;
  }
  *ok = true;
  return x / y;
}



/*!
  \class CosInitAngleFeature
  
  \brief The CosInitAngleFeature class calculates the cosine of the initial
         angle of the given stroke.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
	   \code f1 = cos a = (x2-x0) / sqrt( (x2-x0)^2 + (y2-y0)^2 ) \endcode
 
  \b Note: "The initial angle features, f1 and f2, are computed from the first and
           third mouse point because the result is generally less noisy than when
           computed from the first two points." - Bottom of first column, page 333.
*/

/*!
  \b Note: \em ok is true iff there are three or more points in the given stroke.
 */
StlFeatureResultT
CosInitAngleFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{
  StlFeatureResultT v = 0.0;
  bool resultOk = false;
  if ( stroke.size() >= 3 ) {
    FeatureCalcT dX = ( stroke[0].x      // 1st x
			- stroke[2].x ); // 3rd x
    FeatureCalcT dY = ( stroke[0].y      // 1st y
			- stroke[2].y ); // 3rd y
    v = safe_div( dX, hypot(dX, dY), &resultOk );
  }
  if ( ok != 0 ) *ok = resultOk;
  return v;
}

const std::string& CosInitAngleFeature::classDescription() {
  static const string str( 
"Calculates the cosine of the initial angle of the given stroke.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: f1 = cos a = (x2-x0) / sqrt( (x2-x0)^2 + (y2-y0)^2 )<BR><BR>"
"Note: \"The initial angle features, f1 and f2, are computed from the first and "
"        third mouse point because the result is generally less noisy than when "
"        computed from the first two points.\""
" - Bottom of first column, page 333.");
  return str;
}



/*!
  \class SinInitAngleFeature

  \brief The SinInitAngleFeature class calculates the sine of the initial angle
         of the given stroke.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
	   \code f2 = sin a = (y2-y0) / sqrt( (x2-x0)^2 + (y2-y0)^2 ) \endcode
 
  \b Note: "The initial angle features, f1 and f2, are computed from the first and
           third mouse point because the result is generally less noisy than when
           computed from the first two points." - Bottom of first column, page 333.
*/

/*!
  \b Note: \em ok is true IFF there are three or more points in the given stroke.
 */
StlFeatureResultT
SinInitAngleFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{
  StlFeatureResultT v = 0.0;
  bool resultOk = false;
  if ( stroke.size() >= 3 ) {
    FeatureCalcT dX = ( stroke[0].x      // 1st x
			- stroke[2].x ); // 3rd x
    FeatureCalcT dY = ( stroke[0].y      // 1st y
			- stroke[2].y ); // 3rd y
    v = safe_div( dY, hypot(dX, dY), &resultOk );
  }
  if ( ok != 0 ) *ok = resultOk;
  return v;
}

const std::string& SinInitAngleFeature::classDescription() {
  static const string str( 
"Calculates the sine of the initial angle of the given stroke.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: f2 = sin a = (y2-y0) / sqrt( (x2-x0)^2 + (y2-y0)^2 )<BR><BR>"
"Note: \"The initial angle features, f1 and f2, are computed from the first and "
"        third mouse point because the result is generally less noisy than when "
"        computed from the first two points.\""
" - Bottom of first column, page 333.");
  return str;
}



/*!
  \class BBoxDiagLengthFeature

  \brief The BBoxDiagLengthFeature class calculates the length of the bounding
         box diagonal.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
	   \code f3 = sqrt( (xmax - xmin)^2 + (ymax - ymin)^2 ) \endcode
*/

StlFeatureResultT
BBoxDiagLengthFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{ 
  StlFeatureResultT v = 0.0;
  bool resultOk = false;
  if ( ! stroke.empty() ) {
    StlStrokeCoordT xmin, xmax, ymin, ymax;
    getCoordMinAndMaxs( stroke, xmin, xmax, ymin, ymax );
    v = hypot( xmax-xmin, ymax-ymin );
    resultOk = true; // all seems ok...
  }
  if ( ok != 0 ) *ok = resultOk;
  return v;
}

const std::string& BBoxDiagLengthFeature::classDescription() {
  static const string str( 
"Calculates the length of the bounding box diagonal.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"       (July 1991), 329-337."
"       Page 333: f3 = sqrt( (xmax - xmin)^2 + (ymax - ymin)^2 )");
  return str;
}



/*!
  \class BBoxDiagAngleFeature

  \brief The BBoxDiagAngleFeature class calculates the angle of the bounding box
         diagonal.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
	   \code f4 = arctan( (ymax-ymin) / (xmax-xmin) ) \endcode
*/

StlFeatureResultT
BBoxDiagAngleFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{ 
  StlFeatureResultT v = 0.0;
  bool resultOk = false;
  if ( ! stroke.empty() ) {
    StlStrokeCoordT xmin, xmax, ymin, ymax;
    getCoordMinAndMaxs( stroke, xmin, xmax, ymin, ymax );
    v = atan2( ymax-ymin, xmax-xmin );
    resultOk = true; // all seems ok...
  }
  if ( ok != 0 ) *ok = resultOk;
  return v;
}

const std::string& BBoxDiagAngleFeature::classDescription() {
  static const string str( 
"Calculates the angle of the bounding box diagonal.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: f4 = arctan( (ymax-ymin) / (xmax-xmin) )");
  return str;
}



/*!
  \class FirstToLastDistFeature

  \brief The FirstToLastDistFeature class calculates the distance between the
         first and last point.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
	   \code f5 = sqrt( (xlast - x0)^2 + (ylast - y0)^2 ) \endcode
*/

StlFeatureResultT
FirstToLastDistFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{ 
  StlFeatureResultT v = 0.0;
  bool resultOk = false;
  if ( ! stroke.empty() ) {
    const StlStrokePoint& firstPt = stroke.front();
    const StlStrokePoint& lastPt = stroke.back();
    v = hypot( lastPt.x-firstPt.x, lastPt.y-firstPt.y );
    resultOk = true; // all seems ok...
  }
  if ( ok != 0 ) *ok = resultOk;
  return v;
}

const std::string& FirstToLastDistFeature::classDescription() {
  static const string str( 
"Calculates the distance between the first and last point.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: f5 = sqrt( (xlast - x0)^2 + (ylast - y0)^2 )");
  return str;
}



/*!
  \class CosFirstToLastAngleFeature

  \brief The CosFirstToLastAngleFeature class calculates the cosine of the angle
         between the first and last point.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
\code
f6 = cos b = (xlast - xfirst) / f5
   = (xlast - xfirst)
     / sqrt( (xlast - x0)^2 + (ylast - y0)^2 )
\endcode
*/
StlFeatureResultT
CosFirstToLastAngleFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{ 
  StlFeatureResultT v = 0.0;
  bool resultOk = false;
  if ( ! stroke.empty() ) {
    const StlStrokePoint& firstPt = stroke.front();
    const StlStrokePoint& lastPt = stroke.back();
    v = safe_div( (lastPt.x-firstPt.x),
		  hypot(lastPt.x-firstPt.x, lastPt.y-firstPt.y),
		  &resultOk );
  }
  if ( ok != 0 ) *ok = resultOk;
  return v;
}

const std::string& CosFirstToLastAngleFeature::classDescription() {
  static const string str( 
"Calculates the cosine of the angle between the first and last point.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: <BR><CODE>"
"                f6 = cos b = (xlast - xfirst) / f5"
"                           = (xlast - xfirst)"
"                             / sqrt( (xlast - x0)^2 + (ylast - y0)^2 )</CODE>");
  return str;
}



/*!
  \class SinFirstToLastAngleFeature

  \brief The SinFirstToLastAngleFeature class calculates the sine of the angle
         between the first and last point.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
\code
f7 = sin b = (ylast - yfirst) / f5
   = (ylast - yfirst)
     / sqrt( (xlast - x0)^2 + (ylast - y0)^2 )
\endcode
*/

StlFeatureResultT
SinFirstToLastAngleFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{ 
  StlFeatureResultT v = 0.0;
  bool resultOk = false;
  if ( ! stroke.empty() ) {
    const StlStrokePoint& firstPt = stroke.front();
    const StlStrokePoint& lastPt = stroke.back();
    v = safe_div( (lastPt.y-firstPt.y),
		  hypot(lastPt.x-firstPt.x, lastPt.y-firstPt.y),
		  &resultOk );
  }
  if ( ok != 0 ) *ok = resultOk;
  return v;
}

const std::string& SinFirstToLastAngleFeature::classDescription() {
  static const string str( 
"Calculates the sine of the angle between the first and last point.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: <BR>"
"                f7 = sin b = (ylast - yfirst) / f5"
"                           = (ylast - yfirst)"
"                             / sqrt( (xlast - x0)^2 + (ylast - y0)^2 )");
  return str;
}



/*!
  \class TotalGestureLengthFeature

  \brief The TotalGestureLengthFeature class calculates the total length of the
         gesture.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
\code
dX_p = x_p+1 - x_p
dY_p = y_p+1 - y_p
f8 = sum{ p=0, P-2, sqrt( (dX_p)^2 + (dY_p)^2 ) }
\endcode
*/

StlFeatureResultT
TotalGestureLengthFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{ 
  StlFeatureResultT v = 0.0;
  const int lastPoint = stroke.size() - 2;
  for ( int p=0; p < lastPoint; ++p ) {
    const StlStrokePoint& currPt = stroke[p];
    const StlStrokePoint& nextPt = stroke[p+1];
    v += hypot( nextPt.x-currPt.x, nextPt.y-currPt.y );
  }
  if ( ok != 0 ) *ok = true; // always true
  return v;
}

const std::string& TotalGestureLengthFeature::classDescription() {
  static const string str( 
"Calculates the total length of the gesture.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: dX_p = x_p+1 - x_p"
"                dY_p = y_p+1 - y_p"
"                f8 = sum{ p=0, P-2, sqrt( (dX_p)^2 + (dY_p)^2 ) }");
  return str;
}



/*!
  \class TotalAngleTraversedFeature

  \brief The TotalAngleTraversedFeature class calculates the total angle
         traversed around the gesture.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
\code
dX_p = x_p+1 - x_p
dY_p = y_p+1 - y_p
theta_p = arctan( (dX_p  dY_p-1 - dX_p-1  dY_p)
                  /(dX_p  dX_p-1 + dY_p  dY_p-1) )
f9 = sum{ p=1, P-2, theta_p }
\endcode
*/

StlFeatureResultT
TotalAngleTraversedFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{ 
  StlFeatureResultT v = 0.0;
  bool resultOk = true; // **TRUE** THIS TIME - until proven guilty
  const int lastPoint = stroke.size() - 2;
  for ( int p=1; p < lastPoint; ++p ) {
    // TODO: CACHE THE (p-1)s and (p+1)s !!!! - halve the calcs!!
    const StlStrokePoint& currPt = stroke[p];
    FeatureCalcT dX_p = stroke[p+1].x - currPt.x;
    FeatureCalcT dX_pLessOne = currPt.x - stroke[p-1].x;
    FeatureCalcT dY_p = stroke[p+1].y - currPt.y;
    FeatureCalcT dY_pLessOne = currPt.y - stroke[p-1].y;
    v += atan2( dX_p * dY_pLessOne - dX_pLessOne * dY_p,
		dX_p * dX_pLessOne + dY_p * dY_pLessOne );
  }
  if ( ok != 0 ) *ok = resultOk;
  return v;
}

const std::string& TotalAngleTraversedFeature::classDescription() {
  static const string str( 
"Calculates the total angle traversed around the gesture.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: <BR>"
"                dX_p = x_p+1 - x_p <BR>"
"                dY_p = y_p+1 - y_p <BR>"
"                theta_p = arctan( (dX_p * dY_p-1 - dX_p-1 * dY_p)"
"                                 /(dX_p * dX_p-1 + dY_p * dY_p-1) ) <BR>"
"                f9 = sum{ p=1, P-2, theta_p }");
  return str;
}



/*!
  \class TotalAbsAngleTraversedFeature

  \brief The TotalAbsAngleTraversedFeature class calculates the total absolute
         angle traversed around the gesture.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
\code
dX_p = x_p+1 - x_p
dY_p = y_p+1 - y_p
theta_p = arctan( (dX_p  dY_p-1 - dX_p-1  dY_p)
                  /(dX_p  dX_p-1 + dY_p  dY_p-1) )
f10 = sum{ p=1, P-2, abs(theta_p) }
\endcode
*/

StlFeatureResultT
TotalAbsAngleTraversedFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{
  StlFeatureResultT v = 0.0;
  bool resultOk = true; // **TRUE** THIS TIME - until proven guilty
  const int lastPoint = stroke.size() - 2;
  for ( int p=1; p < lastPoint; ++p ) {
    // TODO: CACHE THE (p-1)s and (p+1)s !!!! - halve the calcs!!
    const StlStrokePoint& currPt = stroke[p];
    FeatureCalcT dX_p = stroke[p+1].x - currPt.x;
    FeatureCalcT dX_pLessOne = currPt.x - stroke[p-1].x;
    FeatureCalcT dY_p = stroke[p+1].y - currPt.y;
    FeatureCalcT dY_pLessOne = currPt.y - stroke[p-1].y;
    v += fabs( atan2( dX_p * dY_pLessOne - dX_pLessOne * dY_p,
		      dX_p * dX_pLessOne + dY_p * dY_pLessOne ) );
  }
  if ( ok != 0 ) *ok = resultOk;
  return v;
}

const std::string& TotalAbsAngleTraversedFeature::classDescription() {
  static const string str( 
"Calculates the total absolute angle traversed around the gesture.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: <BR>"
"                dX_p = x_p+1 - x_p <BR>"
"                dY_p = y_p+1 - y_p <BR>"
"                theta_p = arctan( (dX_p * dY_p-1 - dX_p-1 * dY_p)"
"                                 /(dX_p * dX_p-1 + dY_p * dY_p-1) ) <BR>"
"                f10 = sum{ p=1, P-2, abs(theta_p) }");
  return str;
}



/*!
  \class TotalSqrdAngleTraversedFeature

  \brief The TotalSqrdAngleTraversedFeature class calculates the total squared
         angle traversed around the gesture.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
\code
dX_p = x_p+1 - x_p
dY_p = y_p+1 - y_p
theta_p = arctan( (dX_p  dY_p-1 - dX_p-1  dY_p)
                  /(dX_p  dX_p-1 + dY_p  dY_p-1) )
f11 = sum{ p=1, P-2, (theta_p)^2 }
\endcode
*/

StlFeatureResultT
TotalSqrdAngleTraversedFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{
  StlFeatureResultT v = 0.0;
  bool resultOk = true; // **TRUE** THIS TIME - until proven guilty
  const int lastPoint = stroke.size() - 2;
  for ( int p=1; p < lastPoint; ++p ) {
    // TODO: CACHE THE (p-1)s and (p+1)s !!!! - halve the calcs!!
    const StlStrokePoint& currPt = stroke[p];
    FeatureCalcT dX_p = stroke[p+1].x - currPt.x;
    FeatureCalcT dX_pLessOne = currPt.x - stroke[p-1].x;
    FeatureCalcT dY_p = stroke[p+1].y - currPt.y;
    FeatureCalcT dY_pLessOne = currPt.y - stroke[p-1].y;
    v += pow( atan2( dX_p * dY_pLessOne - dX_pLessOne * dY_p,
		     dX_p * dX_pLessOne + dY_p * dY_pLessOne ), 2 );
  }
  if ( ok != 0 ) *ok = resultOk;
  return v;
}

const std::string& TotalSqrdAngleTraversedFeature::classDescription() {
  static const string str( 
"Calculates the total squared angle traversed around the gesture.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: <BR>"
"                dX_p = x_p+1 - x_p <BR>"
"                dY_p = y_p+1 - y_p <BR>"
"                theta_p = arctan( (dX_p * dY_p-1 - dX_p-1 * dY_p)"
"                                 /(dX_p * dX_p-1 + dY_p * dY_p-1) ) <BR>"
"                f11 = sum{ p=1, P-2, (theta_p)^2 }");
  return str;
}



/*!
  \class MaxSqrdSpeedFeature

  \brief The MaxSqrdSpeedFeature class calculates the maximum speed (squared) of
         the gesture.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
\code
dX_p = x_p+1 - x_p
dY_p = y_p+1 - y_p
dT_p = t_p+1 - t_p
f12 = max{ p=0, P-2, ((dX_p)^2 + (dY_p)^2) / (dT_p)^2 }
\endcode
*/

StlFeatureResultT
MaxSqrdSpeedFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{ 
  StlFeatureResultT max = 0.0;
  bool resultOk = false;
  const int lastPoint = stroke.size() - 2;
  // Step 1. init max with first - prevent checks of point being 1st in the loop
  if ( stroke.size() >= 2 ) {
    const StlStrokePoint& currPt = stroke[0];
    const StlStrokePoint& nextPt = stroke[1];
    max = safe_div( pow(nextPt.x-currPt.x, 2) + pow(nextPt.y-currPt.y, 2),
		    pow((FeatureCalcT)nextPt.milliTime-currPt.milliTime, 2),
		    &resultOk );
  }
  // Step 2. find max speed across all points
  for ( int p=1; p < lastPoint; ++p ) {
    const StlStrokePoint& currPt = stroke[p];
    const StlStrokePoint& nextPt = stroke[p+1];
    StlFeatureResultT s
      = safe_div( pow(nextPt.x-currPt.x, 2) + pow(nextPt.y-currPt.y, 2),
		  pow((FeatureCalcT)nextPt.milliTime-currPt.milliTime, 2),
		  &resultOk );
    if ( s > max ) max = s;
  }
  if ( ok != 0 ) *ok = resultOk;
  return max;
}

const std::string& MaxSqrdSpeedFeature::classDescription() {
  static const string str( 
"Calculates the maximum speed (squared) of the gesture.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337. Page 333:"
"      dX_p = x_p+1 - x_p"
"      dY_p = y_p+1 - y_p"
"      dT_p = t_p+1 - t_p"
"      f12 = max{ p=0, P-2, ((dX_p)^2 + (dY_p)^2) / (dT_p)^2 }");
  return str;
}



/*!
  \class DurationFeature

  \brief The DurationFeature class calculates the duration of the gesture.
 
  \b From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4
           (July 1991), 329-337. Page 333:
	   \code f13 = t_p-1 - t0 \endcode
*/

StlFeatureResultT
DurationFeature::classCalcValue( const StlStroke& stroke, bool* ok )
{ 
  StlFeatureResultT v = 0.0;
  if ( ! stroke.empty() )
    v = stroke.back().milliTime
      - stroke.front().milliTime; // first should always be 0, but we're being safe
  if ( ok != 0 ) *ok = true; // always true even if 0 points, as that would be 0
  return v;
}

const std::string& DurationFeature::classDescription() {
  static const string str( 
"Calculates the duration of the gesture.<BR><BR>"
"From: Rubine, D. Specifying Gestures by Example. Computer Graphics 25, 4 "
"      (July 1991), 329-337."
"      Page 333: f13 = t_p-1 - t0");
  return str;
}
