/**
 * <p>Title:Database Class </p>
 * <p>Description: Connecting to the DB, querying the DB functions</p>
 * @author Eirini Ntoutsi
 * @version 1.0
 */
package database;
import java.sql.*;
import java.util.*;
import types.*;
import utils.*;
import history.*;

public class Database
{
    private Connection DBconn;
    private String dataset;
    
    /**
     * Constructor
     **/
    public Database(){
        DBconn = establishConnection();
    }    
    /**
     *Get connection
     **/
    public Connection getConnection(){
        return DBconn;
    }
    /**
     *Define which dataset is used
     **/
    public void setDataset(String dataName){
        dataset = dataName;
    }
    /**
     *Get the dataset used
     **/
    public String getDataset(){
        return dataset;
    }  
    /**
     * Close the connection
     **/
    public void terminateConnection(){
        try{
            DBconn.close();
        }
        catch (Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
    }  
    /**
     * Establish the connection
     **/
    public Connection establishConnection() {
        Connection DBconn = null;
        try{
            Class.forName("sun.jdbc.odbc.JdbcOdbcDriver");
            DBconn = DriverManager.getConnection("jdbc:odbc:monic", "", "");
          
        }
        catch(Exception e){
            System.out.println("Error: "+e.toString()+e.getMessage());
        }
        return DBconn;
    }

    /**
     *Executes a given sql statement and returns resultset (for select statements only)
     **/
    public ResultSet execSelectQuery(String aSql)throws SQLException,Exception{
      ResultSet rec = null;
      try {
          Statement st = DBconn.createStatement();
          rec = st.executeQuery(aSql);
      }
      catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
      return rec;
  }
    /**
     * Executes a given sql statement
     **/
    public boolean execute(String aSql)throws SQLException{
      PreparedStatement kokos = DBconn.prepareStatement(aSql);
      boolean res = kokos.execute();
      kokos.close();
      return(res);
  }    
    /**
     * executes a given sql statement and returns resultset (for select statements only)
     **/
    public ResultSet execSelectQuery2(String aSql)throws SQLException,Exception{
        Connection DBconn2 = null;
        try{
            Class.forName("sun.jdbc.odbc.JdbcOdbcDriver");
            DBconn2 = DriverManager.getConnection("jdbc:odbc:monic", "", "");
        }
        catch(Exception e){
            System.out.println("Error: "+e.toString()+e.getMessage());
        }
        ResultSet rec2 = null;
      try {
          Statement st2 = DBconn2.createStatement();
          rec2 = st2.executeQuery(aSql);
      }
      catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
      return rec2;
  }

    // ***********************************************************************
    // --- Start of Routines for table: tbl_clusters --- //
    // ***********************************************************************
    /**
     *Returns the id of a given cluster
     **/
    public int getClusterID(String clusterName){
        int clusterID = 90;//set as default the ID of the null cluster
        String sql = "SELECT pk_clusterID FROM tbl_clusters WHERE fld_clusterName='"+clusterName+"'";
        try{
            ResultSet rs = execSelectQuery(sql);
            rs.next();
            clusterID = rs.getInt("pk_clusterID");
            rs.close();
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" " +s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return clusterID;
    }
    /**
     *Returns the name of a given cluster
     **/
    public String getClusterName(int clusterID){
        String clusterName = "";
        String sql = "SELECT fld_clusterName FROM tbl_clusters "+
                    " WHERE pk_clusterID = "+Integer.toString(clusterID);
        try{
            ResultSet rs = execSelectQuery(sql);
            rs.next();
            clusterName = rs.getString("fld_clusterName");
            rs.close();
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" " +s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return clusterName.trim();
    }
   
    /**
     *Returns the time of a given cluster
     **/
    public int getClusterTime(int clusterID){
        int clusterTime = -1;
        String sql = " SELECT tbl_clusterings.fld_time "+
                " FROM tbl_clusters INNER JOIN tbl_clusterings "+
                " ON tbl_clusters.fk_clusteringID = tbl_clusterings.pk_ID "+
                " WHERE (tbl_clusters.pk_clusterID = "+Integer.toString(clusterID)+")";
        try{
            ResultSet rs = execSelectQuery(sql);
            rs.next();
            clusterTime = rs.getInt("fld_time");
            rs.close();
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" " +s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return clusterTime;
    }
    
  
        
  
    
    /**
     *Returns cluster instances for the KDDCup99 dataset
     **/
    public ClusterInstances getClusterInstancesKDDCup99Light(int clusterID){
        Vector instances = new Vector();
        String sql = " SELECT tbl_instances_kddcup.id, tbl_instances_kddcup.type "+
                   " FROM tbl_clustersInstances_asc inner join dbo.tbl_instances_kddcup on fk_instanceID = id"+
                   " WHERE fk_clusterID = "+Integer.toString(clusterID)+" ORDER BY id";      
        int numOfAttributes = 0;
        try{
            //ResultSet rs = execSelectQuery(sql);
            ResultSet rs = execSelectQuery2(sql);
            int inst=0;
            while (rs.next() == true){
                int instanceID = rs.getInt(1);//first column is the id
                String instanceClass = rs.getString(2);
                instances.add(inst,new NumericInstance(instanceID,new Vector(),instanceClass,clusterID));
                inst++;
            }
            rs.close();
            return new ClusterInstances(instances, numOfAttributes);
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" "+s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return null;
    }
        
    /**
     *Returns the cluster centroid for a cluster of KDDCup99 dataset
     **/
    public NumericInstance getClusterCentroidKDDCup99(int clusterID){
        Vector values = new Vector();
        String sql = " SELECT * FROM tbl_kddcup99_centroids "+
                     " WHERE id = "+Integer.toString(clusterID);      
        try{
            //ResultSet rs = execSelectQuery(sql);
            ResultSet rs = execSelectQuery2(sql);
            ResultSetMetaData rsMeta = rs.getMetaData();
            int numCols = rsMeta.getColumnCount();
            while (rs.next() == true){
                for (int i=2; i<numCols-1;i++)
                    values.add(rs.getDouble(i));
            }
            rs.close();
            return new NumericInstance(clusterID, values);
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" "+s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return null;      
    }    
        
    /**
     *Returns the cluster centroid for a cluster of KDDCup98 dataset
     **/
    public NumericInstance getClusterCentroidKDDCup98(int clusterID){
        Vector values = new Vector();
        String sql = " SELECT * FROM tbl_KDD98cup_centroids "+
                     " WHERE id = "+Integer.toString(clusterID);      
        try{
            ResultSet rs = execSelectQuery(sql);  
            ResultSetMetaData rsMeta = rs.getMetaData();
            int numCols = rsMeta.getColumnCount();
            while (rs.next() == true){
                for (int w=2;w<numCols;w++){
                    values.add(rs.getDouble(w));
                }
            }
            rs.close();
            return new NumericInstance(clusterID, values);
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" "+s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return null;      
    }    
    
   
     /**
    * Returns a vector of outcomingEdges for a given cluster
    **/
    public Vector getOutcomingEdges(int clusterID, int experimentID, GraphHistory history){
        Vector outcomingEdges = new Vector();
        GraphNode fromNode = getGraphNode(clusterID);
        String sql = " SELECT fk_clusterID_to, fld_overlap,fld_externalType, "+
                   " fld_internalType, fld_internalValue,fld_transitionDiff,"+
                   " tbl_clusters.fld_clusterName,tbl_clusterings.fld_time"+
                   " FROM tbl_transitions  inner join tbl_clusters "+
                        " on tbl_transitions.fk_clusterID_to=tbl_clusters.pk_clusterID"+
                        " inner join tbl_clusterings on tbl_clusters.fk_clusteringID=tbl_clusterings.pk_ID"+
                   " WHERE (fk_clusterID_from = "+Integer.toString(clusterID)+")"+
                   " AND (fk_runningID IN "+
                   "     (SELECT pk_runningID FROM tbl_runnings "+
                   "       WHERE      fk_experimentID = "+experimentID+"))" ;
        try{
            ResultSet rs = execSelectQuery(sql);
            int toNodeID = -1, oldToNodeID = -1;
            boolean isInternal = false;
            String sizeTransStr = "",compactnessTransStr="",locationTransStr="";
            double sizeTrans = -1,compactnessTrans=-1,locationTrans=-1;
            GraphEdge survivalEdge = new GraphEdge();
            while (rs.next() == true){
                toNodeID = rs.getInt("fk_clusterID_to");
                if (toNodeID != oldToNodeID & (oldToNodeID>-1)){
                    if (isInternal == true) {//in case of survival
                      outcomingEdges.add(survivalEdge);
                      survivalEdge = new GraphEdge();
                      sizeTransStr = "";
                      compactnessTransStr = "";
                      locationTransStr = "";
                      sizeTrans = -1;
                      compactnessTrans = -1;
                      locationTrans = -1;
                      isInternal = false;
                  }
              }
              double overlap = rs.getDouble("fld_overlap");
              String externalTrans = rs.getString("fld_externalType");
              String internalTrans = rs.getString("fld_internalType");
              String internalValue = rs.getString("fld_internalValue");
              double internalDiff = rs.getDouble("fld_transitionDiff");
              String toNodeName = rs.getString("fld_clusterName");
              int toNodeTime = rs.getInt("fld_time");
              GraphNode toNode = history.getNode(toNodeID);
              if (toNode == null)
                  continue;
              if (internalTrans != null){
                  isInternal = true;
                  survivalEdge.setStartNode(fromNode);
                  survivalEdge.setEndNode(toNode);
                  survivalEdge.setOverlap(overlap);
                  survivalEdge.setExternalTransitionLabel(externalTrans);
                  if (internalTrans.equals("size")) {
                      sizeTransStr = internalValue;
                      sizeTrans = internalDiff;
                      survivalEdge.setSizeInternalTransitionLabel(sizeTransStr);
                      survivalEdge.setSizeInternalTransition(sizeTrans);
                  } 
                  else if (internalTrans.equals("location")) {
                      locationTransStr = internalValue;
                      locationTrans = internalDiff;
                      survivalEdge.setLocationInternalTransitionLabel(locationTransStr);
                      survivalEdge.setLocationInternalTransition(locationTrans);
                  } 
                  else if (internalTrans.equals("compactness")) {
                      compactnessTransStr = internalValue;
                      compactnessTrans = internalDiff;
                      survivalEdge.setCompactnessInternalTransitionLabel(compactnessTransStr);
                      survivalEdge.setCompactnessInternalTransition(compactnessTrans);
                  }
              }
              if (isInternal == false) {
                  GraphEdge myEdge = new GraphEdge(fromNode, toNode, overlap,
                          externalTrans, sizeTransStr, sizeTrans,
                          compactnessTransStr, compactnessTrans,
                          locationTransStr, locationTrans);
                  outcomingEdges.add(myEdge);
              }
              oldToNodeID = toNodeID;
         }
         if (isInternal == true) {//in case of survival
             outcomingEdges.add(survivalEdge);
         }
          rs.close();
     }
     catch(SQLException s){System.out.println("[getOutcomingEdges 1] SQL Error: "+s.toString()+" "+s.getErrorCode()+s.getSQLState());}
     catch(Exception e){/*System.out.println("[getOutcomingEdges 2] Error: "+e.toString()+e.getMessage())*/;}
     return outcomingEdges;
  }
  
    // ***********************************************************************
    // --- Start of Routines for table: tbl_clusterings --- //
    // ***********************************************************************
    /**
    *Returns the name of a given clustering
    **/
    public String getClusteringName(int clusteringID){
        String clusteringName = "";
        String sql = "SELECT fld_Name FROM tbl_clusterings "+
                      " WHERE pk_ID = "+Integer.toString(clusteringID);
        try{
            ResultSet rs = execSelectQuery2(sql);
            rs.next();
            clusteringName = rs.getString("fld_Name");
            rs.close();
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" "+s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return clusteringName.trim();
    }
    /**
     *Returns the time of a given clustering
     **/
    public int getClusteringTime(int clusteringID){
        int clusteringTime = -1;
        String sql = "SELECT fld_time FROM tbl_clusterings "+
                     " WHERE pk_ID = "+Integer.toString(clusteringID);
        try{
            //ResultSet rs = execSelectQuery(sql);
            ResultSet rs = execSelectQuery2(sql);
            rs.next();
            clusteringTime = rs.getInt("fld_time");
            rs.close();
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" "+s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return clusteringTime;
    }
      
    /**
     * Returns the clusterings of a history
     **/
    public Vector getClusterings(int datasetID){
        Vector res = new Vector();
        String sql = "SELECT pk_ID FROM tbl_clusterings "+
                     " WHERE fk_datasetID="+datasetID+" ORDER BY fld_time";
        try{
            ResultSet rs = execSelectQuery(sql);
            while (rs.next() == true){
                res.add(""+rs.getInt("pk_ID")+"");
            }
            rs.close();
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" "+s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return res;
    }
    
    // ***********************************************************************
    // --- Start of Routines for table: tbl_datasets --- //
    // ***********************************************************************
    /**
     * Returns the name of a given cluster
     **/
    public String getDatasetName(int historyID){
        String datasetName = "";
        String sql = "SELECT fld_Name FROM tbl_datasets "+
                      " WHERE pk_ID = "+Integer.toString(historyID);
        try{
            ResultSet rs = execSelectQuery(sql);
            rs.next();
            datasetName = rs.getString("fld_Name");
            rs.close();
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" " +s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return datasetName.trim();
    }   
    /**
     * Returns the ID for a given experiment
     **/
    public int getExperimentDatasetID(int experimentID){
        int historyID = -1;
        String sql = "SELECT fk_datasetID FROM tbl_experiments "+
                     " WHERE (pk_experimentID = "+experimentID+")";
        try{
            ResultSet rs = execSelectQuery(sql);
            rs.next();
            historyID = rs.getInt("fk_datasetID");
            rs.close();
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" " +s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return historyID;
    }    
    /**
     *Returns a history
     ***/
    public GraphHistory getHistory(int datasetID, int startPosition, int howMany) {
        GraphHistory myHistory = new GraphHistory();
        myHistory.setHistoryName(getDatasetName(datasetID));
        //get clustering IDs
        Vector clusteringIDs = new Vector();
        String sql =" SELECT pk_ID FROM tbl_clusterings "+
                    " WHERE fk_datasetID = "+Integer.toString(datasetID);
        try{
            ResultSet rs = execSelectQuery(sql);
            //go to startPosition
            for (int i=0; i<startPosition; i++)
                if (!rs.next())
                    break;
            //load howMany
            int j=0;
            while (rs.next() == true){
                clusteringIDs.add(rs.getInt("pk_ID"));
                if ((++j)>=howMany)
                    break;
            }           
            rs.close();
        }
        catch (SQLException s){System.out.println("SQL Error: " + s.toString() + " " +s.getErrorCode() + s.getSQLState());}
        catch (Exception e) {System.out.println("Error: " + e.toString() + e.getMessage());}
         
        //continue for each clustering and load its clusters
        for (int i=0;i<clusteringIDs.size();i++){
            int clusteringID = (new Double(clusteringIDs.get(i).toString())).intValue();
            myHistory.addHistorySlice(getGraphLevel(clusteringID));
            if (i%10==0) System.gc();
        }
        return myHistory;
    }
    /**
     * load edges to the history
     **/
    public void loadHistoryEdges(GraphHistory myHistory, int experimentID){
        for (int i=0; i<myHistory.getSize(); i++){
           GraphLevel myGraphLevel = myHistory.getHistorySlice(i) ;
           for (int j=0; j<myGraphLevel.getSize();j++){
               GraphNode myGraphNode = myGraphLevel.getNode(j);
               int clusterID = myGraphNode.getID();
               myGraphNode.setOutEdges(getOutcomingEdges(clusterID,experimentID,myHistory));
               //myGraphNode.setInEdges(getIncomingEdges(clusterID,experimentID));
               //System.out.println(clusterID);
               //System.out.println("Outcoming edges\n" +myGraphNode.printNodeOutEdges());
               //System.out.println("Incoming edges\n" +myGraphNode.printNodeInEdges());
           }
        }
    }
   /**
    *Returns a GraphNode object
    **/
    public GraphNode getGraphNode(int clusterID){
      String clusterName = getClusterName(clusterID);
      int clusterTime = getClusterTime(clusterID);
      return new GraphNode(clusterID,clusterName, clusterTime, clusterTime);
    }
    /**
     * Returns a GraphLevel object (i.e. a clustering)
    **/
    public GraphLevel getGraphLevel(int clusteringID){
        int clusteringTime = getClusteringTime(clusteringID);
        Vector clusters = new Vector();
        String sql="SELECT pk_clusterID,fld_clusterName,fld_label,fld_shortLabel "+
                   " FROM tbl_clusters WHERE " +
                   " fk_clusteringID = " + clusteringID+" ORDER BY pk_clusterID ASC";
        try{
            //ResultSet rs = execSelectQuery(sql);
            ResultSet rs = execSelectQuery2(sql);
            while (rs.next() == true){
                GraphNode node = new GraphNode(rs.getInt("pk_clusterID"),
                         rs.getString("fld_clusterName"),
                         clusteringTime,clusteringTime,
                        rs.getString("fld_label"),rs.getString("fld_shortLabel"));
                clusters.add(node);
            }
            rs.close();
        }
        catch (SQLException s){System.out.println("SQL Error: " + s.toString() + " " +s.getErrorCode() + s.getSQLState());}
        catch (Exception e) {System.out.println("Error: " + e.toString() + e.getMessage());}
        
        String datasetNameStr = this.getDataset();
        
        //--- for each cluster, load its instances (only IDs)
         for (int i=0;i<clusters.size();i++){
            GraphNode myNode = (GraphNode)clusters.get(i);
            int ID = myNode.getID();//the cluster
            //-- check which dataset is used
            if (datasetNameStr.equals("ACM Library_W=2")){
                //1.-- load instances
                myNode.setInstances(getClusterInstancesACMLight(ID)); //load only ID and class label
            }
            else if (datasetNameStr.equals("KDDcup.99small.W=2")){//("KDDcupSmall_W=2")){
                //1.-- load instances
                myNode.setInstances(getClusterInstancesKDDCup99Light(ID)); //load only ID and class label
            }
            else if (datasetNameStr.equals("KDDcup98_W=2")){
                //1.-- load instances
                myNode.setInstances(getClusterInstancesKDD98Light(ID)); //load only ID and class label
            }   
         }   

        //--- for each cluster, load its centroid
         for (int i=0;i<clusters.size();i++){
            GraphNode myNode = (GraphNode)clusters.get(i);
            int ID = myNode.getID();//the cluster
            //-- check which dataset is used
            if (datasetNameStr.equals("ACM Library_W=2")){
                //2.-- load centroid
                 myNode.setCentroid(getClusterCentroidACM(ID));                
            }
            else if (datasetNameStr.equals("KDDcup.99small.W=2")){
                //2.-- load centroid
                myNode.setCentroid(getClusterCentroidKDDCup99(ID));
            }
            else if (datasetNameStr.equals("KDDcup98_W=2")){
                //2.-- load centroid
                myNode.setCentroid(getClusterCentroidKDDCup98(ID));
            }               
         }   
        return new GraphLevel(clusters, clusteringTime, getClusteringName(clusteringID),clusteringID);
    }
    /**
     * Create a new experiment and return its ID
     **/
    public int createExperiment(int historyID, double survivalThreshold, double splitThreshold, 
                                double sizeThreshold, String mode, String clusterType){
        int ID = -1;
        try{
            //insert experiment info
            String sql ="INSERT INTO tbl_experiments(fld_survivalThreshold,fld_splitThreshold, " +
                        " fld_sizeThreshold,fld_mode,fk_datasetID,fld_date,fld_clustertype)" +
                        " VALUES(" + survivalThreshold + "," + splitThreshold + "," +
                        sizeThreshold + ",'" + mode + "',"+historyID+",GETDATE(),'"+clusterType+"')";
            //System.out.println(sql);
            execute(sql);
            //get newly inserted experiment ID
            ResultSet rs = execSelectQuery("SELECT max(pk_experimentID) as experimentID " +
                                           "FROM tbl_experiments");
            rs.next();
            ID = rs.getInt("experimentID");
            rs.close();
        }
        catch (SQLException s){System.out.println("SQL Error: " + s.toString() + " " +s.getErrorCode() + s.getSQLState());}
        catch (Exception e) {System.out.println("Error: " + e.toString() + e.getMessage());}
        return ID;
    }//
    
    
   
    
  
    /**
     * Load a set of base clusters from the DB ordered on their IDs
     **/
    public SetOfClusters getClustering_Clusters(int clusteringID, String clusterType){
        //first get the IDs of these clusters
        Vector clusters = new Vector();
        try{
            String sql = "SELECT pk_clusterID FROM tbl_clusters "+
                         "WHERE fk_clusteringID=" + clusteringID+" ORDER BY pk_clusterID ASC";
            ResultSet rs = execSelectQuery(sql);
            while (rs.next() == true){
                clusters.add(rs.getInt("pk_clusterID"));
            }
            rs.close();
        }
        catch (SQLException s){System.out.println("SQL Error: " + s.toString() + " " +s.getErrorCode() + s.getSQLState());}
        catch (Exception e){System.out.println("Error: " + e.toString() + e.getMessage());} 

        //then fill these clusters with information
        SetOfClusters myClustering = null;
        try{
            String clusteringName = getClusteringName(clusteringID);
            myClustering = new SetOfClusters(clusteringName);
            for (int i=0;i<clusters.size();i++){
                int clusterID = new Double(clusters.get(i).toString()).intValue();
                if (clusterType.equalsIgnoreCase("BaseCluster")){//add them as of type B1 clusters
                    BaseCluster bc = getCluster_BaseCluster(clusterID);
                    myClustering.addPattern(bc);               
                }
                //else if (clusterType.equalsIgnoreCase("GaussianCluster")){//add them as of type B2 clusters (get their labels from the specified table)
                    //GaussianCluster gc = getCluster_GaussianCluster(clusterID,"tbl_clusterLabels_EM");
                    //myClustering.addPattern(gc);
                //}
                else if (clusterType.equalsIgnoreCase("Gaussian_and_BaseCluster")){//type B1+B2
                    Gaussian_and_BaseCluster gb = getCluster_Gaussian_and_BaseCluster(clusterID,"tbl_clusterLabels_EM");
                    myClustering.addPattern(gb);
                }
                else {
                    System.out.println("The cluster type is not {Base|Gaussian|Base_and_Gaussian}Cluster!!!! SOS SOS SOS");
                }
            }
            myClustering.setID(clusteringID);
        }
        catch (Exception e) {System.out.println("Error: " + e.toString() + e.getMessage());}            myClustering.setID(clusteringID);
        return myClustering;
    }   
    
    /** 
     * Returns a gaussian and base cluster object
     **/
    public Gaussian_and_BaseCluster getCluster_Gaussian_and_BaseCluster(int clusterID, String tableName){
        //Get gaussian type related info
        Vector attributes = new Vector();
        Vector means = new Vector();
        Vector stdDevs = new Vector();
        String clusterName = getClusterName(clusterID);
        String sql = " SELECT fld_attributeName, fld_attributeMeanValue, "+
                     " fld_attributeStdDevValue FROM "+tableName +
                     " WHERE fk_clusterID = "+Integer.toString(clusterID)+
                     " ORDER BY pk_ID";
        try{
            ResultSet rs = execSelectQuery(sql);
            while (rs.next() == true){
                String attrName = rs.getString("fld_attributeName");
                String attrMean = rs.getString("fld_attributeMeanValue");
                String attrStdDev = rs.getString("fld_attributeStdDevValue");
                attributes.add(attrName);
                means.add(attrMean);
                stdDevs.add(attrStdDev);
            }
            rs.close();
        }
        catch (SQLException s){System.out.println("SQL Error: " + s.toString() + " " +s.getErrorCode() + s.getSQLState());}
        catch (Exception e) {System.out.println("Error: " + e.toString() + e.getMessage());}    
        
        //Get base type related info
        Vector instances = new Vector();
        int numOfAttributes = 0;
        ClusterInstances instancesCls = null;
        sql = "SELECT fk_instanceID FROM tbl_clustersInstances_asc "+
                     "WHERE fk_clusterID = "+Integer.toString(clusterID);
        try{
            ResultSet rs = execSelectQuery(sql);
            while (rs.next() == true){
                int instanceID = rs.getInt("fk_instanceID");
                instances.add(new NumericInstance(instanceID,new Vector()));
            }
            instancesCls = new ClusterInstances(instances, numOfAttributes);
            rs.close();
        }
        catch (SQLException s){System.out.println("SQL Error: " + s.toString() + " " +s.getErrorCode() + s.getSQLState());}
        catch (Exception e) {System.out.println("Error: " + e.toString() + e.getMessage());}    
        
        Gaussian_and_BaseCluster myGBCluster = new Gaussian_and_BaseCluster(clusterID,
                                          clusterName,attributes,means,stdDevs,instancesCls);
        return myGBCluster;
        
    }

        
    
    /** 
     * Returns a base cluster object (just the IDs of the instances are contained)
     **/
    public BaseCluster getCluster_BaseCluster(int clusterID){
        Vector instances = new Vector();
        int numOfAttributes = 0;
        ClusterInstances instancesCls = null;
        String clusterName = getClusterName(clusterID);
        String sql = "SELECT fk_instanceID FROM tbl_clustersInstances_asc "+
                     "WHERE fk_clusterID = "+Integer.toString(clusterID);
        //System.out.println(sql);
        try{
            ResultSet rs = execSelectQuery(sql);
            while (rs.next() == true){
                int instanceID = rs.getInt("fk_instanceID");
                instances.add(new NumericInstance(instanceID,new Vector()));
            }
            instancesCls = new ClusterInstances(instances, numOfAttributes);
            rs.close();
        }
        catch (SQLException s){System.out.println("SQL Error: " + s.toString() + " " +s.getErrorCode() + s.getSQLState());}
        catch (Exception e) {System.out.println("Error: " + e.toString() + e.getMessage());}    
        BaseCluster myBaseCluster = new BaseCluster(clusterID,clusterName, instancesCls);
        return myBaseCluster;
    }
    /** 
     * Returns the intersection of instances of the two clusters
     **/
    public int clusterIntersection(int ID1, int ID2){
        String sql = " SELECT     COUNT(*) AS cnt "+
                     " FROM      (SELECT     fk_instanceID "+
                        "      FROM          tbl_clustersInstances_asc "+
                        "      WHERE      (fk_clusterID = "+ID1+") AND fk_instanceID IN "+
                    "                 (SELECT     fk_instanceID "+
                    "                FROM          tbl_clustersInstances_asc "+
                    "                WHERE      (fk_clusterID = "+ID2+"))) DERIVEDTBL";
        try{
         ResultSet rs = execSelectQuery(sql);
         rs.next();
         int res = rs.getInt("cnt");
         rs.close();
         return res;
     }
     catch (SQLException s){System.out.println("SQL Error: " + s.toString() + " " +s.getErrorCode() + s.getSQLState());}
     catch (Exception e) {System.out.println("Error: " + e.toString() + e.getMessage());}
     return -1;
 }    
        
    
   /******************************************/ 
    
     /**
     *Returns cluster instances for the ACM dataset
     **/
    public ClusterInstances getClusterInstancesACMLight(int clusterID){
        Vector instances = new Vector();
        String sql = " SELECT ACM_tbl_instances.id "+
                   " FROM tbl_clustersInstances_asc inner join dbo.ACM_tbl_instances on fk_instanceID = id"+
                   " WHERE fk_clusterID = "+Integer.toString(clusterID)+" ORDER BY id";      
        int numOfAttributes = 0;
        try{
            ResultSet rs = execSelectQuery(sql);  
            int inst=0;
            while (rs.next() == true){
                int instanceID = rs.getInt(1);//first column is the id
                instances.add(inst,new NumericInstance(instanceID,new Vector(),"",clusterID));
                inst++;
            }
            rs.close();
            return new ClusterInstances(instances, numOfAttributes);
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" "+s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return null;
    }
      
     /**
     *Returns the cluster centroid for a cluster of the ACM dataset
     **/
    public NumericInstance getClusterCentroidACM(int clusterID){
        Vector values = new Vector();
        String sql = " SELECT fld_clusterLabel, fld_weight FROM tbl_ACM_centroids "+
                     " WHERE fk_clusterID = "+Integer.toString(clusterID);
        try{
            ResultSet rs = execSelectQuery(sql);  
            ResultSetMetaData rsMeta = rs.getMetaData();
            int numCols = rsMeta.getColumnCount();
            while (rs.next() == true){
                values.add(rs.getDouble("fld_weight"));
                //for (int w=2;w<=numCols;w++){
                  //  try{
                      //  values.add(rs.getDouble(w));
                    //}
                    //catch(Exception e){
                      //  values.add("NULL");
                    //}

                //}
            }
            rs.close();
            return new NumericInstance(clusterID, values);
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" "+s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return null;      
    }    
   

     /**
     *Returns cluster instances for the ACM dataset
     **/
    public ClusterInstances getClusterInstancesKDD98Light(int clusterID){
        Vector instances = new Vector();
        String sql = " SELECT id "+
            " FROM tbl_clustersInstances_asc inner join dbo.tbl_instances_KDD98cup on fk_instanceID = id"+
            " WHERE fk_clusterID = "+Integer.toString(clusterID)+" ORDER BY id";      
        int numOfAttributes = 0;
        try{
            ResultSet rs = execSelectQuery(sql);  
            int inst=0;
            while (rs.next() == true){
                int instanceID = rs.getInt(1);//first column is the id
                instances.add(inst,new NumericInstance(instanceID,new Vector(),"",clusterID));
                inst++;
            }
            rs.close();
            return new ClusterInstances(instances, numOfAttributes);
        }
        catch(SQLException s){System.out.println("SQL Error: "+s.toString()+" "+s.getErrorCode()+s.getSQLState());}
        catch(Exception e){System.out.println("Error: "+e.toString()+e.getMessage());}
        return null;
    }


       
}