package history;

import utils.NumericInstance;
/**
 * <p>Title: Graph Compression</p>
 * <p>Description: Implements different compressions over the evolution graph</p>
 * <p><b>Rule 0:</b> Two nodes x, y are merged if y is a survival without internal changes for x</p>
 * <p><b>Rule 1:</b> Two nodes x, y are merged if y is a survival with constrained internal changes for x</p>
 * <p><b>Rule 2:</b> Two nodes x, y are merged if y is a survival with constrained internal changes for x and 
 * furthermore y is intensionally close to x</p>
 * @author Eirini Ntoutsi
 * @version 1.0
 */
public class GraphCompression {
    GraphHistory myHistory; //the graph history to be compressed
    int[] compressionsCntPerLevel; //keeps stats of how many nodes have been compressed at each time point
    double[] compressionCentroidsDistancePerLevel;//keeps stats of the euclidean distance between the old nodes and the one to which they have been compressed (if any) at each time point
    double[] compressionLabelsKLDistancePerLevel;//keeps stats of the KL distance between the old node label and the one to which they have been compressed (if any) at each time point
    double[] compressionSSQPerLevel;//keeps stats of the SSQ between the old node centroid T1 and its data members D1 
    double[] compressionInfoLossPerLevel;//keeps stats of the distance between the old node centroid T1 and the new clustering (after the compression)
    //and the one to which it has been compressed (if any) T2, i.e. d(t2,d1) at each time point
    boolean display = false;    
    StringBuffer compressionResults; //stores compression information
    /**
     * constructor
    **/
    public GraphCompression(GraphHistory theHistory, boolean display){
        this.myHistory = theHistory;
        this.compressionsCntPerLevel = new int[myHistory.getSize()+3]; //+1 το πρόσθεσα μετά
        this.compressionCentroidsDistancePerLevel = new double[myHistory.getSize()+3];//+1 το πρόσθεσα μετά
        this.compressionLabelsKLDistancePerLevel = new double[myHistory.getSize()+3];//+1 το πρόσθεσα μετά
        this.compressionSSQPerLevel = new double[myHistory.getSize()+3];//+1 το πρόσθεσα μετά
        this.compressionInfoLossPerLevel = new double[myHistory.getSize()+3];//+1 το πρόσθεσα μετά
        this.display = display;
        compressionResults = new StringBuffer();
    }
    
    /************************   Rule 0    ******************************/
    /**
     *Apply Rule 0: x,y are merged if y is a survival without internal changes for x
     **/
    public GraphHistory doCompression_SurvivalsWithoutChanges(int startLevel, int endLevel){
        CompressHistory_SurvivalsWithoutChanges(startLevel,endLevel);
        //RemoveDeletedNodesFromHistory();
        return myHistory;
    }    
    /**
     *Compress survivals with no internal changes (Rule 0)
     **/
    public void CompressHistory_SurvivalsWithoutChanges(int startLevel, int endLevel){
        //for each level/ clustering of the history
        for (int i=startLevel ;i<endLevel;i++){
            GraphLevel level = myHistory.getHistorySlice(i);
            //for each node of this level
            for (int j=0; j<level.getSize();j++){
                GraphNode node = level.getNode(j);
                int nodeLevel = node.getLevel()-1;
                double SSQdistance = node.getSSQ();
                compressionSSQPerLevel[nodeLevel]+=SSQdistance;//add now the SSQ of the node, and then you adjust it according to the compression
                if (node.hasOutEdges()){
                    compressNextNode_SurvivalsWithoutChangesRule(node);
                }
            }
        }
    }
    /**
     *Compress next node for survivals without changes (Rule 0)
     **/
public void compressNextNode_SurvivalsWithoutChangesRule(GraphNode node){
    if ((node.hasOutEdges())==true) {
    for (int m = 0; m < node.getNumOfOutEdges(); m++) {
        GraphEdge edge = node.getOutEdge(m);
        GraphNode toNode = edge.getToNode(); //the end at which our node results
        //if 1: the extensional transition is a survival
        if (edge.getExternalTransition().equals("survival")){
            //if there exist no internal transition for size and location and compacteness
            //if 2: there exist no internal transition for size
            boolean sizeOK=true,locationOK=true,compactnessOK=true;
            //if size trans is determined, check if it is nochange
            if (edge.getSizeInternalTransitionLabel().length()>0){
                if (edge.getSizeInternalTransitionLabel().equals("nochange")){//its no change
                    if (edge.getSizeInternalTransition()==0)//and size diff is zero
                        sizeOK=true;
                }
                else//there is some change in size
                    sizeOK=false;
            }
            //if location trans is determined, check if it is nochange
            if (edge.getLocationInternalTransitionLabel().length()>0){
                if (edge.getLocationInternalTransitionLabel().equals("nochange")){//its no change
                    if (edge.getLocationInternalTransition()==0)//and location diff is zero
                        locationOK=true;
                }
            }
            //if compactness trans is determined, check if it is nochange
            if (edge.getCompactnessInternalTransitionLabel().length()>0){
                if (edge.getCompactnessInternalTransitionLabel().equals("nochange")){//its no change
                    if (edge.getCompactnessInternalTransition()==0)//and compactness diff is zero
                        compactnessOK=true;
                }
            }
            
            //if there is no change, do the compression
            locationOK=true;
            compactnessOK=true;
            if ( sizeOK && locationOK && compactnessOK){
                toNode.setTimeStart(node.getTimeStart());
                double SSQdistance = node.instances.SumSquareDistance(toNode.getCentroid());
                double clusterInfoLoss = node.distance(toNode);
                compressNextNode_SurvivalsWithoutChangesRule(toNode);
                node.setDeletedStatus(true);
                edge.setDeletedStatus(true);
                int nodeLevel = node.getLevel()-1;
                compressionsCntPerLevel[nodeLevel] +=1;
                //substract the old SSQ of the node, and then add its new SSQ value 
                compressionSSQPerLevel[nodeLevel]-=node.getSSQ();
                compressionInfoLossPerLevel[nodeLevel]+=clusterInfoLoss;
                compressionSSQPerLevel[nodeLevel]+=SSQdistance;
            }//if all ok
        }//if (survival)
    }
    }
}
    


    /**
     *Compress survivals with internal changes (Rule 1)
     **/
    public void CompressHistory_SurvivalsWithChanges(int startLevel, int endLevel){
        //-- for each level of the graph
        for (int i=startLevel ;i<endLevel;i++){
            GraphLevel level = myHistory.getHistorySlice(i);
            int nodeLevel =-1;
            //-- for each node of this level
            for (int j=0; j<level.getSize();j++){
                GraphNode node = level.getNode(j);
                nodeLevel = node.getLevel()-1;
                //add now the SSQ of the node to the level, and then 
                    //you adjust it according to the compression                                
                double SSQdistance = node.getSSQ();
                compressionSSQPerLevel[nodeLevel]+=SSQdistance;
                if (node.hasOutEdges())
                    compressNextNode_SurvivalsWithChangesRule(node);
            }
        }//for all levels
    }    
 
    /**
     *Compress next node for survivals with changes (Rule 1)
     **/
    public void compressNextNode_SurvivalsWithChangesRule(GraphNode node){
        int compressionsCnt=0;
        if ((node.hasOutEdges())==true) {
            for (int m = 0; m < node.getNumOfOutEdges(); m++) {
                GraphEdge edge = node.getOutEdge(m);
                GraphNode toNode = edge.getToNode();
                //--1) if the extensional transition is a survival
                if (edge.getExternalTransition().equals("survival")){
                        toNode.setTimeStart(node.getTimeStart());
                        toNode.setCentroid(node.avgCentroid(toNode));
                        double SSQdistance = node.instances.SumSquareDistance(toNode.getCentroid());
                        double clusterInfoLoss = node.distance(toNode);
                        node.updateInstancesClusterID(toNode.getID());//assing old node instances to the new node
                        compressNextNode_SurvivalsWithChangesRule(toNode);
                        node.setDeletedStatus(true);
                        edge.setDeletedStatus(true);
                        compressionsCnt++;
                        int nodeLevel = node.getLevel()-1;
                        compressionsCntPerLevel[nodeLevel] +=1;
                        //substract the old SSQ of the node, and then add its new SSQ value 
                        compressionSSQPerLevel[nodeLevel]-=node.getSSQ();
                        compressionInfoLossPerLevel[nodeLevel]+=clusterInfoLoss;
                        compressionSSQPerLevel[nodeLevel]+=SSQdistance;
                }//if ("survival")
            }//for all edges
        }//if it has outcoming edges
    }//end of function   
    
    /************************   Rule 2    ******************************/
    /**
     * Apply Rule 2: x,y are merged if y is a survival for x and distance(x.label, y.label) is 
     * below a given threshold.
     **/
    public GraphHistory doCompression_SurvivalsWithChangesAndIntensionClose(int startLevel, 
            int endLevel, double maxCentroidsDistance){
        CompressHistory_SurvivalsWithChangesAndIntensionClose(startLevel,endLevel,maxCentroidsDistance);
        return myHistory;
    }    
    /**
     *Compress survivals with internal changes and close intensions(Rule 2)
     **/
    public void CompressHistory_SurvivalsWithChangesAndIntensionClose(int startLevel, int endLevel,
            double maxCentroidsDistance){
        compressionResults.append("NumOfLevels:\t"+(endLevel-startLevel)+"\n");
        compressionResults.append("max centroids distance:\t"+maxCentroidsDistance+"\n");
        compressionResults.append("Level\tNODE\tspaceReduction\tinformationLoss\n");
        for (int i=startLevel ;i<endLevel;i++){
            GraphLevel level = myHistory.getHistorySlice(i);
            int nodeLevel =-1;
            for (int j=0; j<level.getSize();j++){//for each node of this level
                GraphNode node = level.getNode(j);
                nodeLevel = node.getLevel()-1;
                if (!node.getChecked()){
                    if (node.hasOutEdges())
                        compressNextNode_SurvivalsWithChangesAndIntensionCloseRule(node,maxCentroidsDistance);
                }
            }
        }//for all levels
    }   
    /**
     *Compress next node for survivals with changes and close intensional descriptions (Rule 2)
     **/
    public void compressNextNode_SurvivalsWithChangesAndIntensionCloseRule(GraphNode node, 
            double maxCentroidsDistance){
        int compressionsCnt=0;
        int nodeLevel = node.getLevel()-1;
        if ((node.hasOutEdges())==true){
            for (int m = 0; m < node.getNumOfOutEdges(); m++) {
                GraphEdge edge = node.getOutEdge(m);
                GraphNode toNode = edge.getToNode(); //the end at which our node results
                int toNodeLevel = toNode.getLevel()-1;
                System.out.println("from "+nodeLevel+" to "+toNodeLevel);
                //-- C1: Does edge corresponds to a survival transition?
                if (edge.getExternalTransition().equals("survival")){
                    double centroidsDistance = node.distance(toNode);
                    //-- C2: Does centroids distance is below the upper limit?
                    if (centroidsDistance <= maxCentroidsDistance){//do summarization
                            NumericInstance avgCentroid = node.avgCentroid(toNode);
                            //infoLoss due to substitution of -node- from avgCentroid
                            double nodeInfoLoss = utils.Functions.EuclideanDistance(
                                                    (node.getCentroid()).getInstanceValues(),
                                                    avgCentroid.getInstanceValues());                            
                            //infoLoss due to substitution of -toNode- from avgCentroid
                            double toNodeInfoLoss = utils.Functions.EuclideanDistance(
                                                    (toNode.getCentroid()).getInstanceValues(),
                                                    avgCentroid.getInstanceValues());
                            toNode.setTimeStart(node.getTimeStart());                               
                            toNode.setCentroid(avgCentroid);
                            //double SSQdistance = node.instances.SumSquareDistance(toNode.getCentroid());                            
                            compressNextNode_SurvivalsWithChangesAndIntensionCloseRule(toNode, maxCentroidsDistance);    
                            node.setDeletedStatus(true);
                            edge.setDeletedStatus(true);
                            compressionsCnt++;
                            //update cntcompressed and infoLoss for the level of node
                            System.out.println("Update level stats "+nodeLevel);
                            compressionsCntPerLevel[nodeLevel] +=1;
                            compressionInfoLossPerLevel[nodeLevel]+=nodeInfoLoss;
                            compressionResults.append(nodeLevel+"\t"+node.getID()+"\t"+1+"\t"+nodeInfoLoss+"\n");
                            //update infoLoss for the level of toNode
                            compressionInfoLossPerLevel[toNodeLevel]+=toNodeInfoLoss;                            
                            //System.out.println(toNodeLevel+"\t"+toNode.getID()+"\t"+0+"\t"+toNodeInfoLoss);
                            compressionResults.append(toNodeLevel+"\t"+toNode.getID()+"\t"+0+"\t"+toNodeInfoLoss+"\n");
                            //substract the old SSQ of the node, and then add its new SSQ value 
                            //compressionSSQPerLevel[nodeLevel]-=node.getSSQ();
                            //compressionSSQPerLevel[nodeLevel]+=SSQdistance;
                    }//if (centroids distance)
                }//if it is a survival
            }//for all outgoing edges
        }//if has outgoing edges
        node.setChecked(true);
    } //end function

    

    

    

    
    /**
     *Returns the compression results; both space and information loss
     **/
    public StringBuffer getCompressionResults(){
        return this.compressionResults;
    }
    
  
}
