package history;
import java.util.Vector;
import utils.NumericInstance;
/**
 * <p>Title: History</p>
 * <p>Description: A History object describes the evolution between >2 timepoints</p>
 * <p>@author: Eirini Ntoutsi </p>
 * @version 1.0
 */
public class GraphHistory {
    private String historyName;
    private Vector historySlices; //the timePoints of observation, in increasing ordering
    private double[] infoLossPerLevelOfflineCompression;//keeps stats of the distance between the old node centroid T1 and the virtual center to which it has been assigned after the compression
    private int[] compressedCntPerLevelOfflineCompression;
    StringBuffer offlineCompressionResults; //stores compression information

    /**************************************************************************/
    public GraphHistory() {
        historyName = "";
        historySlices = new Vector();
        offlineCompressionResults = new StringBuffer();
    }

  
    /**************************************************************************/
    public void setHistoryName(String val){
       historyName = val;
    }
    /**************************************************************************/
    public void addHistorySlice(GraphLevel level){
        historySlices.add(level);
    }
    /**************************************************************************/
    public GraphLevel getHistorySlice(int i){
        return (GraphLevel)historySlices.get(i);
    }
    /**************************************************************************/
    public int getSize(){
        return this.historySlices.size();
    }
 

    /**
     *Mark all nodes of the history as non summarized
     **/
    public void clearSummarized(){
        for (int i = 0; i<this.getSize();i++){
            GraphLevel level = this.getHistorySlice(i);
            for (int j = 0; j<level.getSize();j++){
                GraphNode node = level.getNode(j);
                node.setOfflineSummarized(false);
                node.setVCenter(new NumericInstance(0, new Vector()));
            }
        }
    }
        
    /**
     *Recursively, visit adjacent nodes of a given node
     **/
    public String getNodeNext(String historyPath, GraphNode startNode){
        if (!startNode.checked){//if this node is not already visited
            for (int m = 0; m < startNode.getNumOfOutEdges(); m++) {
                GraphEdge edge = startNode.getOutEdge(m);
                GraphNode toNode = edge.getToNode();
                if (toNode.hasOutEdges()){//for display purposes
                    historyPath += edge.toString2();
                    historyPath = getNodeNext(historyPath, toNode);
                }
                else{
                    historyPath += edge.toString();
                    historyPath = getNodeNext(historyPath, toNode);
                }
            }
            //after you finish, mark this node as checked
            startNode.setChecked(true);
        }
        return historyPath+"\n";
    }
  


    public GraphNode getNode(int nodeID){
        for (int i = 0; i<this.getSize();i++){
            GraphLevel level = this.getHistorySlice(i);
            for (int j = 0; j<level.getSize();j++){
              GraphNode node = level.getNode(j);
              if (node.getID()==nodeID){
                  return node;
              }
            }
        }
        return null;
    }

    

    /**
     * Print the history
     **/
    public String toString(){
        String res = "";
        for (int i = 0; i<this.getSize();i++){
            res += "Levels:\t "+(i)+" <--> "+(i+1)+"\n";
            res += this.getHistorySlice(i)+"\n";
        }
        return res;
    }
    
    /*--- Traces related functions ---*/   
    /**
     *Splits a trace, and returns a list of subtraces
     **/
    public Vector splitTraceBasedOnDistance(TraceOfNodes theTrace,double maxCentroidsDistance){
        Vector subTracesVector = new Vector();
        TraceOfNodes subTrace = new TraceOfNodes();
        for (int i=0;i<theTrace.getSize()-1;i++){
            GraphNode x = theTrace.getNode(i);
            if (!subTrace.contains(x)) 
                subTrace.addNode(x);
            GraphNode y = theTrace.getNode(i+1);
            double centroidsDistance = x.distance(y);
            if (centroidsDistance<=maxCentroidsDistance){//add y into the current subTrace
                if (!subTrace.contains(y))
                    subTrace.addNode(y);
            }
            else{//export the current subtrace and apply function to rest part of it
                subTracesVector.add(subTrace);//export the current subTrace
                subTrace = new TraceOfNodes();//initialize subtrace
                //find the rest of trace that was not processed so far
                TraceOfNodes restTrace = new TraceOfNodes();
                for (int j=i+1;j<theTrace.getSize();j++) 
                    restTrace.addNode(theTrace.getNode(j));
                //apply the function recursively to the rest of the trace
                splitTraceBasedOnDistance(restTrace, maxCentroidsDistance);
            }
        }//end for       
        if (subTrace.getSize()>0){//just for the case that something has not been exported
            subTracesVector.add(subTrace);
        }
        return subTracesVector;
    }
    
    
    /**
     * Traverse the history within a given time range and detect traces 
     **/
    public void fingerprint(int startLevel, int endLevel,double maxCentroidsDistance){
        offlineCompressionResults.append("NumOfLevels:\t"+(endLevel-startLevel)+"\n");
        offlineCompressionResults.append("max centroids distance:\t"+maxCentroidsDistance+"\n");
        offlineCompressionResults.append("Level\tNODE\tspaceReduction\tinformationLoss\n");        
        this.clearSummarized();//initially set all nodes as non summarized
        for (int i=startLevel ;i<endLevel;i++){
            GraphLevel level = this.getHistorySlice(i);
            for (int j=0; j<level.getSize();j++){
                GraphNode node = level.getNode(j);
                if (!node.getChecked()){//only for nodes non visited/checked so far
                    //-- compute the trace of the cluster
                    TraceOfNodes clusterTrace = getTraceNodeNext(node);
                    //-- split it into subtraces so as C1 condition to hold within a subtrace
                    Vector subTracesVector = splitTraceBasedOnDistance(
                                    clusterTrace,maxCentroidsDistance);
                    for (int k=0;k<subTracesVector.size();k++){
                        //-- check for each subtrace if condition C2 holds
                        TraceOfNodes subTrace = (TraceOfNodes)subTracesVector.get(k);                        
                        StringBuffer tmp = subTrace.vCenter(maxCentroidsDistance); 
                        offlineCompressionResults.append(tmp);
                    }
                }//if the node has not been summarized so far
            }//for all nodes
        }//for all levels
    }
    /**
     *Recursively, visit adjacent nodes of a given node
     **/
    public TraceOfNodes getTraceNodeNext(GraphNode startNode){
        TraceOfNodes traceList = new TraceOfNodes();
        if (!startNode.checked){//if this node is not already visited
            traceList.addNode(startNode);
            //check adjacent node - if it is a survival
            if (startNode.getNumOfOutEdges()==1){
                GraphEdge edge = startNode.getOutEdge(0);
                //only concecutive survivals are added to the trace
                if (edge.getExternalTransition().equalsIgnoreCase("survival")){
                    GraphNode toNode = edge.getToNode();
                    //apply the same procedure to toNode and append its output to traceList
                    TraceOfNodes restTrace = getTraceNodeNext(toNode);                        
                    for (int i=0;i<restTrace.getSize();i++)
                        traceList.addNode(restTrace.getNode(i));
                }
            }
            //after you finish, mark this node as checked
            startNode.setChecked(true);
        }
        return traceList;
    }

    public void evaluateOfflineFingerprintInfoLoss(int startLevel, int endLevel){
        this.infoLossPerLevelOfflineCompression = new double[this.getSize()];
        for (int i=startLevel;i<endLevel;i++){
            GraphLevel level = this.getHistorySlice(i);
            for (int j=0; j<level.getSize();j++){
                GraphNode node = level.getNode(j);
                NumericInstance vCenter = node.getVCenter();//the virtual node to which this node has been assigned
                NumericInstance centroid = node.getCentroid();//the centroid label of this node
                int nodeLevel = node.getLevel()-1;
                if (vCenter!=null){
                    if ((vCenter.getInstanceValues()).size()>0){
                        double clusterInfoLoss = utils.Functions.EuclideanDistance(
                            centroid.getInstanceValues(),vCenter.getInstanceValues());
                        infoLossPerLevelOfflineCompression[i] += clusterInfoLoss;
                    }
                }
            }
        }
    }
    
    public void evaluateOfflineFingerprintSize(int startLevel, int endLevel){
        this.compressedCntPerLevelOfflineCompression = new int[this.getSize()];
        for (int i=startLevel;i<endLevel;i++){
            GraphLevel level = this.getHistorySlice(i);
            for (int j=0; j<level.getSize();j++){
                GraphNode node = level.getNode(j);
                if (node.getOfflineSummarized()){//if it is marked as summarized
                    compressedCntPerLevelOfflineCompression[i]+=1;
                }
            }
        }
    }
    
    /**
     *Returns nodes compressed per level caused by the offline summarization
     **/
    public int[] getOfflineCompressionStats_size(){
        return this.compressedCntPerLevelOfflineCompression;
    }
    
        public StringBuffer getFingerprintResults(){
        return this.offlineCompressionResults;
    }

}
