package history;

import java.util.*;
import utils.ClusterInstances;
import utils.NumericInstance;
/**
 * <p>Title: GraphNode</p>
 * <p>Description: A GraphNode describes a node(cluster) of the Evolution Graph</p>
 * <p>@author: Eirini Ntoutsi </p>
 * @version 1.0
 */
public class GraphNode {
    int ID; //the DB_ID
    String name; //the name of the node/cluster
    int timeStart; //the timepoint of appearance
    int timeEnd; //the timepoint of dissappearance
    Vector outEdges; //the vector of edges starting from this node
    Vector inEdges; //the vector of edges resulting to this node
    boolean checked; //internal variable
    boolean deleted; //internal variable used during online FEG: a node is marked as deleted if it is replaced by another node
    int initialLevel;
    String l;
    String cLabel;//complete label
    String sLabel;//short label
    ClusterInstances instances; //the instances of the cluster
    NumericInstance centroid; //the centroid of the cluster
    NumericInstance virtualCenter;// the virtual center to which it is assigned-it is used in the offline fingerprint
    private boolean isOfflineSummarized;//
    
    /**************************************************************************/
    public GraphNode(int nodeID, String nodeName, int nodeTimeStart, int nodeTimeEnd){
        ID = nodeID;
        name = nodeName;
        timeStart =nodeTimeStart;
        timeEnd =nodeTimeEnd;
        outEdges = new Vector();
        inEdges = new Vector();
        initialLevel=nodeTimeStart;
    }
    
    /**************************************************************************/
    public GraphNode(int nodeID, String nodeName, int nodeTimeStart, int nodeTimeEnd,
            String lbl, String slbl){
        ID = nodeID;
        name = nodeName;
        timeStart =nodeTimeStart;
        timeEnd =nodeTimeEnd;
        outEdges = new Vector();
        inEdges = new Vector();
        initialLevel=nodeTimeStart;
        cLabel=lbl;
        sLabel = slbl;
    }
    
    
    /**************************************************************************/
    public int getID(){
        return ID;
    }
    /**************************************************************************/
    public String getName(){
        return name;
    }
    /**************************************************************************/
    public void setTimeStart(int time){
        this.timeStart =time;
    }
    /**************************************************************************/
    public void setTimeEnd(int time){
        timeEnd =time;
    }
    /**************************************************************************/
    public int getTimeStart(){
        return(timeStart);
    }
    /**************************************************************************/
    public int getTimeEnd(){
        return(timeEnd);
    }
    /**************************************************************************/
    public String getTime(){
        return "[t_"+timeStart+"--t_"+timeEnd+"]";
    }
    public void setLabel( String lbl){
        this.cLabel=lbl;
    }
    public String getCompleteLabel(){
        return this.cLabel;
    }
    
    //check whether there are connections with the next level
    public boolean hasOutEdges(){
        if (this.outEdges.size()>0)
            return true;
        else
            return false;
    }
    /**************************************************************************/
    //check whether there are connections with the previous level
    public boolean hasInEdges(){
        if (this.inEdges.size()>0)
            return true;
        else
            return false;
    }

   
    /**************************************************************************/
    public void setOutEdges(Vector out){
        outEdges = out;
    }
    /**************************************************************************/
    Vector getInEdges(){
        return inEdges;
    }
    /**************************************************************************/
    public void setInEdges(Vector in){
        inEdges = in;
    }
    /**************************************************************************/
    public GraphEdge getOutEdge(int i){
        return (GraphEdge)(outEdges.get(i));
    }
    /**************************************************************************/
    public GraphEdge getInEdge(int i){
        return (GraphEdge)(inEdges.get(i));
    }
    /**************************************************************************/
    public int getNumOfOutEdges(){
        return outEdges.size();
    }    //print nodeEdges
    /**************************************************************************/
    public int getNumOfInEdges(){
        return inEdges.size();
    }
    /**************************************************************************/
     public String printNodeOutEdges(){
         String res = "";
         for (int i=0; i <this.getNumOfOutEdges();i++){
             res += this.getOutEdge(i)+"\n";
         }
        return res;
    }
    /**************************************************************************/
    public String printNodeInEdges(){
        String res = "()";
        if (this.getInEdges().size()>0){
            for (int i = 0; i < this.getNumOfInEdges(); i++) {
                res += this.getInEdge(i) + "\n";
            }
        }
       return res;
   }
   /**************************************************************************/
    public void setChecked(boolean state){
        this.checked = state;
    }
    /**************************************************************************/
    public boolean getChecked(){
        return(this.checked);
    }
    /**************************************************************************/
    public void setDeletedStatus(boolean state){
        this.deleted = state;
    }
    /**************************************************************************/
    public boolean getDeletedStatus(){
        return(this.deleted);
    }
    /**************************************************************************/
    public void setLevel(int state){
        this.initialLevel = state;
    }
    /**************************************************************************/
    public int getLevel(){
        return(this.initialLevel);
    }

    
    /**************************************************************************/
    public String getShortLabel(){
        return this.sLabel;
    }

    //print graphNode
    public String toString(){
        String res = "";
        res += this.name+"("+this.getTime()+")"+"["+this.getShortLabel()+"]";
        return res;
    }
    
    public void setInstances(ClusterInstances insts){
        instances = insts;
    }

    public ClusterInstances getInstances(){
        return instances;
    }
    /**
     *Returns the SSQL of this cluster (i.e. the distance of cluster members from cluster centroid)
     **/
    public double getSSQ(){
        return this.instances.SumSquareDistance(this.centroid);
    }
    
    public int getInstancesCnt(){
        return instances.getInstancesCnt();
    }

    
    public void setCentroid(NumericInstance inst){
        centroid = inst;
    }
    public NumericInstance getCentroid(){
        return centroid;
    }

    public void setVCenter(NumericInstance inst){
        virtualCenter = inst;
    }
    public NumericInstance getVCenter(){
        return virtualCenter;
    }

    public void setOfflineSummarized(boolean status){
        isOfflineSummarized=status;
    }
    public boolean getOfflineSummarized(){
        return isOfflineSummarized;
    }    
    
   
    //Euclidean distance between centroids
    public double distance(GraphNode otherNode){
        NumericInstance thisCentroid = this.centroid;
        NumericInstance otherCentroid = otherNode.centroid;
        double distance = utils.Functions.EuclideanDistance(
                thisCentroid.getInstanceValues(),
                otherCentroid.getInstanceValues());
        //System.out.println(thisCentroid+"\t"+otherCentroid+"\t-->"+distance);
        return distance;
    }
    
    
    /**
     * Computes the average centroid between this node and another node
     **/
    public NumericInstance avgCentroid(GraphNode otherNode){
        NumericInstance thisCentroid = this.centroid;
        NumericInstance otherCentroid = otherNode.centroid;
        Vector thisCentroidValues = thisCentroid.getInstanceValues();
        Vector otherCentroidValues = otherCentroid.getInstanceValues();
        Vector avgCentroidValues = new Vector();
        int n = this.getTimeEnd()-this.getTimeStart();
        if (n<=0)
            n=1;
        for (int i=0;i<thisCentroidValues.size();i++){
            double this_i=((Double)(thisCentroidValues.get(i))).doubleValue();
            double other_i=((Double)(otherCentroidValues.get(i))).doubleValue();
            avgCentroidValues.add((this_i*n+other_i)/(2));            
        }
        return new NumericInstance(this.getID(),avgCentroidValues);
    }
    /**
     *Assign instances to another cluster
     **/
    public void updateInstancesClusterID(int newClusterID){
        ClusterInstances myInstances = this.getInstances();
        for (int i=0;i<myInstances.getInstancesCnt();i++){
            NumericInstance inst = myInstances.getInstance(i);
            //int oldclusterID = inst.getInstanceClusterID();
            inst.setInstanceClusterID(newClusterID);
            //int newclusterID = inst.getInstanceClusterID();
            //System.out.println(oldclusterID+"\t"+newClusterID);
        }
    }
    
    //The MDL cost for modeling a node
    public double MDL(){
        //we only consider the cost of a node
        return 1;
    }

    
}