package history;
import java.util.*;
import utils.NumericInstance;
/**
 * This class describes a trace of nodes (survivals of a cluster in different time points)
 * @author Eirini Ntoutsi
 * @version 1.0
 */
public class TraceOfNodes {
    Vector nodes; //a vector of nodes belonging to this trace    
    //Constructor
    public TraceOfNodes() {
        nodes = new Vector();
    }
    
    //Add a node to the trace
    public void addNode(GraphNode node){
        nodes.add(node);
    }
    //Get the size of the trace
    public int getSize(){
        return nodes.size();
    }    
    //Get node at pos
    public GraphNode getNode(int i){
        return (GraphNode)nodes.get(i);
    }
    //Print the trace
    public String toString(){
        String str="";
        for (int i=0;i<this.getSize();i++){
            str+=getNode(i).getName()+"\t";
        }
        return str;
    }
    
    public boolean contains(GraphNode x){
        boolean res=false;
        for (int i=0;i<this.getSize();i++){
            if (this.getNode(i).equals(x))
                res=true;
        }
        return res;
    }
    

    /**
     *Returns the virtual center of the trace
     **/
    public NumericInstance getVCenter(){
        Vector vCenterValues = new Vector();
        if (this.getSize()>0){
            NumericInstance centroid = (this.getNode(0)).getCentroid();
            Vector centroidValues = centroid.getInstanceValues();
             for (int j=0;j<centroidValues.size();j++)
                vCenterValues.add(j,0.000);
        }
        for (int i=0;i<this.getSize();i++){
            NumericInstance centroid = (this.getNode(i)).getCentroid();
            Vector centroidValues = centroid.getInstanceValues();
            //update vcenter with all attribute-values of this cluster
            for (int j=0;j<centroidValues.size();j++){
                double centroidIValue=((Double)(centroidValues.get(j))).doubleValue();
                double oldVCenterValue=((Double)(vCenterValues.get(j))).doubleValue();
                double newValue = oldVCenterValue+centroidIValue;
                vCenterValues.set(j,newValue);
            }
        }
        int n = this.getSize();
   
        for (int i=0;i<vCenterValues.size();i++){
            double oldValue=((Double)(vCenterValues.get(i))).doubleValue();
            double newValue = (oldValue)/(n);
            vCenterValues.set(i,newValue);
        }        
        return new NumericInstance(0,vCenterValues);
    }
    
    public StringBuffer vCenter(double maxCentroidsDistance){
        StringBuffer results = new StringBuffer();
        //---compute the virtual center of the subtrace
        NumericInstance theVCenter = this.getVCenter();
        //does condition C3 holds for this subtrace?
        boolean C3 = this.checkVCenterDistance(theVCenter,maxCentroidsDistance);
        if (C3){//C3 holds, so just assign the virtual center to these nodes and mark them as summarized
            StringBuffer tmp = this.assingVCenterToNodes(theVCenter);
            results.append(tmp);
        }
        else{//split into the larger distance and re-run vcenter procedure
            Vector splits = splitTraceBasedOnMaxDistance(theVCenter);
            TraceOfNodes leftPart = (TraceOfNodes)splits.get(0);
            TraceOfNodes rightPart = (TraceOfNodes)splits.get(1);
            if (leftPart.getSize()>0) //apply recursively on the left part
                leftPart.vCenter(maxCentroidsDistance);
            if (rightPart.getSize()>0) //apply recursively on the rigth part
                rightPart.vCenter(maxCentroidsDistance);
        }
        return results;
    }
    
    public Vector splitTraceBasedOnMaxDistance(NumericInstance vCenter){
        Vector splits = new Vector();
        double maxDistance = 0.0;
        int foundAt=-1;
        for (int i=0;i<this.getSize();i++){
            NumericInstance nodeCentroid = (this.getNode(i)).getCentroid();
            double distance = utils.Functions.EuclideanDistance(
                   nodeCentroid.getInstanceValues(),vCenter.getInstanceValues());
            if (distance>maxDistance){
                maxDistance = distance;
                foundAt = i;
            }
        }
        TraceOfNodes leftPart = new TraceOfNodes();
        for (int i=0;i<=foundAt;i++){
            leftPart.addNode(this.getNode(i));
        }
        splits.add(leftPart);
        
        TraceOfNodes rightPart = new TraceOfNodes();
        for (int i=foundAt+1;i<this.getSize();i++){
            rightPart.addNode(this.getNode(i));
        }
        splits.add(rightPart);
        return splits;
    }
    
    /**
     *Returns true if vCenter is within <= maxCentroidsDistance from all the clusters of the trace
     **/
    public boolean checkVCenterDistance(NumericInstance vCenter, double maxCentroidsDistance){
        boolean res = true;
        for (int i=0;i<this.getSize();i++){
            GraphNode node = this.getNode(i);
            NumericInstance nodeCentroid = node.getCentroid();
            double distance = utils.Functions.EuclideanDistance(
                   nodeCentroid.getInstanceValues(),vCenter.getInstanceValues());
            if (distance>maxCentroidsDistance){
                res=false;
                break;
            }
        }
        return res;
    }
    /**
     *Set the virtual center of each node/ cluster in the trace to be the vCenter
     **/
    public StringBuffer assingVCenterToNodes(NumericInstance vCenter){
        StringBuffer results = new StringBuffer();
        for (int i=0;i<this.getSize();i++){
            GraphNode node = this.getNode(i);
            int nodeLevel = node.getLevel()-1;
            //infoLoss due to substitution of -node- from avgCentroid
            double nodeInfoLoss = utils.Functions.EuclideanDistance(
                                 (node.getCentroid()).getInstanceValues(),
                                         vCenter.getInstanceValues());                                        
            node.setVCenter(vCenter);
            if (i<this.getSize()-1){//mark all except for the last one
                node.setOfflineSummarized(true);
                results.append(nodeLevel+"\t"+node.getID()+"\t"+1+"\t"+nodeInfoLoss+"\n");
            }
            else
                results.append(nodeLevel+"\t"+node.getID()+"\t"+0+"\t"+nodeInfoLoss+"\n");                
        }
        return results;
    }
}
