
/* Source library do_fits.c
 * Functions for reading and writing FITS files.  */

#define _FILE_OFFSET_BITS 64 
#define _ISOC99_SOURCE

#include <math.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdarg.h>
#include "do_fits.h"

/* Function init_fits
 * This function determines the kind of machine is running your software,
 * and initializes appropriate arrays accordingly.  */

int init_fits(char pname[])
{
    int i;
    unsigned short svalue;
    unsigned int ivalue;
    unsigned long long llvalue;
    static unsigned byte_mask = 255;
    unsigned long long intid;
    char *id;

    dfprogname = pname;
    id = (char *)(&intid);

    /*  Make byte map for machine using this software.  */

    for (i = 0; i < 2; i++)
        id[i] = (char)(1 - i);
    svalue = *((unsigned short *)id);

    for (i = 0; i < 4; i++)
        id[i] = (char)(3 - i);
    ivalue = *((unsigned int *)id);

    for (i = 0; i < 8; i++)
        id[i] = (char)(7 - i);
    llvalue = *((unsigned long long *)id);

    byte_map[0] = 0;
    rev_byte_map[0] = 0;

    for (i = 0; i < 2; i++)
    {
        byte_map[i + 1] = svalue & byte_mask;
        rev_byte_map[(svalue & byte_mask) + 1] = i;
        svalue >>= 8;
    }

    for (i = 0; i < 4; i++)
    {
        byte_map[i + 3] = ivalue & byte_mask;
        rev_byte_map[(ivalue & byte_mask) + 3] = i;
        ivalue >>= 8;
    }

    for (i = 0; i < 8; i++)
    {
        byte_map[i + 7] = llvalue & byte_mask;
        rev_byte_map[(llvalue & byte_mask) + 7] = i;
        llvalue >>= 8;
    }

/*  Diagnostic option:  print the byte map:  */
/*  printf("\nchar:\n");
    for (i = 0; i < 1; i++)
        printf("%5d    %4d %4d\n", i, byte_map[i + 0], rev_byte_map[i + 0]);
    printf("\nshort:\n");
    for (i = 0; i < 2; i++)
        printf("%5d    %4d %4d\n", i, byte_map[i + 1], rev_byte_map[i + 1]);
    printf("\nlong:\n");
    for (i = 0; i < 4; i++)
        printf("%5d    %4d %4d\n", i, byte_map[i + 3], rev_byte_map[i + 3]);
    printf("\nlong long:\n");
    for (i = 0; i < 8; i++)
        printf("%5d    %4d %4d\n", i, byte_map[i + 7], rev_byte_map[i + 7]);
    printf("\n");  */

    inhead       = NULL;
    current_head = NULL;
    text_buff    = NULL;
    nax          = NULL;
    dax          = NULL;
    cdax         = NULL;

    return 1;
}

/* Function map_bytes
 * This function assumes that the character array specified by the second 
 * argument has been read from a FITS file, and rearanges it accordingly
 * so that it can be converted to floating point format.  This same function 
 * reconverts a character array interpretation of floating-point data back to 
 * FITS format ready to be written to the disk.  The proper conversion generally 
 * depends on the BITPIX variable.  */

int map_bytes(unsigned int map[], char buff[], int num_words)
{
    unsigned int i, j, offset, tmp[MAX_DAT_SIZE];

    if (bitpix > 0)
        dat_size =  bitpix/8;
    else
        dat_size = -bitpix/8;

    if (bitpix == 8)
        offset = 0;
    else if (bitpix == 16)
        offset = 1;
    else if (bitpix == 32 || bitpix == -32)
        offset = 3;
    else if (bitpix == 64 || bitpix == -64)
        offset = 7;

    for (i = 0; i < num_words; i++)
    {
        for (j = 0; j < dat_size; j++)
            tmp[j] = buff[dat_size*i + j];
        for (j = 0; j < dat_size; j++)
            buff[dat_size*i + j] = tmp[map[j + offset]];
    }

    return dat_size;
}

/* Function set_scale
 * This function sets the fits variables b_scale, corresponding to the
 * keyword "BSCALE", and b_zero, corresponding to "BZERO" for data that
 * is to be formatted in integers, assuming that the floating point range 
 * of the data so represented is that given by the two arguments.  */

int set_scale(double min, double max)
{
    int i;
    static int num_facts  = 5;
    static double factor[] = {1.0, 2.0, 4.0, 5.0, 8.0};
    double tmp, order, dynam_range, nom_bscale;

    if (min > max)
    {
        tmp = max;
        max = min;
        min = tmp;
    }

    if (bitpix == 8)
        dynam_range = (double)DYNAM_RANGE_BYTE;
    else if (bitpix == 16)
        dynam_range = (double)DYNAM_RANGE_SHORT;
    else if (bitpix == 32)
        dynam_range = (double)DYNAM_RANGE_INT;
    else if (bitpix == 64)
        dynam_range = (double)DYNAM_RANGE_LL;
    else if (bitpix == -32)
        dynam_range = 0.;
    else if (bitpix == -64)
        dynam_range = 0.;
    else
        return EOF;


    if (bitpix == 8)
    {
        max = max - min;
        b_scale = max/dynam_range;
        b_zero  = min;
    }
    else if (bitpix == -32 || bitpix == -64)
    {
        b_scale = 0.;
        b_zero  = 0.;
    }
    else
    {
        if (min < 0.)
            min = -min;
        if (max < 0.)
            max = -max;
        if (min > max)
            max = min;

        b_zero  = 0.;
        if ((nom_bscale = max/dynam_range) == 1.)
        {
            b_scale = nom_bscale;
            return 1;
        }
        else if (nom_bscale > 1.)
        {
            for (i = 0, order = 1.; i <= MAX_SIG_FIGS; i++, order *= 10.)
                if (nom_bscale <= order)
                {
                    b_scale = order;
                    break;
                }
            if (i > MAX_SIG_FIGS)
            {
                b_scale = 10*order;
                return EOF;
            }
        }
        else if (nom_bscale < 1.)
        {
            for (i = 0, order = 1.; i <= MAX_SIG_FIGS; i++, order *= 0.1)
                if (nom_bscale >= order)
                {
                    b_scale = 10*order;
                    break;
                }
            if (i > MAX_SIG_FIGS)
            {
                b_scale = 10*order;
                return 0;
            }
        }
        for (i = 1; i < num_facts; i++)
        {
            if (b_scale*factor[i] > 10.0*nom_bscale)
            {
                b_scale *= .1*factor[i];
                return 1;
            }
        }
    }
    return 1;
}

/* Function pad_file
 * This function determines the current length of the FITS file indicated
 * by it argument and appends the minimum number of zero values to the end 
 * of it to make it an appropriate length for a FITS file, i.e. a multiple
 * of "BSIZE".  */

int pad_file(int fid)
{
    int i, pad;

    if ((fits_excess = (lseek(fid, 0, SEEK_END) % BSIZE)) > 0)
    {
        pad = BSIZE - fits_excess;
        for (i = 0; i < pad; i++)
            io_buff[i] = '\0';
        return write(fid, io_buff, pad);
    }
    return 0;
}

/* Function init_buff
 * This function writes the character value indicated by the
 * second argument into the first n values of the character array
 * indicated by the first argument, where n is the integer indicated
 * by the third argument.  */

int init_buff(char buff[], char c, int n)
{
    int i;

    for (i = 0; i < n; i++)
        buff[i] = c;

    return i;
}

/* Function get_head
 * This function reads the head of the FITS file whose file ID is expressed
 * by the first argument creates a character array of the approriate size,
 * whose address is written into the second argument, and writes the
 * FITS head into that array.  */

int get_head(int fid, char **pbuff)
{
    char line[LINE_LENGTH + 1], seg[LINE_LENGTH + 1];
    int i, num_head_lines, num_recs, stat;
    long long remain;

    remain = lseek(fid, 0, SEEK_END);
    lseek(fid, 0, SEEK_SET);

    if (fid == EOF)
    {
        fprintf(stderr, "%s:  Function get_head:  File not opened.\n", 
            dfprogname);
        return EOF;
    }

    line[LINE_LENGTH] = '\0';
    for (i = 0; (stat = read(fid, line, LINE_LENGTH)) == LINE_LENGTH; i++)
    {
        if (i == 0)
            if (seg_cmp(line, "SIMPLE ", 7) != 0)
                return NOT_FITS_HEAD;

        if (seg_cmp(line, "END ", 4) == 0)
        {
            i++;
            break;
        }
        remain -= LINE_LENGTH;
        if (remain < LINE_LENGTH)
        {
            fprintf(stderr, "%s:  Fcn get_head:  No END keyword in sight.\n",
                dfprogname);
            return NOT_FITS_HEAD;
        }
    }
    num_head_lines = i;

    if (stat < LINE_LENGTH)
    {
        fprintf(stderr, "%s:  Function get_head:\n", dfprogname);
        fprintf(stderr, "File ID %d becomes unreadable at ", fid);
        fprintf(stderr, "line %d; read status %d characters.\n", i, stat); 

        return EOF;
    }

    num_recs = num_head_lines/NUM_LINES;
    if (num_recs*NUM_LINES != num_head_lines)
        num_recs++;

    /*  Make space and store the fits head.  */

    *pbuff = (char *)malloc(BSIZE*(num_recs + HEAD_EXCESS + 1));
    init_buff(*pbuff, ' ', (num_recs + HEAD_EXCESS)*BSIZE);
    (*pbuff)[BSIZE*(num_recs + HEAD_EXCESS)] = '\0';
    lseek(fid, 0, SEEK_SET);
    read(fid, *pbuff, num_recs*BSIZE);

    for (i = 0; i < num_recs*BSIZE; i++) /* Replace all <null> characters */
        if ((*pbuff)[i] == '\0')         /* with something printable so  */
            (*pbuff)[i] = '#';           /* seg_copy will not terminate. */

    return num_recs;
}

/* Function count_head_lines
 * This function returns the number of LINE_LENGTH-character lines in
 * the FITS head contained in the array indicated by the first argument.  */

int count_head_lines(char buff[])
{
    char line[LINE_LENGTH + 1], seg[LINE_LENGTH + 1];
    int i, stat;

    line[LINE_LENGTH] = '\0';
    for (i = 0; (stat = seg_copy(buff + i*LINE_LENGTH, line, LINE_LENGTH)) 
    == LINE_LENGTH; i++)
    {
        if (i == 0)
            if (seg_cmp(line, "SIMPLE ", 7) != 0)
                return NOT_FITS_HEAD;

        if (seg_cmp(line, "END ", 4) == 0)
        {
             i++;
             break;
        }
    }
    if (stat < LINE_LENGTH)
    {
        fprintf(stderr, "%s:  Fcn count_head_lines:  no END keyword.\n",
            dfprogname);
        return EOF;
    }
    return i;
}

/* Function get_keyword_value
 * Read the fits head identified by the first argument, find the keyword
 * indentified by the second argument, and, using the format identified
 * by the third argument, determine the value assigned by the fits head
 * to that keyword and write it into the pointer location identified by
 * the fourth argument.  The function returns the line number in the
 * header at which the keyword occurs.  The usage is as follows:
 *
 *     get_keyword_value(
           char head_name[], 
           char keyword[], 
           char format[], 
                *value); 
 */

int get_keyword_value(
char head[], 
char keyword[], 
char format[], 
...)
{
    int location, length, stat;
    va_list ap;

    va_start(ap, format);

    if ((length = seg_len(keyword)) > 8)
    {
        fprintf(stderr, 
            "%s: Function get_keyword_value:  Keyword %s too long.", 
            dfprogname, keyword);
        fprintf(stderr, "(Max 8 characters.)\n");
        return EOF;
    }
    length++;
    if (length > 8)
        length = 8;

/*  seg_copy(keyword, keywd, 8);  */

    if ((location = find_keyword(head, keyword)) == EOF)
        return EOF;

    stat = EOF;
    if      (seg_cmp(format, "%hi", 3) == 0)
        stat = sscanf(head + LINE_LENGTH*location + 10, "%hd",
            va_arg(ap, short *));
    else if (seg_cmp(format, "%i",  3) == 0)
        stat = sscanf(head + LINE_LENGTH*location + 10, "%d",
            va_arg(ap, int *));
    else if (seg_cmp(format, "%li", 3) == 0)
        stat = sscanf(head + LINE_LENGTH*location + 10, "%ld",
            va_arg(ap, long *));
    else if (seg_cmp(format, "%lli", 3) == 0)
        stat = sscanf(head + LINE_LENGTH*location + 10, "%lld",
            va_arg(ap, long long *));
    else if (seg_cmp(format, "%f",  3) == 0)
        stat = sscanf(head + LINE_LENGTH*location + 10, "%e",
            va_arg(ap, float *));
    else if (seg_cmp(format, "%e",  3) == 0)
        stat = sscanf(head + LINE_LENGTH*location + 10, "%e",
            va_arg(ap, float *));
    else if (seg_cmp(format, "%le",  4) == 0)
        stat = sscanf(head + LINE_LENGTH*location + 10, "%le",
            va_arg(ap, double *));
    else if (seg_cmp(format, "%lf",  4) == 0)
        stat = sscanf(head + LINE_LENGTH*location + 10, "%le",
            va_arg(ap, double *));
    else if (seg_cmp(format, "%s",  3) == 0)
        stat = sscanf(head + LINE_LENGTH*location + 10, "%s",
            va_arg(ap, char *));

    va_end(ap);

    if (stat == EOF)
        return NOT_VALID_FORMAT;
    if (stat == 0)
        return NOT_VALID_VALUE;

    return location;
}

/* Function get_keyword_comment
 * Read the fits head identified by the first argument, find the keyword
 * indentified by the second argument, and, using the format identified
 * by the third argument, determine the value assigned by the fits head
 * to that keyword and write it into the pointer location identified by
 * the fourth argument.  The usage is as follows:
 *
 *     get_keyword_comment(
           char head_name[], 
           char keyword[], 
           char comment[]);
 */

int get_keyword_comment(
char head[], 
char keyword[], 
char comment[])
{
    int i, location, length, stat;

    if ((length = seg_len(keyword)) > 8)
    {
        fprintf(stderr, 
            "%s: Function get_keyword_value:  Keyword %s too long.", 
            dfprogname, keyword);
        fprintf(stderr, "(Max 8 characters.)\n");
        return EOF;
    }
    length++;
    if (length > 8)
        length = 8;

    if ((location = find_keyword(head, keyword)) == EOF)
        return EOF;

    stat = seg_copy(head + LINE_LENGTH*location + 31, comment, 49);

    for (i = 48; i >= 0; i--)
    {
        if (comment[i] != ' ')
        {
            comment[++i] = '\0';
            break;
        }
    }
    if (i == -1)
        comment[0] = '\0';

    return 1;
}

/* Function put_keyword_value
 * This function writes the keyword indicated by the third argument, and
 * its value determined from a variable associated with its fifth argument,
 * in accordance with a format indicated by the fourth argument, into the
 * FITS head contained in the character array indicated by the first 
 * argument, into the line indexed by the second argument, along with a
 * comment whose indicated by the sixth argument.  The usage is as follows:
 *
 *     put_keyword_value(
 *         char head_name[], 
 *         int  line_num, 
 *         char keyword[], 
 *         char format[], 
 *              value, 
 *         char comment[]);
 *
 * The value is left unchanged and only the comment changed if the format
 * format is set to "%0" and the value to "", for example.
 */

int put_keyword_value(
char      head[], 
int       location, 
char      keyword[], 
char      format[], 
...) 
{
    char   kfield[11], vfield[31], cfield[51], *comment;
    char   sfmt[10];
    int    write_vfield, slen;
    short  vsh;
    int    vint;
    long   vl;
    long long vll;
    float  vfl;
    double vdbl;
    char   *pv;
    long   *lp;
    va_list ap;

    va_start(ap, format);

    if (seg_len(keyword) > 8)
    {
        fprintf(stderr, "%s:  put_keyword:  keyword %s is too long.\n", 
            dfprogname, keyword);
        return EOF;
    }

    if (seg_len(keyword) > 0)
    {
        sprintf(kfield, "%-8s= ", keyword);
        seg_copy(kfield, head + LINE_LENGTH*location, 10);
    }

    slen = 20;
    write_vfield = 1;
    if      (seg_cmp(format, "%hi",   3) == 0) 
    { 
        vsh = (short)va_arg(ap, int); 
        sprintf(vfield, "%20hd",  vsh); 
    } 
    else if (seg_cmp(format, "%i",   3) == 0) 
    { 
        vint = va_arg(ap, int); 
        sprintf(vfield, "%20d",   vint); 
    } 
    else if (seg_cmp(format, "%li",   3) == 0) 
    { 
        vl = va_arg(ap, long); 
        sprintf(vfield, "%20ld",   vl); 
    } 
    else if (seg_cmp(format, "%lli",   3) == 0) 
    { 
        vll = va_arg(ap, long long); 
        sprintf(vfield, "%20lld",   vll); 
    } 
    else if (seg_cmp(format, "%f",  4) == 0) 
    {
        vfl = (float)va_arg(ap, double);
        sprintf(vfield, "%20.6f",  vfl);
    }
    else if (seg_cmp(format, "%e",  3) == 0)
    {
        vfl = (float)va_arg(ap, double);
        sprintf(vfield, "%20.6e",  vfl);
    }
    else if (seg_cmp(format, "%lf",  4) == 0) 
    {
        vdbl = va_arg(ap, double);
        sprintf(vfield, "%20.6lf",  vdbl);
    }
    else if (seg_cmp(format, "%le",  3) == 0)
    {
        vdbl = va_arg(ap, double);
        sprintf(vfield, "%20.6le",  vdbl);
    }
    else if (seg_cmp(format, "%s",   3) == 0)
    {
        pv = va_arg(ap, char *);

        if ((slen = seg_len(pv)) < 20)
            slen = 20;

        sprintf(sfmt, "%%%ds", slen);
        sprintf(vfield, sfmt,  pv);
    }
    else
    {
        pv = va_arg(ap, char *);
        write_vfield = 0;
    }

    comment = va_arg(ap, char *);
    if (write_vfield)
        seg_copy(vfield, head + LINE_LENGTH*location + 10, slen);

    if ((slen = seg_len(comment)) > 0)
    {
        sprintf(cfield, " %-49s", comment);
        seg_copy(cfield, head + LINE_LENGTH*location + 30, 50);
    }

    va_end(ap);

    return 1;
}

/* Function find_keyword
 * This function returns the line number in the FITS header indicated by
 * the first argument in which the keyword indicated by its second argument
 * and returns it.  If the keyword is not found, the function returns EOF.  */

int find_keyword(char head[], char keyword[])
{
    char keywd[9];
    int i, length;

    init_buff(keywd, ' ', 8);
    keywd[8] = '\0';

    if ((length = seg_len(keyword)) > 8)
    {
        fprintf(stderr, "%s:  Function find_keyword:  Keyword %s too long.  ", 
            dfprogname, keyword);
        fprintf(stderr, "(Max 8 characters.)\n");
        return EOF;
    }
    length++;
    if (length > 8)
        length = 8;

    seg_copy(keyword, keywd, 8);

    for (i = 0; 1; i++)
    {
        if (seg_cmp(keywd, head + i*LINE_LENGTH, length) == 0)
            return i;
        if (seg_cmp("END ", head + i*LINE_LENGTH, 4) == 0)
            return EOF;
    }
}

/* Function revise_keyword_value
 * This function finds the keyword indicated by the second argument in
 * the fits header contained in the array indicated by the first argument
 * and revises its value to that given by its fifth argument, in 
 * accordance with the format given by the fourth argument.  The usage
 * is as follows:
 *
 *    revise_keyword_value(
 *        char head[], 
 *        char keyword[], 
 *        char format[], 
 *             value,
 *        char comment[]);
 *
 * If the format is "%0", then only the comment is revised.
 * If the comment is not to be revised, then the null comment, "",
 * should be the last argument.  A FREQUENT BUG IS THAT CAUSED BY
 * LEAVING OFF THE COMMENT!!.
 */

int revise_keyword_value(
char head[], 
char keyword[], 
char format[], 
...)
{
    int location;
    va_list ap;
    char *pv, *pc;
    float vfl;
    int vint;

    if ((location = find_keyword(head, keyword)) == EOF)
        return EOF;

    va_start(ap, format);

    if      (seg_cmp(format, "%i",   3) == 0)
    {
        vint = va_arg(ap, int);
        pc   = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, vint, pc);
    }
    else if (seg_cmp(format, "%f",   3) == 0)
    {
        vfl  = va_arg(ap, double);
        pc   = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, vfl, pc);
    }
    else if (seg_cmp(format, "%e",   3) == 0)
    {
        vfl  = va_arg(ap, double);
        pc   = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, vfl, pc);
    }
    else if (seg_cmp(format, "%lf",  4) == 0)
    {
        vfl  = va_arg(ap, double);
        pc   = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, vfl, pc);
    }
    else if (seg_cmp(format, "%le",  4) == 0)
    {
        vfl  = va_arg(ap, double);
        pc   = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, vfl, pc);
    }
    else if (seg_cmp(format, "%s",   3) == 0)
    {
        pv   = va_arg(ap, char *);
        pc   = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, pv, pc);
    }
    else if (seg_cmp(format, "%0",   3) == 0)
    {
        pv   = va_arg(ap, char *);
        pc   = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, "", pc);
    }

    va_end(ap);
    return 1;
}

int revise_keyword(
char head[], 
char keyword[], 
char new_keyword[])
{
    int location;

    if ((location = find_keyword(head, keyword)) == EOF)
        return EOF;

    put_keyword_value(head, location, new_keyword, "%0", "", "");

    return 1;
}

/* Function insert_keyword_value
 * This function inserts a line into the FITS header in the array
 * indicated by the first argument at the location indicated by the second
 * argument.  The line contains the keyword indicated by the third argument
 * and its value is given by the fifth arguent in accordance with the 
 * format given by the fourth argument.  The usage is as follows:
 *
 *     insert_keyword_value(
 *         char head_name[], 
 *         int  line_num, 
 *         char keyword[], 
 *         char format[], 
 *              value, 
 *         char comment[]);
 *
 * If the value of the integer argument line_num is zero, then the 
 * new keyword and it value are inserted just before the "END" keyword
 * in the FITS header.  A FREQUENT BUG IS THAT CAUSED BY LEAVING OFF
 * THE COMMENT!!!
 */

int insert_keyword_value(
char head[], 
long location, 
char keyword[], 
char format[], 
...)
{
    int i, end;
    va_list ap;
    long vlong;
    double vfl;
    char *pv, *pc;

    end = find_keyword(head, "END");
    if (location > end || location < 0)
        return EOF;
    if (location == 0)
        location = end;

    for (i = (end + 1)*LINE_LENGTH; i >= location*LINE_LENGTH; i--)
        head[i + LINE_LENGTH] = head[i];

    for (i = location*LINE_LENGTH; i < (location + 1)*LINE_LENGTH; i++)
        head[i] = ' ';

    va_start(ap, format);

    if      (seg_cmp(format, "%i",   3) == 0)
    {
        vlong = va_arg(ap, long);
        pc    = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, vlong, pc);
    }
    else if (seg_cmp(format, "%f",   3) == 0)
    {
        vfl   = va_arg(ap, double);
        pc    = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, vfl, pc);
    }
    else if (seg_cmp(format, "%e",   3) == 0)
    {
        vfl   = va_arg(ap, double);
        pc    = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, vfl, pc);
    }
    else if (seg_cmp(format, "%lf",   3) == 0)
    {
        vfl   = va_arg(ap, double);
        pc    = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, vfl, pc);
    }
    else if (seg_cmp(format, "%le",   3) == 0)
    {
        vfl   = va_arg(ap, double);
        pc    = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, vfl, pc);
    }
    else if (seg_cmp(format, "%s",   3) == 0)
    {
        pv    = va_arg(ap, char *);
        pc    = va_arg(ap, char *);
        put_keyword_value(head, location, keyword, format, pv, pc);
    }

    va_end(ap);

    return (end + 2)/NUM_LINES - (end + 1)/NUM_LINES;
}

/* Function delete_keyword
 * This function deletes the first occurence of the line containing
 * the keyword indicated by the second argument in the FITS header
 * in the character array whose address is indicated by the first 
 * argument.  */

int delete_keyword(char head[], char keyword[])
{
    int i, end, location;

    if ((end = find_keyword(head, "END")) == EOF)
    {
        fprintf(stderr, "%s:  Function delete_keyword:  ", dfprogname);
        fprintf(stderr, "no END keyword in FITS header.\n");
        return EOF;
    }
    if ((location = find_keyword(head, keyword)) == EOF)
        return EOF;

    for (i = location*LINE_LENGTH; i < end*LINE_LENGTH; i++)
        head[i] = head[i + LINE_LENGTH];

    init_buff(head + end*LINE_LENGTH, ' ', LINE_LENGTH);

    return (end + 1)/NUM_LINES - end/NUM_LINES;
}

/* Function read_head
 * This function reads the FITS head contained in the first argument and
 * sets the nominal FITS dimensional and scaling variables appropriately. */

int read_head(char buff[])
{
    char line[LINE_LENGTH + 1], keyword[LINE_LENGTH + 1];
    int i, num_head_lines;

    if ((num_head_lines = count_head_lines(buff)) < 1)
    {
        fprintf(stderr, "%s:  Function read_head:  Defective FITS head.\n",
            dfprogname);
        return EOF;
    }

    naxis       = 0;
    bitpix      = 0;
    dat_size    = 0;
    b_zero      = 0.;
    b_scale     = 1.;
    ref_min     = 0.;
    ref_max     = 0.;
    ref_min_set = 0;
    ref_max_set = 0;

    if (get_keyword_value(buff, "SIMPLE", "%s", line) != 0)
    {
        fprintf(stderr, "%s:  Function read_head:  ", dfprogname);
        fprintf(stderr, "Argument not a FITS header.\n");
        return EOF;
    }
/*  if (seg_cmp(line, "T", 2) != 0)
    {
        fprintf(stderr, "%s:  Function read_head:  ", dfprogname);
        fprintf(stderr, "Argument not a standard FITS head.\n");
        fprintf(stderr, "Continuing--\n");
    }  */
    if (get_keyword_value(buff, "BITPIX", "%i", &bitpix) == EOF)
    {
        fprintf(stderr, "%s:  Function read_head:  Keyword BITPIX missing.\n",
            dfprogname);
        return EOF;
    }
    if (bitpix != 8 && bitpix != 16 && bitpix != 32 && bitpix != -32 
    && bitpix != 64 && bitpix != -64)
    {
        bitpix = 0;
        fprintf(stderr, "%s:  Function read_head:  BITPIX = %d ", 
            dfprogname, bitpix);
        fprintf(stderr, "not acceptable value.\n");
        return EOF;
    }
    if (bitpix > 0)
        dat_size =  bitpix/8;
    else
        dat_size = -bitpix/8;

    if (get_keyword_value(buff, "NAXIS", "%i", &naxis) == EOF)
    {
        fprintf(stderr, "%s:  Function read_head:  ", dfprogname);
        fprintf(stderr, "Keyword NAXIS not found.\n");
        return EOF;
    }

    if (naxis <= 0)
    {
        fprintf(stderr, "%s:  Function read_head:  NAXIS = %d must be > 0.\n", 
            dfprogname, naxis);
        return EOF;
    }

    nax  = (int    *)malloc((naxis + AXSLACK)*sizeof(int   ));
    dax  = (double *)malloc((naxis + AXSLACK)*sizeof(double));
    cdax = (double *)malloc((naxis + AXSLACK)*sizeof(double));

    for (i = 0; i < naxis; i++)
    {
        sprintf(keyword, "NAXIS%d", i + 1);
        if (get_keyword_value(buff, keyword, "%i", nax + i) == EOF)
        {
            fprintf(stderr, "%s:  Function read_head:  \n", dfprogname);
            fprintf(stderr, "Keyword %s not found.\n", keyword);
            return EOF;
        }

        sprintf(keyword, "DAXIS%d", i + 1);
        if (get_keyword_value(buff, keyword, "%le", dax + i) == EOF)
        {
            sprintf(keyword, "DAXIS%d", i + 1);
            if (get_keyword_value(buff, keyword, "%le", dax + i) == EOF)
                dax[i] = 1.0;
        }

        sprintf(keyword, "CDELT%d", i + 1);
        if (get_keyword_value(buff, keyword, "%le", cdax + i) == EOF)
        {
            sprintf(keyword, "CDELT%d", i + 1);
            if (get_keyword_value(buff, keyword, "%le", cdax + i) == EOF)
                cdax[i] = 1.0;
        }
    }

    if (get_keyword_value(buff, "BZERO",  "%le", &b_zero)   == EOF)
        b_zero = 0.;

    if (get_keyword_value(buff, "BSCALE", "%le", &b_scale)  == EOF)
        b_scale = 1.;

    if (get_keyword_value(buff, "SIGMIN", "%le", &sig_min)  != EOF)
        ref_min_set = 1;

    if (get_keyword_value(buff, "SIGMAX", "%le", &sig_max)  != EOF)
        ref_max_set = 1;

    num_head_recs = num_head_lines/NUM_LINES;
    if (num_head_recs*NUM_LINES != num_head_lines)
        num_head_recs++;

    for (i = 0, num_pts = 1; i < naxis; i++)
        num_pts *= nax[i];

    return num_head_recs;
}

/* Function make_head
 * This function creates a nominal FITS head, referring to standard
 * FITS dimensional and scaling variables, and writes it into an array
 * whose address is indicated by the first argument.  */

int make_head(char **pbuff)
{
    char keyword[8];
    int i, line_count, num_recs;

    num_recs = 1;
    if (bitpix == 0)
    {
        fprintf(stderr, "%s:  Function make_head: ", dfprogname);
        fprintf(stderr, "Variable bitpix is zero.  Cannot make head.\n");
        return EOF;
    }

    if (bitpix > 0)
        dat_size =  bitpix/8;
    else
        dat_size = -bitpix/8;

    *pbuff = (char *)malloc(num_recs*BSIZE + 1);
    init_buff(*pbuff, ' ', num_recs*BSIZE);
    (*pbuff)[num_recs*BSIZE] = '\0';

    line_count = 0;

    put_keyword_value(*pbuff, line_count++, "SIMPLE", "%s", "T",     " ");
    put_keyword_value(*pbuff, line_count++, "BITPIX", "%i", bitpix,  " ");
    put_keyword_value(*pbuff, line_count++, "NAXIS",  "%i", naxis,   " ");
    for (i = 0; i < naxis; i++)
    {
        sprintf(keyword, "NAXIS%d", i + 1);
        put_keyword_value(*pbuff, line_count++, keyword,  "%i", nax[i],  " ");
    }
    if (bitpix > 0)
    {
        put_keyword_value(*pbuff, line_count++, "BZERO",  "%le", b_zero,  " ");
        put_keyword_value(*pbuff, line_count++, "BSCALE", "%le", b_scale, " ");
    }
    for (i = 0; i < naxis; i++)
    {
        sprintf(keyword, "DAXIS%d", i + 1);
        put_keyword_value(*pbuff, line_count++, keyword, "%le", dax[i], " ");
    }
    for (i = 0; i < naxis; i++)
    {
        sprintf(keyword, "CDELT%d", i + 1);
        put_keyword_value(*pbuff, line_count++, keyword, "%le", cdax[i], " ");
    }

    seg_copy("END", *pbuff + LINE_LENGTH*line_count++, 3);

    return num_recs;
}

/* Function update_head
 * This function updates the FITS head to which its argument points, 
 * in accordance with standard FITS dimensional and scaling variables. 
 * If the update can increase the length of the head, it is important
 * that the head buffer have the space to accommodate it.  This can
 * be assured by proper application of the function "copy_head" before
 * running this function.  */

int update_head(char head[])
{
    int i, old_naxis;
    int num_head_lines, nhr, wcs_set, dax_set;
    char varname[LINE_LENGTH + 1];
    char maxnax_name[LINE_LENGTH + 1];
    int  ref_loc;

    wcs_set = 0;
    dax_set = 0;

    get_keyword_value(head, "NAXIS", "%i", &old_naxis);

    if (revise_keyword_value(head, "BITPIX", "%i", bitpix,  "") == EOF)
        return EOF;
    if (revise_keyword_value(head, "NAXIS",  "%i", naxis,   "") == EOF)
        return EOF;
    if (bitpix > 0
    && revise_keyword_value(head, "BSCALE", "%le", b_scale, "") == EOF)
    {
        sprintf(maxnax_name, "NAXIS%d", naxis);
        ref_loc = find_keyword(head, maxnax_name) + 1;
        if (insert_keyword_value(head, ref_loc, "BSCALE", "%le", b_scale, "") 
        == EOF)
            return EOF;
    }
    if (bitpix > 0
    &&  revise_keyword_value(head, "BZERO",  "%le", b_zero,  "") == EOF)
    {
        ref_loc = find_keyword(head, "BSCALE") + 1;
        if (insert_keyword_value(head, ref_loc, "BZERO", "%le", b_zero, "") 
        == EOF)
            return EOF;
    }

    if (naxis <= old_naxis)
    {
        for (i = 0; i < naxis; i++)
        {
            sprintf(varname, "NAXIS%d", i + 1);
            if (revise_keyword_value(head, varname, "%i", nax[i], "") 
            == EOF)
                return EOF;

            sprintf(varname, "DAXIS%d", i + 1);
            if (revise_keyword_value(head, varname, "%le", dax[i], "") 
            == EOF)
            {
                fprintf(stderr, "Function update_head:  ");
                fprintf(stderr, "Keyword %s not found.  ", varname);
                fprintf(stderr, "No update made.\n");
            }
            else
                dax_set = 1;

            sprintf(varname, "CDELT%d", i + 1);
            if (revise_keyword_value(head, varname, "%le", cdax[i], "") 
            == EOF)
            {
                fprintf(stderr, "Function update_head:  ");
                fprintf(stderr, "Keyword %s not found.  ", varname);
                fprintf(stderr, "No update made.\n");
            }
            else
                wcs_set = 1;
        }

        for (i = naxis; i < old_naxis; i++)
        {
            sprintf(varname, "NAXIS%d", i + 1);
            if (delete_keyword(head, varname) == EOF)
                return EOF;

            sprintf(varname, "DAXIS%d", i + 1);
            if (delete_keyword(head, varname) == EOF)
            {
                fprintf(stderr, "Function update_head:  ");
                fprintf(stderr, "Keyword %s not found, ", varname);
                fprintf(stderr, "hence not deleted.\n");
            }
            else
                dax_set = 1;

            sprintf(varname, "CDELT%d", i + 1);
            if (delete_keyword(head, varname) == EOF)
            {
                fprintf(stderr, "Function update_head:  ");
                fprintf(stderr, "Keyword %s not found, ", varname);
                fprintf(stderr, "hence not deleted.\n");
            }
            else
                wcs_set = 1;
        }
    }
    else /* naxis > old_naxis */
    {
        for (i = 0; i < old_naxis; i++)
        {
            sprintf(varname, "NAXIS%d", i + 1);
            if (revise_keyword_value(head, varname, "%i", nax[i], "") == EOF)
                return EOF;

            sprintf(varname, "DAXIS%d", i + 1);
            if (revise_keyword_value(head, varname, "%le", dax[i], "") == EOF)
            {
                fprintf(stderr, "Function update_head:  ");
                fprintf(stderr, "Keyword %s not found, ", varname);
                fprintf(stderr, "hence not deleted.\n"); 
            }
            else
                dax_set = 1;

            sprintf(varname, "CDELT%d", i + 1);
            if (revise_keyword_value(head, varname, "%le", cdax[i], "") 
            == EOF)
            {
                fprintf(stderr, "Function update_head:  ");
                fprintf(stderr, "Keyword %s not found, ", varname);
                fprintf(stderr, "hence not deleted.\n"); 
            }
            else
                wcs_set = 1;
        }



        sprintf(varname, "NAXIS%d", old_naxis);
        ref_loc = find_keyword(head, varname) + 1;
        for (i = old_naxis; i < naxis; i++)
        {
            sprintf(varname, "NAXIS%d", i + 1);
            if (insert_keyword_value(head, ref_loc, varname, "%i", nax[i], "")
            == EOF)
                return EOF;
            ref_loc++;
        }

        if (dax_set)
        {
            sprintf(varname, "DAXIS%d", old_naxis);
            ref_loc = find_keyword(head, varname) + 1;
            for (i = old_naxis; i < naxis; i++)
            {
                sprintf(varname, "DAXIS%d", i + 1);
                if (insert_keyword_value(head, ref_loc, varname, "%le", 
                dax[i], "") == EOF)
                {
                    fprintf(stderr, "Function update_head:  ");
                    fprintf(stderr, "Unable to insert keyword %s ", varname);
                    fprintf(stderr, "at location %d.\n", ref_loc);
                    fprintf(stderr, "Proceeding without insertion.\n");
                }
                ref_loc++;
            }
        }

        if (wcs_set)
        {
            sprintf(varname, "CDELT%d", old_naxis);
            ref_loc = find_keyword(head, varname) + 1;
            for (i = old_naxis; i < naxis; i++)
            {
                sprintf(varname, "CDELT%d", i + 1);
                if (insert_keyword_value(head, ref_loc, varname, "%le", 
                cdax[i], "") == EOF)
                {
                    fprintf(stderr, "Function update_head:  ");
                    fprintf(stderr, "Unable to insert keyword %s ", varname);
                    fprintf(stderr, "at location %d.\n", ref_loc);
                    fprintf(stderr, "Proceeding without insertion.\n");
                }
                ref_loc++;
            }
        }
    }


    num_head_lines = count_head_lines(head);
    nhr = num_head_lines/NUM_LINES;
    if (nhr*NUM_LINES < num_head_lines)
        nhr++;

    return nhr;
}

/* Function put_head
 * This function writes the FITS head contained in the array indicated by
 * its second argument into the beginning of the file whose file ID is
 * expressed by the first argument.  */

int put_head(int fid, char buff[])
{
    int i, num_recs, num_head_lines;
    char line[LINE_LENGTH + 1];

    /*  Count lines in inhead and check for existence of "END" keyword.  */

    line[LINE_LENGTH] = '\0';
    for (i = 0; seg_copy(buff + i*LINE_LENGTH, line, LINE_LENGTH) 
    == LINE_LENGTH; i++)
    {
        if (seg_cmp(line, "END ", 4) == 0)
        {
            i++;
            break;
        }
    }

    if (seg_cmp(line, "END ", 4) != 0)
    {
        fprintf(stderr, "%s:  Function put_head:  No END keyword in sight.\n",
            dfprogname);
        return EOF;
    }

    num_head_lines = i;
    num_recs = num_head_lines/NUM_LINES;
    if (num_recs*NUM_LINES != num_head_lines)
        num_recs++;

    if (write(fid, buff, num_recs*BSIZE) != num_recs*BSIZE)
        return EOF;

    return num_recs;
}

/* Function read_fits
 * This function reads the FITS file whose file ID is indicated by the 
 * first argument and set the pointer indicated by the second argument 
 * to the address an array that contains the array data.  The standard 
 * FITS dimensional and scaling variables are read from the header and 
 * set accordingly.  The FITS file must be small enough that its contents
 * can be fit into available memory.  */

int read_fits(int fid, double **pdat_fp)
{
    if ((num_head_recs = get_head(fid, &inhead)) < 0)
        return EOF;
    if (read_head(inhead) < 0)
        return EOF;

    *pdat_fp = (double *)malloc(num_pts*sizeof(double));
    if (input_data(fid, *pdat_fp, num_pts) == EOF)
    {
        fprintf(stderr, "%s:  Premature end of file encountered in read.  ",
            dfprogname); 
        fprintf(stderr, "Exiting.\n"); 
        exit(1);
    }
    min_max(*pdat_fp, num_pts);

    /* Set default value for bitpix for output. */

    return 1;
}

/* Function write_fits
 * This function writes the data contained in the second argument to
 * a FITS file whose file ID is indicated by the first argument.
 * If the global pointer current_head is set to NULL when write_fits
 * is called, a minimal header is created and written into the output
 * FITS file.  If current_head is set to a non-null address, it is 
 * assumed that this address points to the beginning of a FITS header.
 * That header is examined and rescaled appropriately, its dimensions 
 * revised in accordance with the global variable naxis and the 
 * naxis-dimensional arrays nax[] and dax[], and the revised header
 * written into the output FITS file.  It is the responsibility of
 * the programer to secure that current_head is allotted sufficient
 * space for the foregoing revision.  If current_head may not have
 * sufficient space for the revision, then the function copy_head 
 * can provide the extra space needed before write_fits is called,
 * for example, as follows:  
 *
 *     copy_head(current_head, &new_head, NUM_EXTRA_RECS_NEEDED);
 *     free(current_head);
 *     current_head = new_head;
 */

int write_fits(int fid, double dat_fp[])
{
    char varname[LINE_LENGTH + 1];
    int i, stat, old_naxis;

    if (fid == EOF)
        return EOF;

    for (i = 0, num_pts = 1; i < naxis; i++)
        num_pts *= nax[i];

    min_max(dat_fp, num_pts);
    set_scale(sig_min, sig_max);

    if (current_head == NULL)
    {
        if ((num_head_recs = make_head(&current_head)) == EOF)
        {
            fprintf(stderr, "%s:  Function write_fits:  ", dfprogname);
            fprintf(stderr, "Error in making head.\n");
            return EOF;
        }
    }
    else
        update_head(current_head);

    put_head(fid, current_head);

    stat = output_data(fid, dat_fp, num_pts);

    pad_file(fid);

    return 1;
}

/* This function is probably obsolete */

int put_line(char line[], char buff[]) 
{
    static int line_count;

    if (seg_len(line) == 0)  /* Reset */
    {
        line_count = 0;
        return 0;
    }

    seg_copy(line, buff + line_count*LINE_LENGTH, LINE_LENGTH);
    line_count++;

    return line_count;
}

/* Function inint
 * This function returns the long long-integer round-off value of its floating
 * point argument.  */

long long inint(double x)
{
    if (x >= 0.)
        return (long long)(x + .5);

    return (long long)(x - .5);
}

/* Function input_data
 * This function reads the first n values from the FITS file indicated by
 * the file ID given by the first argument, starting at the current location
 * of the file pointer, where n is the integer value given by the third 
 * argument.  It interprets these appropriately as floating point values, 
 * using FITS scaling variables that have to have been read from the FITS 
 * header.  It creates a floating point array, whose address it writes 
 * into the location given by the second argument, and writes the foregoing
 * floating-point values into that array.  */

int input_data(int fid, double data_fp[], int n)
{
    int i, count, num_read, bytes_to_read;

    if (bitpix > 0)
        dat_size =  bitpix/8;
    else
        dat_size = -bitpix/8;

    dat_byte   = (unsigned char *)io_buff;
    dat_short  = (short         *)io_buff;
    dat_int    = (int           *)io_buff;
    dat_ll     = (long long     *)io_buff;
    dat_flt    = (float         *)io_buff;
    dat_dbl    = (double        *)io_buff;

    for(count = 0; (bytes_to_read = block_size(n, count)) > 0; 
    count += num_read)
    {
        if ((num_read = read(fid, io_buff, bytes_to_read)/dat_size) == 0) 
            return EOF;

        if (dat_size == 2 || dat_size == 4 || dat_size == 8)
            map_bytes(rev_byte_map, io_buff, BSIZE/dat_size);

        if (bitpix == 8)
            for (i = 0; i < num_read; i++)
            {
            /*  if (dat_byte[i] == DYNAM_RANGE_BYTE)
                    data_fp[i + count] = NAN;
                else */
                    data_fp[i + count] = b_scale*(double)dat_byte[i]  + b_zero;
            }
        else if (bitpix ==  16)
        {
            for (i = 0; i < num_read; i++)
            {
                if (dat_short[i] == -DYNAM_RANGE_SHORT - 1)
                    data_fp[i + count] = NAN;
                else
                    data_fp[i + count] = b_scale*(double)dat_short[i] + b_zero;
            }
        }
        else if (bitpix ==  32)
        {
            for (i = 0; i < num_read; i++)
            {
                if (dat_int[i] == -DYNAM_RANGE_INT - 1)
                    data_fp[i + count] = NAN;
                else
                    data_fp[i + count] = b_scale*(double)dat_int[i]  + b_zero;
            }
        }
        else if (bitpix ==  64)
            for (i = 0; i < num_read; i++)
            {
                if (dat_ll[i] == DYNAM_RANGE_LL)
                    data_fp[i + count] = NAN;
                else
                    data_fp[i + count] = b_scale*(double)dat_ll[i]    + b_zero;
            }
        else if (bitpix == -32)
            for (i = 0; i < num_read; i++)
                data_fp[i + count] = dat_flt[i];
        else if (bitpix == -64)
            for (i = 0; i < num_read; i++)
                data_fp[i + count] = (double)dat_dbl[i];
    }

    return count;
}

/* Function output_data
 * This function reads the first n values from the floating-point array
 * given by the second argument, where n is the integer value given by
 * the third argument.  It uses appropriate FITS scaling and mapping
 * variables to write the values into the FITS file identified by the
 * first argument, starting at the current location of the file pointer.  */ 

int output_data(int fid, double data_fp[], int n)
{
    int i, count, num_written, bytes_to_write;
    float factor, offset, value;
    double mindat, maxdat;

    dat_byte  = (unsigned char *)io_buff;
    dat_short = (short *)io_buff;
    dat_int   = (int   *)io_buff;
    dat_ll    = (long long *)io_buff;
    dat_flt   = (float *)io_buff;
    dat_dbl   = (double *)io_buff;

    if (b_scale == 0.)
    {
        factor = 0.;
        offset = 0.;
    }
    else
    {
        factor = 1./b_scale;
        offset = b_zero;
    }

    if (bitpix == 8)
    {
        maxdat = b_zero + b_scale*(double)DYNAM_RANGE_BYTE;
        mindat = b_zero;
    }
    else if (bitpix == 16)
    {
        maxdat = b_zero + b_scale*(double)DYNAM_RANGE_SHORT;
        mindat = b_zero - b_scale*(double)DYNAM_RANGE_SHORT;
    }
    else if (bitpix == 32)
    {
        maxdat = b_zero + b_scale*(double)DYNAM_RANGE_INT;
        mindat = b_zero - b_scale*(double)DYNAM_RANGE_INT;
    }
    else if (bitpix == 64)
    {
        maxdat = b_zero + b_scale*(double)DYNAM_RANGE_LL;
        mindat = b_zero - b_scale*(double)DYNAM_RANGE_LL;
    }

    if (bitpix > 0)
        dat_size =  bitpix/8;
    else
        dat_size = -bitpix/8;

    if (bitpix == 8)
    {
        for (count = 0; (bytes_to_write = block_size(n, count)) > 0; 
        count += num_written)
        {
            for (i = 0; i < bytes_to_write/dat_size; i++)
            {
                value = data_fp[i + count];
                if (isnormal(value) == 0 && fpclassify(value) == FP_SUBNORMAL)
                    value = 0.;
                else if (value != 0. && isnormal(value) == 0)
                    dat_byte[i] = (unsigned char)DYNAM_RANGE_BYTE;
                else if (value >= maxdat)
                    dat_byte[i] = (unsigned char)DYNAM_RANGE_BYTE;
                else if (value <= mindat)
                    dat_byte[i] = (unsigned char)0;
                else
                    dat_byte[i] = 
                        (unsigned char)inint(factor*(value - b_zero));
            }
            if ((num_written = write(fid, io_buff, bytes_to_write)) 
            != bytes_to_write)
                return EOF;
            num_written /= dat_size;
        }
    }

    else if (bitpix == 16)
    {
        for (count = 0; (bytes_to_write = block_size(n, count)) > 0; 
        count += num_written)
        {
            for (i = 0; i < bytes_to_write/dat_size; i++)
            {
                value = data_fp[i + count];
                if (isnormal(value) == 0 && fpclassify(value) == FP_SUBNORMAL)
                    value = 0.;
                if (value != 0. && isnormal(value) == 0)
                    dat_short[i] = -(short)DYNAM_RANGE_SHORT - 1;
                else if (value >= maxdat)
                    dat_short[i] =  (short)DYNAM_RANGE_SHORT;
                else if (value <= mindat)
                    dat_short[i] = -(short)DYNAM_RANGE_SHORT;
                else
                    dat_short[i] = (short)inint(factor*(value - b_zero));
            }
            map_bytes(byte_map, io_buff, BSIZE/dat_size);
            if ((num_written = write(fid, io_buff, bytes_to_write)) 
            != bytes_to_write)
                return EOF;
            num_written /= dat_size;
        }
    }

    else if (bitpix == 32)
    {
        for (count = 0; (bytes_to_write = block_size(n, count)) > 0; 
        count += num_written)
        {
            for (i = 0; i < bytes_to_write/dat_size; i++)
            {
                value = data_fp[i + count];
                if (isnormal(value) == 0 && fpclassify(value) == FP_SUBNORMAL)
                    value = 0.;
                if (value != 0. && isnormal(value) == 0)
                    dat_int[i] = -(long)DYNAM_RANGE_INT - 1;
                else if (value >= maxdat)
                    dat_int[i] =  (long)DYNAM_RANGE_INT;
                else if (value <= mindat)
                    dat_int[i] = -(long)DYNAM_RANGE_INT;
                else
                    dat_int[i] = (long)inint(factor*(value - b_zero));
            }
            map_bytes(byte_map, io_buff, bytes_to_write/dat_size);
            if ((num_written = write(fid, io_buff, bytes_to_write)) 
            != bytes_to_write)
                return EOF;
            num_written /= dat_size;
        }
    }

    else if (bitpix == 64)
    {
        for (count = 0; (bytes_to_write = block_size(n, count)) > 0; 
        count += num_written)
        {
            for (i = 0; i < bytes_to_write/dat_size; i++)
            {
                value = data_fp[i + count];
                if (isnormal(value) == 0 && fpclassify(value) == FP_SUBNORMAL)
                    value = 0.;
                if (value != 0. && isnormal(value) == 0)
                    dat_ll[i] = -(long long)DYNAM_RANGE_LL - 1;
                else if (value >= maxdat)
                    dat_ll[i] =  (long long)DYNAM_RANGE_LL;
                else if (value <= mindat)
                    dat_ll[i] = -(long long)DYNAM_RANGE_LL;
                else
                    dat_ll[i] =  (long long)inint(factor*(value - b_zero));
            }
            map_bytes(byte_map, io_buff, BSIZE/dat_size);
            if ((num_written = write(fid, io_buff, bytes_to_write)) 
            != bytes_to_write)
                return EOF;
            num_written /= dat_size;
        }
    }

    else if (bitpix == -32)
    {
        for (count = 0; (bytes_to_write = block_size(n, count)) > 0;
        count += num_written)
        {
            for (i = 0; i < bytes_to_write/dat_size; i++)
                dat_flt[i] = (float)data_fp[i + count];
            map_bytes(byte_map, io_buff, BSIZE/dat_size);
            if ((num_written = write(fid, io_buff, bytes_to_write))
            != bytes_to_write)
                return EOF;
            num_written /= dat_size;
        }
    }

    else if (bitpix == -64)
    {
        for (count = 0; (bytes_to_write = block_size(n, count)) > 0;
        count += num_written)
        {
            for (i = 0; i < bytes_to_write/dat_size; i++)
                dat_dbl[i] = data_fp[i + count];
            map_bytes(byte_map, io_buff, BSIZE/dat_size);
            if ((num_written = write(fid, io_buff, bytes_to_write))
            != bytes_to_write)
                return EOF;
            num_written /= dat_size;
        }
    }

    return count;
}

/* Function min_max
 * This function determines the mininum and maximum values of the first n 
 * values of the floating-point array indicated by its first argument, 
 * where n is given by the integer second argument.  It writes these into 
 * the external FITS variables sig_min and sig_max.  */

int min_max(double data[], int n)
{
    int i;

    for (i = 0; i < n; i++)
        if (data[i] == 0.0 || isnormal(data[i]) != 0)
            break;
   
    if (i == n)
    {
        sig_min = NAN;
        sig_max = NAN;
        return 0;
    }

    for (sig_min = (sig_max = data[i]); i < n; i++)
        if (data[i] == 0.0 || isnormal(data[i]) != 0)
        {
            if (data[i] < sig_min)
                sig_min = data[i];
            if (data[i] > sig_max)
                sig_max = data[i];
        }

    if (ref_min_set == 0)
        ref_min = sig_min;
    if (ref_max_set == 0)
        ref_max = sig_max;

    if (sig_max  > sig_min)
        return 1;
    if (sig_max == sig_min)
        return 0;

    return EOF;
}

/*  function nom_ord_range
 *  This function determines the ordinate range of the FITS file whose 
 *  file identifier, "fid", is its first argument and writes it into the 
 *  external variables sig_min and sig_max.  If BITPIX for the file is
 *  positive and the second argument, "mode", is set to HEAD_RANGE, the 
 *  ordinate range is the full range accommodated by the value of BSCALE
 *  and BZERO in the FITS header.  If "mode" is set to DATA_RANGE or
 *  BITPIX is negative, the ordinate range is simply that in the data
 *  contained in the file.  */

int nom_ord_range(int fid, int mode)
{
    int    i, read_size;
    double cdr, sig_min_comp, sig_max_comp;
    off_t  remain;

    /*  The argument "mode" can have only one of two values.  Otherwise,
     *  squawk and exit.  */

    if (mode != HEAD_RANGE && mode != DATA_RANGE)
    {
        fprintf(stderr, "%s:  Value %d not permitted for argument mode ",
            dfprogname, mode);
        fprintf(stderr, "in function nom_ord_range\n");

        return EOF;
    }

    /*  Read the FITS header.  */

    lseek(fid, 0, SEEK_CUR);
    if ((num_head_recs = get_head(fid, &inhead)) < 0)
        return EOF;

    if (nax != NULL)
        free(nax);
    if (dax != NULL)
        free(dax);
    if (read_head(inhead) < 0)
        return EOF;

    /*  If BITPIX is positive, set the dynamic range accordingly.  */

    if (bitpix == 8)
        cdr = DYNAM_RANGE_BYTE;
    else if (bitpix == 16)
        cdr = DYNAM_RANGE_SHORT;
    else if (bitpix == 32)
        cdr = DYNAM_RANGE_INT;
    else if (bitpix == 64)
        cdr = DYNAM_RANGE_LL;

    /*  If BITPIX is positive and the argument "mode" is set to HEAD_RANGE,
     *  then set the ordinate range to the full range accommodated by the
     *  BSCALE and BZERO, and return.  */

    if (bitpix > 0 && mode == HEAD_RANGE)
    {
        sig_min = -b_scale*cdr + b_zero;
        sig_max =  b_scale*cdr + b_zero;

        return 1;
    }

/*  if (bitpix < 0 || mode == DATA_RANGE)  */

    /*  Scan the ordinate values of the file, determine the minimum and
     *  maximum value, and write them into sig_min and sig_max, 
     *  respectively.  This is done in blocks.  */

    read_size = STD_BLOCK_SIZE;
    for (i = 0, remain = num_pts; remain > 0; i++)
    {
        if (remain < read_size)
            read_size = remain; 
        remain -= input_data(fid, df_datbuff, read_size);
        min_max(df_datbuff, read_size);
        if (i == 0)
        {
            sig_min_comp = sig_min;
            sig_max_comp = sig_max;
        }
        else
        {
            if (sig_min < sig_min_comp)
                sig_min_comp = sig_min;
            if (sig_max > sig_max_comp)
                sig_max_comp = sig_max;
        }
    }
    sig_min = sig_min_comp;
    sig_max = sig_max_comp;

    return 1;
}

/*  Function get_text
 *  This function reads text from the input indicated by the first argument
 *  and writes it into a character buffer created to contain the text and 
 *  whose address is written into the address specified by the second 
 *  argument.  */

int get_text(FILE *fp, char **pbuff)
{
    char line[LINE_LENGTH + 2];
    int i, j, len, num_head_lines;

    for (i = 0; fgets(line, LINE_LENGTH + 2, fp) != NULL; i++)
        ;
    num_head_lines = i;
    *pbuff = (char *)malloc(num_head_lines*(LINE_LENGTH + 2));
    rewind(fp);
    for (i = 0, j = 0; fgets(line, LINE_LENGTH + 2, fp) != NULL; i++)
    {
        len = seg_copy(line, *pbuff + j, LINE_LENGTH + 1);
        (*pbuff)[j + len] = '\n';
        j += len + 1;
    }
    return num_head_lines;
}

/* Function head_to_text
 * This function reads the FITS head whose address is expressed by the
 * first argument, converts it to standard text, with appropriate
 * line feeds, creates a character array whose address is written to
 * the location specified by the second argument, and writes the text
 * into that array.  */

int head_to_text(char head[], char **ptext)
{
    char line[LINE_LENGTH + 1];
    int i, j, k, end, num_head_lines;

    if ((num_head_lines = count_head_lines(head)) <= 0)
        return NOT_FITS_HEAD;
    *ptext = (char *)malloc(num_head_lines*(LINE_LENGTH + 1));

    line[LINE_LENGTH] = '\0';

    for (i = 0, k = 0, end = 0; end == 0; i++)
    {
        seg_copy(head + i*LINE_LENGTH, line, LINE_LENGTH);
        if (seg_cmp(line, "END ", 4) == 0)
            end = 1;
        for (j = LINE_LENGTH - 1; j >= 0; j--)
            if (line[j] != ' ')
                break;
        seg_copy(line, (*ptext) + k, ++j);
        (*ptext)[k + j] = '\n';
        j++;
        k += j;
    }
    (*ptext)[k] = '\0';
    return k; 
}

/* Function text_to_head
 * This function reads the text array whose address is expressed by the
 * first argument, converts it to standard FITS-head format, creates a 
 * character array whose address is written to the location specified by 
 * the second argument, and writes the FITS head into that array.  */

int text_to_head(char text[], char **phead)
{
    char line[LINE_LENGTH + 1];
    int i, j, k, num_head_lines, len;

    for (i = 0, num_head_lines = 0; text[i] != '\0'; i++)
        if (text[i] == '\n')
            num_head_lines++;

    num_head_recs = num_head_lines/NUM_LINES;
    if (num_head_recs*NUM_LINES != num_head_lines)
        num_head_recs++;

    *phead = (char *)malloc(num_head_recs*BSIZE);
    init_buff(*phead, ' ', num_head_recs*BSIZE);

    for (i = 0, j = 0, k = 0; i < num_head_lines; i++)
    {
        len = seg_copy(text + j, line, LINE_LENGTH);
        init_buff(line + len, ' ', LINE_LENGTH - len);
        line[LINE_LENGTH] = '\0';
        seg_copy(line, *phead + k, LINE_LENGTH);
        j += len + 1;
        k += LINE_LENGTH;
    }

    return num_head_lines;
}

/* Function print_head
 * This function prints the text contained in the array indicated by its
 * argument to the standard output.  */

int print_head(char head[])
{
    char *tmptext; 

    if (head_to_text(head, &tmptext) <= 0)
    {
        fprintf(stderr, "%s:  Function print_head:  ", dfprogname);
        fprintf(stderr, "Not a proper fits head.\n");
        free(tmptext);
        return EOF;
    }

    fprintf(stdout, "%s", tmptext);
    free(tmptext);
    return 1;
}

/* Function block_size
 * This function returns how many of "n" elements are to be read or
 * written having after "count" groups of "BSIZE" elements have been
 * read or written.  */

int block_size(int n, int count)
{
    int remain;

    remain = (n - count)*dat_size; 
    if (remain > BSIZE)
        return BSIZE;

    return remain;
}

/* Function copy_head
 * This function reads the FITS header pointed to by its first argument,
 * allocates appropriate space and coppies it into that space setting
 * the pointer indicated by the second argument to that space.  For
 * the eventuality of extra space being needed, it will add the number
 * of header records specified by the third argument to the nominal
 * space allotment required by the header being read.
 */ 

int copy_head(char head[], char **pnewhead,
int num_extra_recs)
{
    int i, end, num_recs;

    if ((end = find_keyword(head, "END ")) == EOF)
    {
        fprintf(stderr, "%s:  Function copy_head:  ", dfprogname);
        fprintf(stderr, "No END keyword in FITS header.\n");
        return EOF;
    }

    num_recs = end/NUM_LINES;
    if (num_recs*NUM_LINES != end)
        num_recs++;
    num_recs += num_extra_recs;
    *pnewhead = (char *)malloc(num_recs*BSIZE + 1);
    for (i = 0; i < num_recs*BSIZE; i++)
        (*pnewhead)[i] = ' ';
    (*pnewhead)[num_recs*BSIZE] = '\0';
    seg_copy(head, *pnewhead, (end + 1)*LINE_LENGTH);

    return num_recs;
}

/* Function seg_cmp
 * This function compares the character strings indicated by the first two 
 * arguments up to character n, given by the third argument.  It return 
 * zero if the two strings are equal thereto or to the end of the string, 
 * whichever comes first, and a non zero integer otherwise. */

int seg_cmp(char line1[], char line2[], int n)
{
    int i, diff;

    for (i = 0; i < n; i++)
    {
        if ((diff = line2[i] - line1[i]) != 0)
            return diff;
        if (line1[i] == '\0')
            return 0;
    }
    return 0;
}

/* Function seg_copy
 * This function coppies the first n characters of the string indicated 
 * by the first argument onto the string indicated by the second argument,
 * unless a string terminator or end-of-line character is encountered,
 * at which point the function returns the number of characters coppied
 * up to that point without duplicating the end-of-line or end-of-string. */

int seg_copy(char line1[], char line2[], int n)
{
    int i, j, start;

    for (i = 0; i < n; i++)
    {
        if (line1[i] == '\0' || line1[i] == '\n')
            return i;
        line2[i] = line1[i];
    }
    return i;
}

/* Function seg_len
 * This function returns the length of the character string indicated by
 * its argument. */

int seg_len(char line[])
{
    int i;

    for (i = 0; i < LINE_LENGTH; i++)
        if (line[i] == '\0' || line[i] == '\n')
            return i;

    return EOF;
}

/* Function compare_fp_values
 * Compare floating point arguments.  If float arguments do not compare 
 * within the fraction specified by the macro ERROR_FRAC_LIMIT, then 
 * return 0.  Otherwise return 1. */

int compare_fp_values(double x, double y)
{
    float error_frac;

    error_frac = 2.*(x - y)/(x + y);
    if (error_frac > ERROR_FRAC_LIMIT || error_frac < -ERROR_FRAC_LIMIT)
        return 0;

    return 1;
}

int reverse_endian(char buff[], int n)
{
    char temp;
    int  i, j, nwords, hds;

    if (bitpix > 0)
        dat_size =  bitpix/8;
    else
        dat_size = -bitpix/8;

    if (dat_size == 1)
        return 0;            /* Nothing to be done. */

    if (n%dat_size != 0)
        return -1;

    nwords = n/dat_size;
    hds    = dat_size/2;
    for (i = 0; i < nwords; i++)
        for (j = 0; j < hds; j++)
        {
            temp                 = buff[dat_size*i + j];
            buff[dat_size*i + j] = buff[dat_size*i + dat_size - j - 1];
            buff[dat_size*i + dat_size - j - 1] = temp;
        }

    return 1;
}

/* Function extend_data 
 * This function moves the contents of a file whose identification is
 * specified by the integer first argument endward a number of bytes 
 * specified by the integer third argument starting with the byte whose
 * index is specified by the integer second argument.  In so doing, the 
 * size of the file is increased by the value of the third argument.
 * If the extension is less than zero, the file is contracted accordingly.
 */

int extend_data(int fid, int start, int extension)
{
    char *data, temp;
    long long int i, read_size, end0;

    if (extension == 0)
        return 0;

    end0   = lseek(fid, 0, SEEK_END);

    data = (char *)malloc(STD_BLOCK_SIZE*sizeof(char));

    if (extension > 0)
        for (i = end0; i > start; )
        {
            read_size = i - start;
            if (read_size > STD_BLOCK_SIZE)
                read_size = STD_BLOCK_SIZE;
            i -= read_size;

            lseek(fid, i, SEEK_SET);
            read(fid, data, read_size);
            lseek(fid, i + extension, SEEK_SET);
            write(fid, data, read_size);
        }
    if (extension < 0)
        for (i = start; i < end0; i += read_size)
        {
            read_size = end0 - i;
            if (read_size > STD_BLOCK_SIZE)
                read_size = STD_BLOCK_SIZE;

            lseek(fid, i, SEEK_SET);
            read(fid, data, read_size);
            lseek(fid, i + extension, SEEK_SET);
            write(fid, data, read_size);
        }

    free(data);
    return 1;
}
