/*  Program laplacian
 *  This program computes the gradient vector of a two-dimensional fits
 *  file and writes it to a three-dimensional fits file, the third
 *  dimension spanning two values to accommodate (0) the x-component
 *  and (1) the y-component of the gradient.  Compile this program
 *  with the command
 *
 *               "gcc -O3 laplacian.c -o laplacian -lm".
 */

#define _FILE_OFFSET_BITS 64

#include <math.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include "do_fits.h"
 
#define PI                            3.14159265359

#define RESET                         0
#define CONTINUE                      1
#define END_OF_CYCLE                  1
#define CONTINUING                    2
#define START                         3
#define WRONG_NUM_COLS               -1
#define ROW_FAILURE                  -2
#define COL_FAILURE                  -3

void print_diag(char filename[]);
int  init_laplacian(double dat_in[]);
int  compute_laplacian(double ampl[]);

static off_t  bpf, ppf;
static double *gradx, *grady, nu;
static double twopi = 2.*PI;
static double **data_in, **data_out;

int main(int argc, char *argv[])
{
    char   line[LINE_LENGTH + 1];
    char   maxnax[10];
    int    i, j, fid_in, fid_out;
    int    num_frames;
    int    maxnax_loc;
    off_t  dat_start;
    double *ampl;
    double bscale_in, bzero_in, bscale_out, *bs;
    FILE   *dest;

    init_fits(argv[0]);

    if (argc != 3)
    {
        if (argc == 1)
            dest = stdout;
        else
            dest = stderr;

        fprintf(dest, "\nUsage:  %s input output\n", dfprogname);

        if (argc == 1)
            exit(0);
        else
            exit(1);
    }

    /*  Open the input file and read the fits head.  */

    if ((fid_in = open(argv[1], O_RDONLY)) == EOF)
    {
        fprintf(stderr, "%s:  Cannot open file %s.  Exiting.\n", 
            dfprogname, argv[1]);
        exit(1);
    }

    if (get_head(fid_in, &inhead) == EOF)
    {
        fprintf(stderr, "%s:  Defective fits head in file %s.  Exiting.\n",
            dfprogname, argv[3]);
        exit(1);
    }
    if (read_head(inhead) == EOF)
    {
        fprintf(stderr, "%s:  Cannot read fits head in file %s.  Exiting.\n",
            dfprogname, argv[3]);
        exit(1);
    }

    lseek(fid_in, 0, SEEK_CUR);
    if (bitpix == 0)
    {
        fprintf(stderr, 
            "%s:  Read failed:  BITPIX zero or unspecified.\n", 
            dfprogname);
        exit(1);
    }
    if (naxis < 2)
    {
        fprintf(stderr, 
            "%s:  FITS keyword NAXIS (here %d) must be at least 2.  ", 
            dfprogname, naxis);
        fprintf(stderr, "Exiting.\n");
        exit(1);
    }
    bscale_in = b_scale;
    bzero_in  = b_zero;

    if ((fid_out = open(argv[2], O_RDWR | O_CREAT | O_EXCL, 0644))
    == EOF)
    {
        printf("%s:  File %s already exists.  Overwrite?  ", 
            dfprogname, argv[2]);
        if (fgets(line, LINE_LENGTH + 1, stdin), line[0] != 'y')
            exit(0);
        if ((fid_out = open(argv[2], 
            O_WRONLY | O_CREAT | O_TRUNC, 0644)) == EOF)
        {
            fprintf(stderr, "%s:  Cannot open file %s.  Exiting.\n", 
                dfprogname, argv[2]);   
            exit(1);
        }
    }

    for (i = 2, num_frames = 1; i < naxis; i++)
        num_frames *= nax[i];

    dat_size = bitpix/8;
    if (dat_size < 0)
        dat_size = -dat_size;
    ppf = nx*ny;
    bpf = ppf*dat_size;

    ampl  = (double *)malloc(2*ppf*sizeof(double));

    put_head(fid_out, inhead);
    dat_start = lseek(fid_out, 0, SEEK_CUR);

    bs = (double *)malloc(num_frames*sizeof(double));

    init_laplacian(ampl);
    for (i = 0; i < num_frames; i++)
    {
        printf("%s:  Processing frame %d--\n", 
            dfprogname, i);
        b_zero  = bzero_in;
        b_scale = bscale_in;
        input_data(fid_in, ampl, ppf);

        compute_laplacian(ampl);
        min_max(ampl + nx*ny, nx*ny);
        set_scale(sig_min, sig_max);

        if (b_scale > bscale_out)
            bscale_out = b_scale;
        else
            b_scale = bscale_out;
        bs[i] = b_scale;

        b_scale = bscale_out;
        b_zero  = 0.0;
        lseek(fid_out, dat_start + bpf*i, SEEK_SET);
        output_data(fid_out, ampl + nx*ny, ppf);
    }

    /*  Read each frame from the input file, re-scale it, and write it
     *  to the output file.  */

    b_zero  = 0.;
    if (bitpix > 0)
        for (i = 0; i < num_frames; i++)
        {
            if (bs[i] == bscale_out)
                break;

            printf("Rescaling frame %d\n", i);

            b_scale = bs[i];
            lseek(fid_out, dat_start + bpf*i, SEEK_SET);
            input_data(fid_out, ampl, ppf);

            b_scale = bscale_out;
            lseek(fid_out, dat_start + bpf*i, SEEK_SET);
            output_data(fid_out, ampl, ppf);
        }

    /*  Revise the scaling parameters in the FITS header.  */

    if (bitpix > 0)
    {
        if (revise_keyword_value(inhead, "BZERO",  "%e", 0.0,  "") == EOF)
            insert_keyword_value(inhead, naxis + 4, "BZERO", "%e", 0.0, "");
        if (revise_keyword_value(inhead, "BSCALE", "%e", bscale_out, "")
        == EOF)
            insert_keyword_value(inhead, naxis + 5, "BSCALE", "%e",
                bscale_out, "");
    }

    /*  Insert the revised FITS header into the pad file, and pad the
     *  datacube to a multiple of BSIZE.  */

    lseek(fid_out, 0, SEEK_SET);
    put_head(fid_out, inhead);
    pad_file(fid_out);

    close(fid_in);
    close(fid_out);
    free(ampl);
}

int init_laplacian(double dat_in[])
{
    int i;

    data_in  = (double **)malloc(ny*sizeof(double  *));
    data_out = (double **)malloc(ny*sizeof(double  *));

    for (i = 0; i < ny; i++)
    {
        data_in[i]  = dat_in + nx*      i ;
        data_out[i] = dat_in + nx*(ny + i);
    }

    return 1;
}

int compute_laplacian(double dat_in[])
{
    int i, j;
    double one_third;

    one_third = 1./3.;

    for (i = 0; i < ny; i++)
        for (j = 0; j < nx; j++)
            data_out[i][j] = 0.0;

    for (i = 1; i < ny - 1; i++)
        for (j = 1; j < nx - 1; j++)
            data_out[i][j]   =    data_in[i][j - 1] + data_in[i][j + 1]
                             +    data_in[i - 1][j] + data_in[i + 1][j]
                             - 4.*data_in[i][j];

    for (j = 1; j < nx - 1; j++)
    {
        data_out[0     ][j]  =     data_in[     0][j - 1]
                             +     data_in[     0][j + 1]
                             +     data_in[     1][    j]
                             -  3.*data_in[     0][    j];
        data_out[ny - 1][j]  =     data_in[ny - 1][j - 1]
                             +     data_in[ny - 1][j + 1]
                             +     data_in[ny - 2][    j]
                             -  3.*data_in[ny - 1][    j];
    }

    for (i = 1; i < ny - 1; i++)
    {
        data_out[i][     0]  =     data_in[i - 1][     0]
                             +     data_in[i + 1][     0]
                             +     data_in[    i][     1]
                             -  3.*data_in[    i][     0];
        data_out[i][nx - 1]  =     data_in[i - 1][nx - 1]
                             +     data_in[i + 1][nx - 1]
                             +     data_in[    i][nx - 2]
                             -  3.*data_in[    i][nx - 1];
    }

    data_out[     0][     0] =     data_in[     0][     1]
                             +     data_in[     1][     0]
                             -  2.*data_in[     0][     0];
    data_out[     0][nx - 1] =     data_in[     0][nx - 2]
                             +     data_in[     1][nx - 1]
                             -  2.*data_in[     0][nx - 1];
    data_out[ny - 1][     0] =     data_in[ny - 1][     1]
                             +     data_in[ny - 2][     0]
                             -  2.*data_in[ny - 1][     0];
    data_out[ny - 1][nx - 1] =     data_in[ny - 1][nx - 2]
                             +     data_in[ny - 2][nx - 1]
                             -  2.*data_in[ny - 1][nx - 1];

    return 1;
}
