// Simple example editor for drawing Finite State Automata

package fsa_editor;

import au.edu.monash.csse.tonyj.cider.interpreter.*;
import au.edu.monash.csse.tonyj.cider.canvas.InterpretedTokenCanvas;
import au.edu.monash.csse.tonyj.cider.canvas.DrawableToken;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import java.awt.event.KeyListener;
import java.awt.event.KeyEvent;
import javax.swing.JFrame;
import javax.swing.JButton;
import javax.swing.JPanel;
import java.awt.BorderLayout;
import java.awt.GridLayout;

public class Editor extends JFrame implements ActionListener, KeyListener, MouseListener {

	// GUI elements
	private JButton			addArrow;
	private JButton			addCircle;
	private JButton			addText;
	private JButton			nextObject;
	private JButton			previousObject;
	private JButton			deleteObject;
	private JButton			processButton;
	private JButton			quit;
	private JPanel			buttonPanel;
	private InterpretedTokenCanvas	canvas;

	// Needed for mouse dragging
	private boolean			mouseDrag;
	private int			startX;
	private int			startY;

	// Needed to deactivating solver constraints on a symbol
	private boolean			shiftKeyDown;

	// Constants
	public static final int		IMAGE_WIDTH = 600;
	public static final int		IMAGE_HEIGHT = 400;
	public static final int		MOUSE_TOLERANCE = 5;
	public static final int		MIN_RADIUS = 10;

	// Constructor
	public Editor()
	{
		mouseDrag = false;
		shiftKeyDown = false;

		// Make buttons
		addArrow = new JButton("Add Arrow");
		addCircle = new JButton("Add Circle");
		addText = new JButton("Add Text");
		nextObject = new JButton("Next Object");
		previousObject = new JButton("Previous Object");
		deleteObject = new JButton("Delete Object");
		processButton = new JButton("Process Input");
		quit = new JButton("Quit");

		JButton buttonArray[] = new JButton[] { addCircle, addText, addArrow, processButton, nextObject, previousObject, deleteObject, quit};

		// Make button panel
		buttonPanel = new JPanel(new GridLayout(2, 4, 5, 5));
		for (int i = 0; i < buttonArray.length; i++)
			buttonPanel.add(buttonArray[i]);

		// Make token canvas
		canvas = new InterpretedTokenCanvas(IMAGE_WIDTH, IMAGE_HEIGHT, new FSAInterpreter());

		// Add components to frame and initialize
		this.getContentPane().setLayout(new BorderLayout(5, 5));
		this.getContentPane().add(canvas, BorderLayout.CENTER);
		this.getContentPane().add(buttonPanel, BorderLayout.SOUTH);
		this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		this.pack();

		// Add listeners
		canvas.addMouseListener(this);
		canvas.addKeyListener(this);
		for (int i = 0; i < buttonArray.length; i++) {
			buttonArray[i].addActionListener(this);
			buttonArray[i].addKeyListener(this);
		}

		// Show the frame
		this.setTitle("FSA Editor");
		this.show();
	}

	// Deals with button presses
	public void actionPerformed(ActionEvent ae)
	{
		if (ae.getSource() == quit)
			System.exit(0);
		else if (ae.getSource() == nextObject) {
			if (canvas.getTopToken() == null)
				return;
			deactivateTop();
			canvas.moveToTop(canvas.getBottomToken());
			activateTop();
			canvas.repaint();
		}
		else if (ae.getSource() == previousObject) {
			if (canvas.getTopToken() == null)
				return;
			deactivateTop();
			canvas.moveToBottom(canvas.getTopToken());
			activateTop();
			canvas.repaint();
		}
		else if (ae.getSource() == deleteObject) {
			if (canvas.getTopToken() == null)
				return;
			canvas.remove(canvas.getTopToken());
			activateTop();
			canvas.repaint();
		}
		else if (ae.getSource() == addCircle) {
			deactivateTop();
			addToCanvasTop(new DrawableCircle(20.0, 40.0, 200.0, true));
			canvas.repaint();
		}
		else if (ae.getSource() == addArrow) {
			deactivateTop();
			addToCanvasTop(new DrawableArrow(20.0, 200.0, 40.0, 200.0, 60.0, 200.0, 0.0, 0.0, true));
			canvas.repaint();
		}
		else if (ae.getSource() == addText) {
			deactivateTop();
			addToCanvasTop(new DrawableText("", 40.0, 200.0, false, true));
			canvas.repaint();
		}
		else if (ae.getSource() == processButton) {
			deactivateTop();
			canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_MopUpEmpties);
			boolean success = canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_CheckForConclusion);
			if (!success)
				success = canvas.getInterpreter().applyTransformationInParallel(FSAInterpreter.TRANSFORMATION_ProcessInput);
			activateTop();
			canvas.repaint();
		}
	}

	// Activates the top token
	private void activateTop()
	{
		if (canvas.getNumTokens() != 0) {
			if (canvas.getTopToken() instanceof Text)
				((Text) canvas.getTopToken()).set_active(true);
			else if (canvas.getTopToken() instanceof Circle)
				((Circle) canvas.getTopToken()).set_active(true);
			else if (canvas.getTopToken() instanceof Arrow)
				((Arrow) canvas.getTopToken()).set_active(true);
		}
	}

	// Deactivates the top token
	private void deactivateTop()
	{
		if (canvas.getNumTokens() != 0) {
			if (canvas.getTopToken() instanceof Text)
				((Text) canvas.getTopToken()).set_active(false);
			else if (canvas.getTopToken() instanceof Circle)
				((Circle) canvas.getTopToken()).set_active(false);
			else if (canvas.getTopToken() instanceof Arrow)
				((Arrow) canvas.getTopToken()).set_active(false);
		}
	}

	// Determine if shift key is being held down for breaking constraints
	public void keyPressed(KeyEvent ke) 
	{ 
		if (ke.getKeyCode() == KeyEvent.VK_SHIFT)
			shiftKeyDown = true;
	}

	// Determine if shift key is being held down for breaking constraints
	public void keyReleased(KeyEvent ke) 
	{ 
		if (ke.getKeyCode() == KeyEvent.VK_SHIFT)
			shiftKeyDown = false;
	}

	// Allows a subset of characters to be used to set Text symbol labels:  A-Z, a-z, 0-9, _
	public void keyTyped(KeyEvent ke)
	{
		if (canvas.getTopToken() instanceof Text) {
			Text t = (Text) canvas.getTopToken();
			if (ke.getKeyChar() == '\b') {
				if (!t.get_label().equals(""))
					t.set_label(t.get_label().substring(0, t.get_label().length() - 1));
				canvas.repaint();
			}
			else if ((ke.getKeyChar() >= 'a' && ke.getKeyChar() <= 'z') || (ke.getKeyChar() >= 'A' && ke.getKeyChar() <= 'Z') ||
				(ke.getKeyChar() >= '0' && ke.getKeyChar() <= '9') || (ke.getKeyChar() == '_')) {
				t.set_label(t.get_label() + ke.getKeyChar());
				canvas.repaint();
			}
		}
	}

	// Needed for the listener
	public void mouseEntered(MouseEvent me) { ; }
	public void mouseExited(MouseEvent me) { ; }
	public void mouseClicked(MouseEvent me) { ; }

	// Get the start location of a mouse movement
	public void mousePressed(MouseEvent me)
	{
		if (me.getButton() != MouseEvent.BUTTON1 || canvas.getNumTokens() == 0)
			return;
		startX = me.getX();
		startY = me.getY();
		mouseDrag = true;
	}

	// Get the end location of a mouse movement, and determine how to modify the symbol that is currently active 
	// (if the start location of the movement corresponds to a modifiable part of the symbol)
	public void mouseReleased(MouseEvent me)
	{
		if (me.getButton() != MouseEvent.BUTTON1 || !mouseDrag)
			return;

		// Don't get movement outside canvas
		if (me.getX() < 0 || me.getX() > IMAGE_WIDTH || me.getY() < 0 || me.getY() > IMAGE_HEIGHT)
			return;

		// See which type of symbol is being dealt with
		GrammarSymbol current = (GrammarSymbol) canvas.getTopToken();
		if (current instanceof Text) 
			modifyText((Text) current, me);
		else if (current instanceof Circle) 
			modifyCircle((Circle) current, me);
		else if (current instanceof Arrow) 
			modifyArrow((Arrow) current, me);
		mouseDrag = false;
	}

	// Modifies a Text object (if appropriate)
	private void modifyText(Text t, MouseEvent me)
	{
		// See if mid-point has been moved
		if (startX > ((int) t.get_mid_x() - MOUSE_TOLERANCE) && startX < ((int) t.get_mid_x() + MOUSE_TOLERANCE) &&
			startY > ((int) t.get_mid_y() - MOUSE_TOLERANCE) && startY < ((int) t.get_mid_y() + MOUSE_TOLERANCE)) {
			if (shiftKeyDown) {
				canvas.remove((DrawableToken) t);
				addToCanvasTop(new DrawableText(t.get_label(), me.getX(), me.getY(), false, true));
			}
			else {
				canvas.getInterpreter().setAutoConstraintSolving(false);
				t.set_mid_x(me.getX());
				canvas.getInterpreter().setAutoConstraintSolving(true);
				t.set_mid_y(me.getY());
			}
			canvas.repaint();
		}
	}

	// Modifies a Circle object (if appropriate)
	private void modifyCircle(Circle c, MouseEvent me)
	{
		// See if mid-point has been moved
		if (startX > ((int) c.get_mid_x() - MOUSE_TOLERANCE) && startX < ((int) c.get_mid_x() + MOUSE_TOLERANCE) &&
		    startY > ((int) c.get_mid_y() - MOUSE_TOLERANCE) && startY < ((int) c.get_mid_y() + MOUSE_TOLERANCE)) {
			if (shiftKeyDown) {
				canvas.remove((DrawableToken) c);
				addToCanvasTop(new DrawableCircle(c.get_radius(), me.getX(), me.getY(), true));
			}
			else {
				canvas.getInterpreter().setAutoConstraintSolving(false);
				c.set_mid_x(me.getX());
				canvas.getInterpreter().setAutoConstraintSolving(true);
				c.set_mid_y(me.getY());
			}
			canvas.repaint();
		}

		// Otherwise see if the radius must be altered
		else if (startX > ((int) (c.get_mid_x() + c.get_radius()) - MOUSE_TOLERANCE) && 
		    startX < ((int) (c.get_mid_x() + c.get_radius()) + MOUSE_TOLERANCE) &&
		    startY > ((int) c.get_mid_y() - MOUSE_TOLERANCE) && startY < ((int) c.get_mid_y() + MOUSE_TOLERANCE) &&
		    me.getX() > (int) (c.get_mid_x()) + MIN_RADIUS) {
			if (shiftKeyDown) {
				canvas.remove((DrawableToken) c);
				addToCanvasTop(new DrawableCircle(me.getX() - c.get_mid_x(), c.get_mid_x(), c.get_mid_y(), true));
			}
			else 
				c.set_radius(me.getX() - c.get_mid_x());
			canvas.repaint();
		}
	}

	// Modifies an Arrow object (if appropriate)
	private void modifyArrow(Arrow a, MouseEvent me)
	{
		// See if start-point has been moved
		if (startX > ((int) a.get_start_x() - MOUSE_TOLERANCE) && startX < ((int) a.get_start_x() + MOUSE_TOLERANCE) &&
		    startY > ((int) a.get_start_y() - MOUSE_TOLERANCE) && startY < ((int) a.get_start_y() + MOUSE_TOLERANCE) &&
		    !(me.getX() == (int) a.get_end_x() && me.getY() == (int) a.get_end_y())) {
			if (shiftKeyDown) {
				canvas.remove((DrawableToken) a);
				addToCanvasTop(new DrawableArrow(me.getX(), me.getY(), 
					a.get_mid_x(), a.get_mid_y(), a.get_end_x(), a.get_end_y(), 0.0, 0.0, true));
			}
			else {
				canvas.getInterpreter().setAutoConstraintSolving(false);
				a.set_start_x(me.getX());
				canvas.getInterpreter().setAutoConstraintSolving(true);
				a.set_start_y(me.getY());
			}
			canvas.repaint();
		}

		// See if end-point has been moved
		else if (startX > ((int) a.get_end_x() - MOUSE_TOLERANCE) && startX < ((int) a.get_end_x() + MOUSE_TOLERANCE) &&
		    startY > ((int) a.get_end_y() - MOUSE_TOLERANCE) && startY < ((int) a.get_end_y() + MOUSE_TOLERANCE) &&
		    !(me.getX() == (int) a.get_start_x() && me.getY() == (int) a.get_start_y())) {
			if (shiftKeyDown) {
				canvas.remove((DrawableToken) a);
				addToCanvasTop(new DrawableArrow(a.get_start_x(), a.get_start_y(), 
					a.get_mid_x(), a.get_mid_y(), me.getX(), me.getY(), 0.0, 0.0, true));
			}
			else {
				canvas.getInterpreter().setAutoConstraintSolving(false);
				a.set_end_x(me.getX());
				canvas.getInterpreter().setAutoConstraintSolving(true);
				a.set_end_y(me.getY());
			}
			canvas.repaint();
		}

		// See if mid-point has been moved
		else if (startX > ((int) a.get_mid_x() - MOUSE_TOLERANCE) && startX < ((int) a.get_mid_x() + MOUSE_TOLERANCE) &&
		    startY > ((int) a.get_mid_y() - MOUSE_TOLERANCE) && startY < ((int) a.get_mid_y() + MOUSE_TOLERANCE)) {
			if (shiftKeyDown) {
				canvas.remove((DrawableToken) a);
				addToCanvasTop(new DrawableArrow(a.get_start_x(), a.get_start_y(), 
					me.getX(), me.getY(), a.get_end_x(), a.get_end_y(), 0.0, 0.0, true));
			}
			else {
				canvas.getInterpreter().setAutoConstraintSolving(false);
				a.set_mid_x(me.getX());
				canvas.getInterpreter().setAutoConstraintSolving(true);
				a.set_mid_y(me.getY());
			}
			canvas.repaint();
		}
	}

	// Add a token to the top of the canvas
	private void addToCanvasTop(DrawableToken dt)
	{
		canvas.add(dt);
		canvas.moveToTop(dt);
	}

	// Main method which creates a new Editor instance
	public static void main(String argv[])
	{
		JFrame editor = new Editor();
	}
}

