/****************************************************************************

 This file is part of the QGLViewer library
 Copyright (C) 2002-2004  Gilles Debunne (Gilles.Debunne@imag.fr)
 Version 1.3.5 Release 8. Packaged on Monday December 22, 2003.

 http://www-imagis.imag.fr/Membres/Gilles.Debunne/CODE/QGLViewer

 libQGLViewer is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 libQGLViewer is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with libQGLViewer; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*****************************************************************************/

#ifndef QGLVIEWER_MANIPULATED_FRAME_H
#define QGLVIEWER_MANIPULATED_FRAME_H

#include <qstring.h>
#include <qtimer.h>
#include <qdatetime.h>

#include "spinningFrame.h"
#include "mouseGrabber.h"

class QGLViewer;

namespace qglviewer {

  /*! \brief A Frame that can be rotated and translated with the mouse.
     
  A ManipulatedFrame is a local coordinate system (a Frame) that can be moved in the 3D scene using
  the mouse. The ManipulatedFrame converts the mouse motion in a translation and orientation update.
  You can then use the ManipulatedFrame to move an object or a point in the scene.

  A ManipulatedFrame is attached to a QGLViewer using QGLViewer::setManipulatedFrame():
  \code
  init() { setManipulatedFrame( new ManipulatedFrame() ); }

  draw()
  {
    glPushMatrix();
    glMultMatrixd(manipulatedFrame()->matrix());
    // draw the manipulated object here
    glPopMatrix();
  }
  \endcode
  You have to press the QGLViewer::FRAME state key (default is \c Control) to move the
  QGLViewer::manipulatedFrame().
  
  The <a href="../examples/manipulatedFrame.html">manipulatedFrame</a> example details a complete
  application. See the <a href="../mouse.html">mouse page</a> for a description of the mouse button
  bindings.
  
  <h3>Inherited functionalities</h3>
  
  A ManipulatedFrame is an overloaded instance of a Frame. The powerful coordinate system
  transformation functions can hence be applied to a ManipulatedFrame.

  A ManipulatedFrame is also a SpinningFrame, and you can make the ManipulatedFrame spin is you
  release the rotation mouse button while moving the mouse fast enough (see spinningSensitivity()).

  Finally, a ManipulatedFrame is a MouseGrabber. If the mouse gets within a distance of 10 pixels
  from the projected position of the ManipulatedFrame, the ManipulatedFrame becomes the new
  QGLViewer::mouseGrabber(). It can then be manipulated directly, without selection or a specific
  state key. Note that QWidget::setMouseTracking() needs to be enabled in order to use this feature
  (see the MouseGrabber documentation). See the
  <a href="../examples/mouseGrabber.html">mouseGrabber</a> example an illustration.

  <h3>Advanced functionalities</h3>
  
  A QGLViewer can handle at most one ManipulatedFrame at a time. If you want to move several objects
  in the scene, you simply have to keep a list of the different ManipulatedFrames, and to activate
  the right one, for instance according to an object selection (see the
  <a href="../examples/luxo.html">luxo</a> example).
    
  When manipulated with the mouse, the isManipulated() function returns \c true, and it might be
  checked for a specific action. The ManipulatedFrame also emits a manipulated() signal when
  manipulated. This signal is automatically connected to all the viewer's updateGL() functions. See
  the manipulated() documentation for details.

  Combined with the object selection (see the <a href="../examples/select.html">select</a> example),
  the MouseGrabber properties and a dynamic update of the scene, the ManipulatedFrame introduces a
  great reactivity in your applications.
  \nosubgrouping */
  class QGLVIEWER_EXPORT ManipulatedFrame : public SpinningFrame, public MouseGrabber
  {
#ifndef DOXYGEN
    friend class Camera;
    friend class ::QGLViewer;
#endif

    Q_OBJECT

  public:
    ManipulatedFrame();
    /*! Virtual destructor. Empty. */
    virtual ~ManipulatedFrame() {};
    
    signals:
    /*! This signal is emitted whenever the frame is manipulated using the mouse. Connect this
    signal to any object that must be notified.

    Note that when a ManipulatedFrame is created, this signal is automatically connected to \e all
    the viewers' updateGL() functions (see QGLViewer::connectSignalToAllViewers()), so that the
    QGLViewers' displays are automatically updated.

    This signal connection technique is actually used for all the Frames that may be updated in your
    application. The KeyFrameInterpolator::interpolated() and the SpinningFrame::spinned() signals are
    also connected to all the viewers' updateGL() functions when they are created. In certain cases,
    this automatic display update may generate too many updates. Simply disconnect one of the
    signals to prevent this:
    \code
    QGLViewer::disconnectSignalToAllViewers(pointerToTheManipulatedFrame, SIGNAL(manipulated()));
    \endcode */
    void manipulated();

    /*! @name Intrinsic parameters */
    //@{    
  public slots:
    /*! Defines the influence of a mouse displacement on the frame rotation. Default value is 1.0. Defined with setRotationSensitivity(). */
    void setRotationSensitivity(const float s) { rotSensitivity_ = s; };
    /*! Defines the influence of a mouse displacement on the frame translation. Default value is
    1.0. Defined with setTranslationSensitivity().

    \note You should not have to modify this value as the frame translation is computed so that
    the frame intuitively exactly matches the mouse trajectory.

    \note When the manipulated frame is used to move a \e Camera, you may be worried when you
     are looking at a small region of your scene, which translate too fast on the screen when you move the camera.

    For a camera, it is the Camera::revolveAroundPoint() that exactly matches the mouse
    displacement. Instead of changing the translationSensitivity(), you may prefer to temporarily
    set the Camera::revolveAroundPoint() to a point of the small region of your scene (see the
    <a href="../mouse.html">mouse page</a> for the mouse binding). */
    void setTranslationSensitivity(const float s) { transSensitivity_ = s; };

    /*! Defines the spinningSensitivity(). */
    void setSpinningSensitivity(const float s) { spinningSensitivity_ = s; };
  public:
    /*! Returns the current rotationSensitivity() as defined by setRotationSensitivity(). Default value is 1.0 */
    float rotationSensitivity() const { return rotSensitivity_; };
    /*! Returns the current translationSensitivity() as defined by setTranslationSensitivity(). Default value is 1.0 */
    float translationSensitivity() const { return transSensitivity_; };
    /*! Returns the minimum mouse speed required to make the Frame spin when the mouse button is
    released. Mouse speed is expressed in pixels per milliseconds. Default value is 0.3 (300 pixels
    per second). Use setSpinningSensitivity() to tune this value. Setting to a higher value will
    make auto spinning more difficult (a value of 100.0 means forbid spinning in practice). */
    float spinningSensitivity() const { return spinningSensitivity_; };
    //@}
    
  public:
    /*! @name Current state */
    //@{
    /*! \p True when the ManipulatedFrame is being manipulated with the mouse. Can be used to
    manually simulate the QGLViewer::fastDraw() function. Used by QGLViewer. */
    bool isManipulated() const { return action_ != NO_ACTION; };
    //@}

  public:
    /*! @name XML representation */
    //@{    
    virtual QDomElement domElement(const QString& name, QDomDocument& doc) const;
    virtual void initFromDOMElement(const QDomElement& de);
    //@}    
    
    /*! @name Mouse event handlers */
    //@{
  protected:
    virtual void mousePressEvent(QMouseEvent* const, Camera* const);
    virtual void mouseReleaseEvent(QMouseEvent* const, Camera* const);
    virtual void mouseMoveEvent(QMouseEvent* const, const Camera* const);
    virtual void wheelEvent(QWheelEvent* const, const Camera* const camera);
    //@}

    /*! @name Mouse Grabber implementation */
    //@{
    void checkIfGrabsMouse(int x, int y, const Camera* const camera); 
    //@}

  protected:
    Quaternion quaternionFromDeformedBall(const int x, const int y, const float cx, const float cy, const Camera* const camera);

  public:
    // P o s s i b l e   a c t i o n s
    enum MouseMotion { NO_ACTION,
		       ROTATE, ZOOM, TRANSLATE,
		       MOVE_FORWARD, LOOK_AROUND, MOVE_BACKWARD,
		       SCREEN_ROTATE, SCREEN_ROTATE_BOTTOM,
		       SCREEN_TRANSLATE, ZOOM_ON_REGION };
    //#CONNECTION# QGLViewer::setMouseBinding
    
  protected:    
    MouseMotion action_;
    bool withConstraint_;

    virtual void startAction(MouseMotion ma, bool withConstraint=true);
    void computeMouseSpeed(const QMouseEvent* const e);    
    int mouseOriginalDirection(const QMouseEvent* const e);

    // Previous mouse position, used for incremental updates.
    int prevX_,  prevY_;
    // Mouse press position
    int pressX_, pressY_;  
    
  private:
    // Copy constructor and opertor= are declared private and undefined
    // Prevents everyone from trying to use them
    ManipulatedFrame(const ManipulatedFrame& mf);
    ManipulatedFrame& operator=(const ManipulatedFrame& mf);

    // Sensitivity
    float rotSensitivity_;
    float transSensitivity_;
    float spinningSensitivity_;
  
    // Mouse speed and spinning
    QTime last_move_time;
    float mouseSpeed_; 
    int delay_; 

    // Whether the SCREEN_TRANS direction (horizontal or vertical) is fixed or not.
    bool dirIsFixed_;

    // MouseGrabber
    bool keepsGrabbingMouse_;
  };

} // namespace qglviewer

#endif // QGLVIEWER_MANIPULATED_FRAME_H
