/****************************************************************************

 File:    cmdparam.h
 Created: by Aidan Lane, November 25, 2003
 Updated: by Aidan Lane, February 15, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#ifndef __CMDPARAM_H__
#define __CMDPARAM_H__


#include <qobject.h>
#include <qwidget.h>
#include <qstring.h>
#include <qptrlist.h>

#include "superptrlist.h"

#define MAX_WIDGET_WIDTH 100


/*! \brief Abstract base class that represents a commands parameter.
 *
 * It has a type name, formal argument name and default value properties.
 *
 * It is designed to be inherited by classes to implement the behaviour.
 * It ALSO handles a QT widget (e.g. QLineEdit) to edit the given parameter.
 *
 * <strong>Note:</strong> Glicth COMMUNICATES with command parameters ON A
 * STRING VALUE LEVEL.
 */


class CmdParam : public QObject {

	Q_OBJECT	// make QT's signal/slot system work

public:
	CmdParam( const char* formalVarName, const char* defaultValue,
		const char* exactTypeName = "unknown_type",
		SuperPtrList<CmdParam>* extVarElements = NULL )
		: myFormalVarName(formalVarName), myDefaultValue(defaultValue),
			myTypeName(exactTypeName), myExtVarElements(extVarElements) {}
	virtual ~CmdParam() {}

	// getters
	virtual const QString& formalVarName() const { return myFormalVarName; }
	virtual const QString& defaultValue() const { return myDefaultValue; }
	virtual const QString& typeName() const { return myTypeName; }
	virtual QPtrList<CmdParam>* extVarElements() { return myExtVarElements; }
	virtual int numExtVarElements() const
		{ return (myExtVarElements==NULL) ? 0 : myExtVarElements->count(); }

	// setters
	virtual void setFormalVarName( const QString& name ) { myFormalVarName = name; }
	virtual void setDefaultValue( const QString& value ) { myDefaultValue = value; }
	virtual void setTypeName( const QString& name ) { myTypeName = name; }
	virtual void setExtVarElements( QPtrList<CmdParam>* extVarElements )
		{ myExtVarElements = extVarElements; }

	/*!
	 * Create an editor widget for editing the parameter.
	 *
	 * <strong>Note:</strong> the member slot will be called when the parameter
	 * value is changed
	 *
	 * Example usage:
	 * \code
	 * QWidget* w = param->createWidget( editorParent, param->formalVarName(),
	 *                                   this, SLOT(widgetValueChanged()) );
	 * \endcode
	 *
	 * This method is designed to be reimplemented by derivitive classes.
	 */
	virtual QWidget* createWidget( QWidget* parent, const char* name,
					const QObject *slotOwner, const char* member ) = 0;
	//	-- note: widget in setWidgetValue refers to to the one created from the one above
	//  -- ... a sort of wrapper to support different types of widgets ...
	virtual void setWidgetValue( QWidget* widget, const QString& value ) = 0;
	virtual QString getWidgetValue( QWidget* widget ) = 0;

	/*!
	 * Get a hint on what width to use for the widget.
	 * For example, combo boxes need to be wider than line edit boxes.
	 *
	 * Hence, it is designed to be reimplemented by derivitive classes that
	 * wish to change this value.
	 */
	virtual int maxWidgetWidth() const { return MAX_WIDGET_WIDTH; }

	/*!
	 * This function is helpful with parameters that have a different string
	 * representation to that of their value, such as with enumerations.
	 *
	 * Hence, only re-implement it if the string representation is different.
	 *
	 * <strong>Note:</strong> EnumParam re-implements this
	 */
	virtual QString getStringRep( const QString& value ) const { return value; }

protected:
	QString myFormalVarName;
	QString myDefaultValue;
	QString myTypeName;
	QPtrList<CmdParam>* myExtVarElements;
};


#endif	// __CMDPARAM_H__
