#!/usr/bin/perl
################################################################################
#
# Title:    GENTREE - Generate a C++ command tree header file for GLITCH.
# Filename: gentree.pl
# Created:  by Aidan Lane, December 06, 2003
# Updated:  by Aidan Lane, February 15, 2004
# Desc:     This is designed to produce C++ command tree code for Glitch using
#           gl.h's and glu.h's prototypes and macros.
#
################################################################################

use POSIX 'strftime';


sub getMacroValue;


# Make sure that either 2 or 3 command line argument are specified
# - search paths, configuration file, and OPTIONALLY the output header filename
# If the output header filename is not specified, then output will be dumped /
# printed to standard out.
# Note: the last index for a array of length 2 is 1, and array of length 3 is 2
if ( $#ARGV != 1 and $#ARGV != 2 ) {
	die "Usage: $0 [SEARCH PATHS] [CONFIG FILE] [OUTPUT HEADER FILE]\n"
	  . "Note: To have multiple search paths, use the format: \"Path1:Path2:Path3\"\n"
	  . "Note: The output header filename is OPTIONAL, if it is not specified\n"
	  . "      then output will be dumped to standard-out (stdout).\n";
}
@searchPaths = split ":", $ARGV[0];
$cfg_filename = $ARGV[1];
$header_filename = $ARGV[2];

# Make sure that the config file is ACTUALLY a file, which can be read from
if ( not -r $cfg_filename ) {
	die "Error: Failed to open \"$cfg_filename\" for reading from. Exiting.\n";
}


# Evaluate $className, $groupTitle, @sourceFiles, $includeSourceFiles, %trees and %cmds,
# which are defined in the config file
if ( not do $cfg_filename ) {
	warn "Error: The configuration file \"$cfg_filename\" contains errors:\n";
	exec $^X, "$cfg_filename"; # execute the configuration script, as to diagnose it
	die "Exiting.\n"; # control should NOT reach here!
}


# Get the contents of the source files, as specified by the @sourceFiles array
# Concatenate the lines/records, as command declarations may span MULTIPLE lines
for ($i = 0; $i <= $#sourceFiles; $i++ )
{
	# Find source file with its full path by looking through the specified search paths
	$srcFilename = "";
	if ( -e "$sourceFiles[$i]" ) {
		$srcFilename = $sourceFiles[$i];
	}
	else
	{
		foreach ( @searchPaths )
		{
			if ( -e "$_/$sourceFiles[$i]" ) {
				$srcFilename = "$_/$sourceFiles[$i]";
				last;
			}
		}
	}
	if ( $srcFilename eq "" ) {
		die "Error: Failed to locate source file \"$sourceFiles[$i]\". Exiting.\n";
	}

	open (TMP, "$srcFilename")
		|| die "Error: Failed to open the source file \"$srcFilename\","
		       . " as specified by \"$cfg_filename\" for reading from. Exiting.\n";

	while ( <TMP> )
	{
		$input .= $_;
	};

	close (TMP) || die "Error: Failed to close the source file \"$srcFilename\". Exiting.\n";
}


# Process the command TREES
foreach $key (keys %trees)
{
	# The name 'root' is a special case, which equals 'mySubTree'.
	# Otherwise the root's name is "tree_$trees{$key}{'root'}"
	if ( $trees{$key}{'root'} eq 'root' ) {
		$root = 'mySubTree';
	}
	else {
		$root = "tree_" . $trees{$key}{'root'};
	}
		
	# Add the C++ subtree declaration and initialization
	$declareStr .= "        CmdTreeNode* tree_$key;\n";
	$initStr .= "        tree_$key = new CmdTreeNode( \"$key\", new QPtrList<CmdTreeNode> );\n";
	$initStr .= "        tree_$key->setAutoDelete( true );\n";
	$tressInitStr .= "        $root->append( tree_$key );\n";
}


# Search for and process OpenGL command declarations
$_ = $input;
while (1)
{	
		# Check for the next occurrence
#		if ( /[\s]*gl[\w]*[\s]*\([\w\s,\*\&]*\);/ )	# OpenGL specific
		if ( /[\s]*[\w]+[\s]*\([\w\s,\*\&]*\);/ )
		{
			# What follows this match is what we need to check next
			$theRest = $';

			# Get the command name and its arguments
			$& =~ /[\s]*\([\s]*/;
			$nameSide = $`;
			$paramsSide = $';

			# Get the command name by itself - last word in the string
#			$nameSide =~ /gl[\w]*$/; # OpenGL specific
			$nameSide =~ /[\w]+$/;
			$name = $&;

			# Don't proceed if the given command name is NOT one of what we want
			if ( not exists $cmds{$name} ) {
				$_ = $theRest;
				next;
			}

			# Get the params (together), without any trailing space characters
			$paramsSide =~ /[\s]*\);/;
			$paramsStr = $`;
			# Split the params up on commas
			@params = split( /[\s]*,[\s]*/, $paramsStr );

			# The name 'root' is a special case, which equals 'mySubTree'.
			# Otherwise the root's name is "tree_$trees{$key}{'root'}"
			if ( $cmds{$name}{'root'} eq 'root' ) {
				$root = 'mySubTree';
			}
			else {
				$root = "tree_" . $cmds{$name}{'root'} . '->subTree()';
			}

			# Get the array of default parameter values

			# START: Append the command to the initialization list (inserted into constructor)
			$cmdInitStr .= "        $root->append( new CmdTreeNode( \"$name\", NULL, cmd_$name, ";

			# check if the function has any parameters, or just 'void' or ''
			if ( $paramsStr eq "void" || $paramsStr eq "" ) {
				$cmdInitStr .= "NULL ) );\n";		# close it off
				$wrappedCmdsStr .= "    static void cmd_$name( const QStringList& ) {\n"
				                 . "        $name();\n"
				                 . "    }\n";

				$_ = $theRest;
				next;			# nothing else to do...
			}
			# START: Append the command to the wrapped command list
			$wrappedCmdDef = "    static void cmd_$name( const QStringList& p ) {\n";
			$wrappedCmdInit = "";
			$wrappedCmdCall = "        $name( ";


			$cmdInitStr .= "\n                            new SuperPtrList<CmdParam>( " . scalar(@params) . ",\n";

			# MIDDLE: Process the parameters
			$arrayOffsets = 0; # as arrays share the values from the string list too
			for ($i = 0; $i <= $#params; $i++ )
			{
				# Split each param up into a type and formal argument variable name
				# Note: the formal argument variable name is the last word in the string
				# Note: pointers to arrays ARE supported, BUT referances are NOT
				$params[$i] =~ /[\w]+[\s]*$/;	# match on the variable name, as it must be 1 word, unlike the type
				$typeSide = $`; # if a `*' or `&' is present, then it will end up at the end of this
				$nameSide = $&;

				# Get the name without any trailing space characters
				$typeSide =~ /[\s]*$/;
				$type = $`;

				# Get the default value for the parameter
				$value = $cmds{$name}{'defaults'}->[$i];

				# MIDDLE - PART A

				# Process the parameter appropriately, depending on its type
				if ( $type =~ /[\s]*\*$/ )
				{
					# TYPE: ARRAY
					$typeClassesUsed{'ArrayParam'} = 'arrayparam.h';
					# Get rid of the `*' and any preceeding spaces
					$coreType = $`;

					# Get rid of `const' if it is at the start of the core type
					if ( $coreType =~ /^const[\s]+/ ) {
						$coreType = $';
					}

					if ( $coreType eq 'GLint' or $coreType eq 'int' ) {
						$typeClassesUsed{'IntParam'} = 'intparam.h';
						$convFunc = 'toInt';
						$elementParamClass = 'IntParam';
					}
					elsif ( $coreType eq 'GLfloat' or $coreType eq 'float' ) {
						$typeClassesUsed{'FloatParam'} = 'floatparam.h';
						$convFunc = 'toFloat';
						$elementParamClass = 'FloatParam';
					}
					elsif ( $coreType eq 'GLdouble' or $coreType eq 'double' ) {
						$typeClassesUsed{'DoubleParam'} = 'doubleparam.h';
						$convFunc = 'toDouble';
						$elementParamClass = 'DoubleParam';
					}
					else {
						# NOTE: arrays of enums are NOT supported, as they are simply not needed
						die "Error: Unknown/unsupported parameter type \"$type\" in $name. Exiting.\n";
					}

					my $elements = $cmds{$name}{'arrays'}{$i}{'elements'};
					my $elementDefaults = $cmds{$name}{'arrays'}{$i}{'defaults'};

					$cmdInitStr .= '                                '
					             . "new ArrayParam( "
					             . "\"$nameSide\", \"$value\",\n"
								 . '                                    '
								 . "new SuperPtrList<CmdParam>( $elements,\n";
					$wrappedCmdInit .= "        $coreType $nameSide" . '[' . scalar($elements) . '] = { ';
					# process each of the array's elements
					for ( $j=0; $j < $elements; $j++ )
					{
						$cmdInitStr .= '                                        '
						             . "new $elementParamClass( \"$j\", "
									 . '"' . $elementDefaults->[$j]. '" )';
						$wrappedCmdInit .= 'p[' . scalar($j+$i+$arrayOffsets) . "].$convFunc()";

						if ( $j < $elements-1 ) {	# check that this is NOT the last element
							$cmdInitStr .= ",\n";
							$wrappedCmdInit .= ", ";
						}
					}
					$cmdInitStr .= " ),\n";	# close SuperList
					$cmdInitStr .= '                                    '
					             . "\"$coreType*\"";
					$cmdInitStr .= ' )';	# close ArrayParam
					$wrappedCmdInit .= " };\n";
					$wrappedCmdCall .= $nameSide;
					$arrayOffsets += $elements-1; # $elements - 1(we get 1 just for being a parameter)
				}
				elsif ( $type eq 'GLint' or  $type eq 'GLsizei' or $type eq 'GLshort' or $type eq 'int' )
				{
					# TYPE: INT
					$typeClassesUsed{'IntParam'} = 'intparam.h';
					$cmdInitStr .= "                                "
					             . "new IntParam( \"$nameSide\", \"$value\", \"$type\" )";
					$wrappedCmdCall .= 'p[' . ($i + $arrayOffsets) . '].toInt()';
				}
				elsif ( $type eq 'GLuint' or $type eq 'GLushort' )
				{
					# TYPE: GLuint
					$typeClassesUsed{'UnsignedIntParam'} = 'unsignedintparam.h';
					$cmdInitStr .= "                                "
					             . "new UnsignedIntParam( \"$nameSide\", \"$value\", \"$type\" )";
					$wrappedCmdCall .= 'p[' . ($i + $arrayOffsets) . '].toInt()';
				}
				elsif ( $type eq 'GLfloat' or $type eq 'float')
				{
					# TYPE: FLOAT
					$typeClassesUsed{'FloatParam'} = 'floatparam.h';
					$cmdInitStr .= "                                "
					             . "new FloatParam( \"$nameSide\", \"$value\", \"$type\" )";
					$wrappedCmdCall .= 'p[' . ($i + $arrayOffsets) . '].toFloat()';
				}
				elsif ( $type eq 'GLclampf' )
				{
					# TYPE: GLclampf - clamped float, value range = [0.0, 1.0]
					$typeClassesUsed{'ClampFloatParam'} = 'clampfloatparam.h';
					$cmdInitStr .= "                                "
					             . "new ClampFloatParam( \"$nameSide\", \"$value\", \"$type\" )";
					$wrappedCmdCall .= 'p[' . ($i + $arrayOffsets) . '].toFloat()';
				}
				elsif ( $type eq 'GLdouble' or $type eq 'double' )
				{
					# TYPE: DOUBLE
					$typeClassesUsed{'DoubleParam'} = 'doubleparam.h';
					$cmdInitStr .= "                                "
					             . "new DoubleParam( \"$nameSide\", \"$value\", \"$type\" )";
					$wrappedCmdCall .= 'p[' . ($i + $arrayOffsets) . '].toDouble()';
				}
				elsif ( $type eq 'GLclampd' )
				{
					# TYPE: GLclampd - clamped double, value range = [0.0, 1.0]
					$typeClassesUsed{'ClampDoubleParam'} = 'clampdoubleparam.h';
					$cmdInitStr .= "                                "
					             . "new ClampDoubleParam( \"$nameSide\", \"$value\", \"$type\" )";
					$wrappedCmdCall .= 'p[' . ($i + $arrayOffsets) . '].toDouble()';
				}
				elsif ( $type eq 'GLenum' or $type eq 'enum' or $type eq 'GLbitfield' )
				{
					# TYPE: ENUM
					$typeClassesUsed{'EnumParam'} = 'enumparam.h';
					my @enumStrs = @{$cmds{$name}{'enums'}{$i}};

					# Retrive the values of the specified enum macros
					my @enumVals;
					foreach ( @enumStrs ) {
#						push @enumVals, getMacroValue $_, $input;
						# MAKE SURE THAT THE MACRO EXISTS!!!
						getMacroValue $_, $input;
						push @enumVals, $_;
					}

					# Get the integer value of the default macro value
					# Convert it from hex (to int) if it is in hex form
					$macroValue = getMacroValue($value, $input);
					if ( $macroValue =~ /^0x[a-fA-F0-9]+/ )	{
						$macroValue = hex $macroValue;
					}

					$cmdInitStr .= "                                "
					             . "new EnumParam( \"$nameSide\", \""
					             . $macroValue . "\", \n"
					             . "                                    "
					             . "SuperValueList<char*>( "
					             . scalar(@enumStrs) . ",\n"
					             . "                                        \""
					             . join("\",\n                                        \"", @enumStrs) . "\" ),\n"
					             . "                                    SuperValueList<int>( "
					             . scalar(@enumVals) . ",\n"
					             . "                                        "
					             . join(",\n                                        ", @enumVals) . " ),\n"
					             . "                                    "
								 . "\"$type\" )";


					$wrappedCmdCall .= 'p[' . ($i + $arrayOffsets) . '].toInt()';
				}
				else
				{
					die "Error: Unknown/unsupported parameter type \"$type\" in $name. Exiting.\n";
				}

				# Check if this is NOT the last parameter
				if ( $i < $#params ) {
					# MIDDLE - PART B
					$cmdInitStr .= ", \n";
					$wrappedCmdCall .= ", ";
				}
			}

			# END
			$cmdInitStr .= " ) ) );\n";
			$wrappedCmdCall .= " );\n";
			$wrappedCmdsStr .= "\n"
			                 . $wrappedCmdDef
			                 . $wrappedCmdInit
			                 . $wrappedCmdCall
			                 . "    }";

			$_ = $theRest;
		}
		else
		{
			# No matches left, lets break out of this loop
			last;
		}
	};

	#$header_filename = lc $className . ".h";
	$defName = "__" . uc $className . "_H__";
	$time = POSIX::strftime( "%B %d, %Y @ %H:%M:%S", localtime );
	foreach (@extraIncludes) {
		$includeStr .= "#include $_\n";
	}
	$typeClassHeaders = "#include \""
	                    . join( "\"\n#include \"", values %typeClassesUsed )
	                    . "\"";
	$_ =
"/*******************************************************************************\
*
* Title:     $className
* Filename:  $header_filename
* Generated: $time
*        by: GENTREE - Generates C++ command tree header files for GLITCH.
*
* WARNING! All changes made in this file will be lost!
*
*******************************************************************************/

#ifndef $defName
#define $defName


#include <qgl.h>
#include <qptrlist.h>
#include <qstring.h>
#include <qstringlist.h>

#include \"cmdtreenode.h\"
#include \"cmdparam.h\"
#include \"superptrlist.h\"
#include \"supervaluelist.h\"

$typeClassHeaders

$includeStr

/*! \\brief Generated by gentree.pl. This inherits CmdTreeNode to provide the \"$groupTitle\" tree.
 *
 */

class $className : public CmdTreeNode
{
public:
    $className()
        : CmdTreeNode( \"$groupTitle\", new QPtrList<CmdTreeNode> )
    {
$initStr
        // Trees
$tressInitStr
        // Commands
$cmdInitStr    }

    // Note: In the unlikely (but serious) event that too few elements exist in
    //       the list of strings that is passed, the QStringList class will
    //       catch and handle the case as appropiate.
$wrappedCmdsStr

private:
$declareStr
};


#endif // $defName
";


if ( $header_filename eq "" )
{
	print $_;
}
else
{
	open (OUT, ">$header_filename")
	  || die "Error: could not open \"$header_filename\" for writing to. Exiting.";

	print OUT $_;

	close (OUT) || die "Error: could not close \"$header_filename\". Exiting.";
}


################################################################################
# Takes a macro name (string), search for it in some source (string) and then
# if found, it returns its value. Otherwise `die' will be called with a message.
################################################################################
sub getMacroValue
{
	($macroName, $source) = @_;

	# Search for the macro definition in the source text
	if ( $source =~ /#define[\s]+$macroName[\s]+[\S]+/ )
	{
		# Extract the value of the macro - the last value in the match,
		# which comes after the macro name + 1 space characters
		$& =~ /$macroName[\s]+/;
		return $';
	}

	die "Error: could not find \"$macroName\" in the specified header(s). Exiting.";
}
