/****************************************************************************

 File:    enumparam.h
 Created: by Aidan Lane, December 08, 2003
 Updated: by Aidan Lane, February 15, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#ifndef __ENUMPARAM_H__
#define __ENUMPARAM_H__


#include <qcombobox.h>
#include <qstringlist.h>
#include <qvaluelist.h>

#include "cmdparam.h"
#include "supervaluelist.h"


// Note: combo boxes will be created as wide AS REQUIRED BY THE CONTENTS
#define MAX_COMBO_BOX_WIDTH 300


/*! \brief Represents a parameter of type <em>GLenum</em>, as used by commmands.
 *
 * This class inherits CmdParam to provide a parameter that has a formal
 * variable name, a default value and an exact type name.<br>
 * Noticeably though, it has two extra properties (vs intparam, floatparam,
 * etc.) for the actual enumerations, which are <em>myEnumIndices</em> and
 * <em>myEnumStrings</em>.
 *
 * As an editor widget, it provides a simple integer QComboBox.
 */

class EnumParam : public CmdParam {

	Q_OBJECT	// make QT's signal/slot system work

public:
	EnumParam( const char* formalVarName, const char* defaultValue,
		const SuperValueList<char*>& enumStrs, const SuperValueList<int>& enumVals,
		const char* exactTypeName = "GLenum" )
		: CmdParam( formalVarName, defaultValue, exactTypeName ),
			myEnumVals(enumVals)
	{
		// Copy the enum char*'s into QStrings
		uint i;
		for ( i=0; i < enumStrs.count(); ++i ) {
			myEnumStrs.append( enumStrs[i] );
		}
	}

	// widgets
	virtual QWidget* createWidget( QWidget* parent, const char* name,
		const QObject *slotOwner, const char* member )
	{
		QComboBox *w = new QComboBox(parent,name);
		connect( w, SIGNAL( activated(int) ), slotOwner, member );
		// insert the strings
		uint i;
		for ( i=0; i < myEnumStrs.count(); ++i ) {
			w->insertItem( myEnumStrs[i] );
		}
		return (QWidget*)w;
	}
	// VALUE IN THIS CASE VALUE IS ACTUALLY THE MACRO VALUE (AN INTEGER), NOT THE MACRO STRING REPRESENTATION!
	virtual void setWidgetValue( QWidget* widget, const QString& value )
	{
		if ( widget == NULL ) {
			return;
		}

		int index = myEnumVals.findIndex( value.toInt() );

		// If the value was invalid, then select the first possible value... if possible!
		if ( index == -1 )
		{
			if ( myEnumStrs.count() > 0 ) {
				index = 0; // handle the error - set index to the first element
			} else {
				return; // FATAL ERROR - return without doing anything
			}
		}

		((QComboBox*)widget)->setCurrentItem( index );
	}
	virtual QString getWidgetValue( QWidget* widget ) {
		return (widget!=NULL) ? QString::number( myEnumVals[((QComboBox*)widget)->currentItem()] ) : QString::null;
	}

	// Re-implement this, as the string representation is different to that of
	// the internal value
	virtual QString getStringRep( const QString& value ) const
	{
		int index = myEnumVals.findIndex( value.toInt() );

		// If the value was invalid, then select the first possible value... if possible!
		if ( index == -1 )
		{
			if ( myEnumStrs.count() > 0 ) {
				return myEnumStrs[0]; // handle the error - subsitution
			} else {
				return "ERROR"; // FATAL ERROR - return with an error notifier
			}
		}

		return myEnumStrs[index];
	}

	// Note: combo boxes will be created as wide AS REQUIRED BY THE CONTENTS
	virtual int maxWidgetWidth() const {
		return MAX_COMBO_BOX_WIDTH;
	}

private:
	// After lots of testing (like with classes, structs, etc.), the following proved
	// to be the most efficient and flexile manner of storage
	QStringList     myEnumStrs;
	QValueList<int> myEnumVals;
};


#endif	// __ENUMPARAM_H__
