/****************************************************************************

 File:    glproject.cpp
 Created: by Aidan Lane, December 01, 2003
 Updated: by Aidan Lane, February 15, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include "glproject.h"

#include <qfile.h>
#include <qtextstream.h>
#include <qstringlist.h>


#define ML_NAME              "GLProjectML"	// mark-up Language name
#define DOC_VERSION_ATTRIB   "version"
#define TITLE_TAG            "title"
#define DESC_TAG             "description"
#define DOC_TAG              "documentation"
#define INIT_CMDS_TAG        "init_commands"
#define DISPLAY_CMDS_TAG     "display_commands"
#define CMD_TAG              "command"
#define CMD_NAME_ATTRIB      "name"
#define CMD_ENABLED_ATTRIB   "enabled"
#define CMD_PARAM_TAG        "param"
#define PARAM_POS_ATTRIB     "pos"
#define PARAM_VALUE_ATTRIB   "value"
#define PARAM_ELEMENT_TAG    "element"
#define ELEMENT_POS_ATTRIB   "pos"
#define ELEMENT_VALUE_ATTRIB "value"

#define CURRENT_DOC_VERSION  "1.0"

#define TRUE_STR  "true"
#define FALSE_STR "false"


/*!
 * Glitch Project Constructor.
 *
 * <strong>Note:</strong> Auto-delete is enabled here for both
 * <em>myInitCmds</em> and <em>myDisplayCmds</em>, so that when an pointer entry
 * is removed from either list, the object that it is pointing to is also deleted.
 */
GLProject::GLProject( const QString& title, const QString& filename )
	: myTitle(title), myFilename(filename)
{
	myInitCmds.setAutoDelete( true );
	myDisplayCmds.setAutoDelete( true );
}



/*!
 * Store a project to disk, with an XML representation.
 *
 * <strong>Note:</strong> This method WILL overwrite any existing file.
 * Please ensure that the user is happy with overwriting a file BEFORE calling
 * this method.
 */
bool GLProject::saveFile()
{
	QFile file( myFilename );

	// Attempt to open the file for writing to - else return false
	if ( !file.open(IO_WriteOnly) ) {
		return false;
	}

	QTextStream stream( &file );

	// Write the project XML string representation to disk
	stream << toString();

	file.close();

	return true;
}


/*!
 * Read a project from disk, with an XML representation.
 *
 * It requires the command tree to lookup the commands from and check for validity.
 *
 * On success, the project instance will contain the file's data and the
 * project's filename will also be set the one specified to the method.
 */
//	enum OpenStatus = { Success, DiskReadError, UnknownDocType,
//	                    UnknownDocVersion, DocParseError };
GLProject::OpenResult GLProject::openFile( const QString& filename,
                                           QPtrList<CmdTreeNode>* tree )
{
	QDomDocument doc( ML_NAME );
	QFile file( filename );

	// Attempt to open the file for writing to - else return false
	if ( !file.open(IO_ReadOnly) ) {
		return GLProject::DiskReadError;
	}

	// Attempt to parse the contents of the file into an XML structure
	QString errorMsg;
	if ( !doc.setContent( &file, &errorMsg ) ) {
		file.close();
		return GLProject::UnknownDocType; // not an XML document
	}
	
	file.close();

	// Clear the existing project data (e.g. tilte, text and command lists)
	myTitle = QString::null;
	myDescText = QString::null;
	myDocText = QString::null;
	myInitCmds.clear();
	myDisplayCmds.clear();

	// Get the document root element
	QDomElement root = doc.documentElement();

	// Ensure that the document is of the right type - ML_NAME (GLProjectML)
	if ( root.tagName() != ML_NAME ) {
		return GLProject::UnknownDocType; // incorrect XML document ML
	}

	// Ensure that we can read in this document version
	QDomAttr docVersion;
	if ( (docVersion = root.attributeNode(DOC_VERSION_ATTRIB)).isNull()
		|| docVersion.value() != CURRENT_DOC_VERSION ) {
		return GLProject::UnknownDocVersion;	// unknown document version
	}

	// Attempt to parse the init and display commands
	QDomNode n = root.firstChild();
	while( ! n.isNull() )
	{
		// Check if the node really is NOT an element - we are only checking for elements
		QDomElement e = n.toElement(); // try to convert the node to an element
		if( e.isNull() ) {
			continue;	// try the next node
		}

		// Parse the node elements
		if ( e.tagName() == TITLE_TAG ) {
			myTitle = getElementText( e );
		}
		else if ( e.tagName() == DESC_TAG ) {
			myDescText = getElementText( e );
		}
		else if ( e.tagName() == DOC_TAG ) {
			myDocText = getElementText( e );
		}
		else if ( e.tagName() == INIT_CMDS_TAG )
		{
			// Attempt parse the initialization commands
			if ( !parseCmds( e, myInitCmds, tree ) ) {
				return GLProject::DocParseError;	// document parse error
			}
		}
		else if ( e.tagName() == DISPLAY_CMDS_TAG )
		{
			// Attempt parse the display commands
			if ( !parseCmds( e, myDisplayCmds, tree ) ) {
				return GLProject::DocParseError;	// document parse error
			}
		}

		n = n.nextSibling();
	}

	// All seems good...
	myFilename = filename;
	return GLProject::Success;
}


/*!
 * Concatenate the strings from all of the level-0 DOM text nodes of a DOM element.
 * (there should only be one text element though)
 */
QString GLProject::getElementText( QDomElement e ) const
{
	QString text;
	QDomNode n = e.firstChild();

	while( !n.isNull() )
	{
		QDomText t = n.toText();
		if ( !t.isNull() )
		{
			text += t.nodeValue();
		}

		n = n.nextSibling();
	}

	return text;
}


/*!
 * Parse the commands from an XML DOM document into a list of command instances.
 */
bool GLProject::parseCmds( QDomElement &root, QPtrList<CmdInstance>& cmds,
	QPtrList<CmdTreeNode>* tree )
{
	// Parse commands
	QDomNode n = root.firstChild(); // command node
	while( !n.isNull() )
	{
		// Try to convert the node to an element - skip it if it is not
		QDomElement e;
		if ( (e = n.toElement()).isNull() ) {
			continue;
		}

		// Check if the element is a command, if so then parse it
		if ( e.tagName() == CMD_TAG )
		{
			bool instanceEnabled = true;

			// Try to get the command's name
			QDomAttr commandName;
			if ( (commandName = e.attributeNode(CMD_NAME_ATTRIB)).isNull() ) {
				return false;	// parse error - command name not specified
			}

			// Find the command in the command tree
			CmdTreeNode* cmd = CmdTreeNode::findNode( tree, commandName.value() );

			if ( cmd == NULL ) {
				return false;	// unknown command name
			}

			/* Try to get the command instance's enabled status.
			 * If a command does not have the enabled attribute, then thats OK,
			 * we will just leave it as enabled.
			 * NOTE: below, we are checking if the attribute IS _NOT_ null.
			 */
			QDomAttr enabledStatus;
			if ( ! (enabledStatus = e.attributeNode(CMD_ENABLED_ATTRIB)).isNull() )
			{
				if ( enabledStatus.value() == TRUE_STR ) {
					// instanceEnabled = true; *** NOT needed, is by default
				}
				else if ( enabledStatus.value() == FALSE_STR ) {
					instanceEnabled = false;
				}
				else {
					// The attribute's value was neither "true" or "false"
					return false;	// parse error - invalid boolean value
				}
			}

			QStringList paramValues;
			QMap<uint,QStringList> variableValues;
			QMap<uint,QString> paramValueMap;

			// Parse parameters, if there are any
			QPtrList<CmdParam>* params = cmd->params();
			if ( params != NULL )
			{
				QDomNode pn = e.firstChild(); // parameter node
				while( ! pn.isNull() )
				{
					// Try to convert the node to an element - skip it if it is not
					QDomElement pe;
					if ( (pe = pn.toElement()).isNull() ) {
						continue;
					}
						
					// Check if the element is a parameter, if so then parse it
					if ( pe.tagName() == CMD_PARAM_TAG )
					{
						QDomAttr paramPos = pe.attributeNode( PARAM_POS_ATTRIB );
						QDomAttr paramValue = pe.attributeNode( PARAM_VALUE_ATTRIB );

						if ( paramPos.isNull() || paramValue.isNull() ) {
							return false;	// parse error - parameter pos and/or value not specified
						}

						uint pos = paramPos.value().toInt();

						// Check if the parameter uses an external variable, if
						// so then parse the elements.
						if ( params->at(pos)->numExtVarElements() > 0 )
						{
							QMap<uint,QString> variableValueMap;
							QDomNode vn = pe.firstChild(); // variable node
							while ( ! vn.isNull() )
							{
								// Try to convert the node to an element - skip it if it is not
								QDomElement ve;
								if ( (ve = vn.toElement()).isNull() ) {
									continue;
								}

								QDomAttr v_attr_pos = ve.attributeNode( ELEMENT_POS_ATTRIB );
								QDomAttr v_attr_value = ve.attributeNode( ELEMENT_VALUE_ATTRIB );

								variableValueMap[v_attr_pos.value().toInt()]
									= v_attr_value.value();
								
								vn = vn.nextSibling();
							}
							// Create the variable value list
							int i;
							QStringList values;
							for ( i=0; i < params->at(pos)->numExtVarElements(); ++i ) {
								values.append( variableValueMap[i] );
							}
							variableValues[pos] = values;
						}
					
						// Get the parameter's value
						paramValueMap[pos] = paramValue.value();
					}
				
					pn = pn.nextSibling();
				}
				// Create the parameter list
				uint i;
				for ( i=0; i < params->count(); ++i )
				{
					// If the parameter is a variable, then append its element(s)
					// instead of its name
					if ( params->at(i)->numExtVarElements() > 0 ) {
						paramValues += variableValues[i];
					}
					else {
						paramValues.append( paramValueMap[i] );
					}
				}
			}

			// CREATE AND APPEND THE COMMAND
			CmdInstance* instance = new CmdInstance(cmd, paramValues);
			instance->setEnabled( instanceEnabled );
			// If the instance has any external variables, then initialize them
			QMap<uint,VarInstance*> externVarsMap = instance->externVarsMap();
			if ( externVarsMap.count() > 0 )
			{
				QMap<uint,VarInstance*>::Iterator it;
				for ( it = externVarsMap.begin(); it != externVarsMap.end(); ++it )
				{
					if ( (*it) != NULL )
					{
						(*it)->setVarName( paramValueMap[it.key()] );
						(*it)->setElementValues( variableValues[it.key()] );
					}
				}
			}
			cmds.append( instance );
		}

		n = n.nextSibling();
	}

	// All seems good...
	return true;
}


/*!
 * Get the XML representation of the project.
 *
 * This is used by the GLProject::saveFile() method.
 */
QString GLProject::toString()
{
	// Create the document and its root element node
	QDomDocument doc( ML_NAME );
	QDomElement root = doc.createElement( ML_NAME );

	root.setAttribute( DOC_VERSION_ATTRIB, CURRENT_DOC_VERSION );
	doc.appendChild( root );

	// Insert the project's title
	QDomElement titleElem = doc.createElement( TITLE_TAG );
	root.appendChild( titleElem );
	QDomText titleElemText = doc.createTextNode( myTitle );
	titleElem.appendChild( titleElemText );

	// Insert the project's description
	QDomElement descElem = doc.createElement( DESC_TAG );
	root.appendChild( descElem );
	QDomText descElemText = doc.createTextNode( myDescText );
	descElem.appendChild( descElemText );

	// Insert the project's documentation
	QDomElement docElem = doc.createElement( DOC_TAG );
	root.appendChild( docElem );
	QDomText docElemText = doc.createTextNode( myDocText );
	docElem.appendChild( docElemText );

	// Insert the initialization commands
	QDomElement initCmdsRoot = doc.createElement( INIT_CMDS_TAG );
	root.appendChild( initCmdsRoot );
	insertCmds( doc, initCmdsRoot, myInitCmds );

	// Insert the display commands
	QDomElement displayCmdsRoot = doc.createElement( DISPLAY_CMDS_TAG );
	root.appendChild( displayCmdsRoot );
	insertCmds( doc, displayCmdsRoot, myDisplayCmds );

	return doc.toString();
}



/*!
 * Used to insert a list of commands into a XML DOM document.
 */
void GLProject::insertCmds( QDomDocument &doc, QDomElement &root, QPtrList<CmdInstance>& cmds )
{
	// Insert the command nodes
	QPtrListIterator<CmdInstance> c_it( cmds );
	CmdInstance* instance;
	while ( (instance = c_it.current()) != NULL )
	{
		++c_it;

		// If the instance's command has not been set, then skip it
		if ( instance->cmd() == NULL ) {
			continue;
		}

		QMap<uint,VarInstance*> externVarsMap = instance->externVarsMap();
		QPtrList<CmdParam>* params = instance->cmd()->params();
		QStringList paramValues = instance->paramValues();

		// Insert a new command element with attribute CMD_NAME_ATTRIB set to the command's name
		QDomElement cmd_tag = doc.createElement( CMD_TAG );
		root.appendChild( cmd_tag );
		cmd_tag.setAttribute( CMD_NAME_ATTRIB, instance->cmd()->name() );

		/* Note: we do not need to explicity set each command instance that is
		 *       enabled to read enable="true", as the parser interprets a
		 *       command without the attribute as also being enabled.
		 */
		if ( ! instance->isEnabled() ) {
			cmd_tag.setAttribute( CMD_ENABLED_ATTRIB, FALSE_STR );
		}

		// If the instance does not have any parematers, then skip the rest of the loop
		if ( params == NULL ) {
			continue;
		}

		// Insert the parameters
		uint i, arrayOffsets = 0;
		for ( i=0; i < params->count(); ++i )
		{
			/* Note: The reason that I am outputting the parameter position
			 *       instead of the parameter names, is to provide a safety net.
			 *       For example, parameter names can be modfied or duplicated,
			 *       yet they will cause no problems for existing projects.
			 */
			QDomElement param_tag = doc.createElement( CMD_PARAM_TAG );
			cmd_tag.appendChild( param_tag );
			param_tag.setAttribute( PARAM_POS_ATTRIB, QString::number(i) );
			
			// Check if the parameter uses an external variable, such as an array

			if ( params->at(i) != NULL && params->at(i)->numExtVarElements() > 0 )
			{
				uint numExtVarElements = params->at(i)->numExtVarElements();
				
				if ( externVarsMap.contains(i) && externVarsMap[i] != NULL )
				{
					// Use the variable's name, not its element values
					param_tag.setAttribute( PARAM_VALUE_ATTRIB, externVarsMap[i]->varName() );

					// Append the element values
					uint j;
					QStringList elementValues = externVarsMap[i]->elementValues();
					for ( j=0; j < numExtVarElements; ++j )
					{
						QDomElement element_tag = doc.createElement( PARAM_ELEMENT_TAG );
						param_tag.appendChild( element_tag );
						element_tag.setAttribute( ELEMENT_POS_ATTRIB, QString::number(j) );
						element_tag.setAttribute( ELEMENT_VALUE_ATTRIB, elementValues[j] );
					}
				}
				
				arrayOffsets += numExtVarElements;	
			}
			else {
				param_tag.setAttribute( PARAM_VALUE_ATTRIB, paramValues[i+arrayOffsets] );
			}
		}
	}
}
