/****************************************************************************

 File:    cmdinstance.cpp
 Created: by Aidan Lane, December 02, 2003
 Updated: by Aidan Lane, February 26, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include "cmdinstance.h"

#include <qtooltip.h>
#include <qcombobox.h>

#include <typeinfo>

#include "cmdparam.h"


/*!
 * Default constructor, which gives the ability to initialize the parameters
 * to the command's defaults.
 */
CmdInstance::CmdInstance( CmdTreeNode* cmd, bool autoInitParams )
{
	sharedInit();
	setCmd( cmd, autoInitParams );
}


/*!
 * Constructor, which allow you to specify a list of parameters.
 *
 * <strong>Assumption:</strong> The <em>paramValues</em> contains the correct
 * number of values.
 */
CmdInstance::CmdInstance( CmdTreeNode* cmd, const QStringList& paramValues )
{
	sharedInit();
	myParamValues = paramValues;
	setCmd( cmd, false );
}


/*!
 * Destructor.
 *
 * Deallocate any external variables that the command instance used.
 */
CmdInstance::~CmdInstance()
{
	QMap<uint,VarInstance*>::Iterator c_it;
	for ( c_it = myExternVarsMap.begin(); c_it != myExternVarsMap.end(); ++c_it ) {
		delete (*c_it);
	}
}


/*!
 * This method is used by the constructors to perform tasks that are common to
 * all of them.
 */
void CmdInstance::sharedInit()
{
	myCmdSelector = NULL;

	cmdEditorBox = NULL;
	cmdEnabledCheckBox = NULL;
	cmdSpacerLabel = NULL;
	cmdSelectorButton = NULL;
	cmdOpenBraceLabel = NULL;
	cmdCloseBraceLabel = NULL;
	cmdParamsBox = NULL;

	myEnabledStatus = true;
}


/*!
 * Set the command tree command prototype node for the instance.
 *
 * <strong>Note:</strong> This will return straight away if <em>cmd</em> = 0.
 *
 * <strong>Note:</strong> This destroys existing external variables and creates
 * new ones for the given command. In the case where new external variables are
 * created, a varInstanceCreated will be emitted for each one.
 *
 */
void CmdInstance::setCmd( CmdTreeNode* cmd, bool autoInitParams )
{
	// DON'T GO ANY FURTHER IF THE COMMAND == NULL
	// THIS IS THE CUT OFF POINT
	if ( cmd == NULL ) {
		return;
	}

	myCmd = cmd;

	// Clear the current (if any) external variables
	QMap<uint,VarInstance*>::Iterator c_it;
	for ( c_it = myExternVarsMap.begin(); c_it != myExternVarsMap.end(); ++c_it ) {
		delete (*c_it);
	}
	myExternVarsMap.clear();

	// Create any required external variables, and store the pointers to them
	// along with their assosiate parameter index.
	QPtrList<CmdParam>* params = myCmd->params();

	if ( params != NULL )
	{
		// Process each of the command's parameters
		uint i;
		for ( i=0; i < params->count(); ++i )
		{
			// If the parameter requires external variables, then create them
			int elements;
			if ( (elements = params->at(i)->numExtVarElements()) > 0 )
			{
				// Create a new variable instance
				VarInstance* externVar = new VarInstance(
					params->at(i)->defaultValue(), params->at(i) );

				myExternVarsMap[i] = externVar;

				emit varInstanceCreated( this, externVar );

				// Make a signal/slot connection so that if one of the elements
				// values of the external variable is changed then it will call
				// paramValueChanged, so that this myParamValues is updated
				// and cmdParamChanged will be signaled.
				connect( externVar, SIGNAL( varElementChanged() ),
				                    SLOT( paramValueChanged() ) );
			}
		}
	}

	// Initialize the parameter values to their defaults if required
	// NOTE! This has to be done AFTER myExternVarsMap is created!
	if ( autoInitParams ) {
		initParamValues();
	}
}


/*!
 * Execute the command instance, using the current command and the current
 * parameters.
 *
 * <strong>Note:</strong> The command instance will NOT execute if it has been
 * disabled using the CmdInstance::setEnabled() method.
 */
void CmdInstance::exec() const
{
	if ( myCmd != NULL && myEnabledStatus == true ) {
		myCmd->exec( myParamValues );
	}
}


/*!
 * Get a string representation of the command instanance (including its params).
 */
QString CmdInstance::toString() const
{
	if ( !myCmd || !myCmd->funcPtr() ) {
		return "";	// no command specified, therefore we can't build a representation
	}

	QString text = myCmd->name() + "( ";

	// Build the string representation of the parameters IF THERE ARE PARAMETERS
	QStringList paramStrs;
	QPtrList<CmdParam>* params = myCmd->params();
	if ( params )
	{
		text += " ";

		CmdParam* p;
		uint i;

		// get the command instance's parameters
		for ( i=0; i < params->count(); ++i )
		{
			p = params->at(i);

			// Output an error if the formal param is not initialized, there are
			// insufficient actual parameters, or the actual param is empty
			if ( !p || myParamValues.count() <= i || myParamValues[i].isEmpty() ) {
				paramStrs += "[error]";
				break;
			}

			// Check if the parameter is an array
			if ( p->numExtVarElements() > 0 ) {
				// Get the variable's name
				paramStrs += myExternVarsMap[i]->varName();
			}
			else {
				// See cmdparam.h for details regrading why getStringRep is needed
				paramStrs += p->getStringRep( myParamValues[i] );
			}
		}
	}
	text += paramStrs.join( ", " );

	text += " );";

	return text;
}


/*!
 * Initialize the parameters to the defaults that are specified by the command.
 *
 * If myCmd is NULL, then this will NOT do anything (we don't know the formal
 * parameters).
 *
 * <strong>Note:</strong> This has to be done AFTER myExternVarsMap is created!
 * Otherwise if a parameter uses an array, their values will not be appended.
 */
void CmdInstance::initParamValues()
{
	if ( myCmd == NULL || myCmd->funcPtr() == NULL ) {
		return;	// no command specified, therefore we can't do anything
	}

	// Clear the current parameter string list, ready for new ones to be appended
	myParamValues.clear();

	// Build the string representation of the parameters, IF THERE ARE ANY parameters
	QPtrList<CmdParam>* params = myCmd->params();
	if ( params )
	{
		// Initialize the command instance's parameter values from the formal paremeter defaults
		uint i;
		for ( i=0; i < params->count(); ++i )
		{
			CmdParam* param = params->at(i);
			
			// Check if the parameter is an array
			if ( param->numExtVarElements() > 0 )
			{
				if ( myExternVarsMap.contains(i) && myExternVarsMap[i] != NULL ) {
					myParamValues += myExternVarsMap[i]->elementValues();
				}
			}
			else {
				myParamValues.append( param->defaultValue() );
			}
		}
	}
}


/*!
 * Create an instance of an editor for THIS command instance.
 *
 * <strong>Note:</strong> You need to ensure that you have made a
 * CmdInstance::setCmdSelector() call first before calling this method.
 * Without doing this, the method will simply return NULL without doing anything.
 * Though, this only needs to be done once.
 *
 * <strong>Note:</strong> Only one editor can exist at once, as the command
 * selector is SHARED.
 */
/* NOTE: cmdEditorBox is deleted first. QT will look after the clean-up of:
 * cmdSelectorButton, cmdEnabledCheckBox, cmdOpenBraceLabel, cmdCloseBraceLabel.
 * as well as all of the parameter editing widgets, as stored in cmdParamWidgets.
 * They ALL inherit from QObject
 */
QHBox* CmdInstance::createEditorWidget( QWidget* parent )
{
	if ( myCmdSelector == NULL ) {
		return NULL;
	}

	// Destroy any existing editor. (cmdEditorBox==NULL if not alive)
	delete cmdEditorBox;

	cmdEditorBox = new QHBox( parent, "cmdEditorBox" );
	cmdEnabledCheckBox = new QCheckBox( cmdEditorBox, "cmdEnabledCheckBox" );
	cmdSpacerLabel = new QLabel( " ", cmdEditorBox, "cmdSpacerLabel" ); // an ugly hack!
	cmdSelectorButton = new QPushButton( cmdEditorBox, "cmdSelectorButton" );
	QFont f = cmdSelectorButton->font();
	f.setBold( true );
	cmdSelectorButton->setFont( f );
	cmdOpenBraceLabel = new QLabel( "<b>(</b>", cmdEditorBox, "cmdOpenBraceLabel" );

	QToolTip::add( cmdEnabledCheckBox, "Enable / disable the command instance" );
	cmdEnabledCheckBox->setChecked( myEnabledStatus );

	if ( myCmd != NULL )
	{
		// update the button name and label...
		cmdSelectorButton->setText( myCmd->name() + "..." );
		QToolTip::add( cmdSelectorButton, myCmd->toString() );	// TOOLTIP: cmd's prototype
	}

	// Create the parameter widgets
	cmdParamsBox = new QHBox( cmdEditorBox, "cmdParamsBox" );


	// append the closing ");" onto the command row - fills the rest of the line
	// note that the any previous one would be destroyed with it QHBox container
	cmdCloseBraceLabel = new QLabel( "<b>);</b>", cmdParamsBox, "cmdCloseBraceLabel" );

	// NOTE! This has to be done AFTER cmdCloseBraceLabel is created, because if
	// there are no parameters, the abpve creation is needed, otherwise it will
	// be deleted and recreated in the correct position (as the last widget).
	createParamWidgets();

	connect( cmdEditorBox, SIGNAL( destroyed() ),
	                       SLOT( editorWidgetCleanup() ) );

	connect( cmdEnabledCheckBox, SIGNAL( toggled(bool) ),
	                             SLOT( cmdEnableCheckBoxToggled(bool) ) );

	connect( cmdSelectorButton, SIGNAL( clicked() ),
	                          SLOT( cmdSelectorClicked() ) );
							  
	if ( myCmdSelector != NULL )
	{
		// Disconnect ANYTHING currently connected to the command selector,
		// as to ensure that no other commmand instance is affected by it.
		myCmdSelector->disconnect();
		
		connect( myCmdSelector, SIGNAL( cmdSelected(CmdTreeNode*) ),
		                    SLOT( cmdSelectionChanged(CmdTreeNode*) ) );
	}

	// Update the enabled/disabled status of the editor widgets by simply
	// calling the cmdEnableCheckBoxToggled slot with the current  status
	cmdEnableCheckBoxToggled( myEnabledStatus );

	return cmdEditorBox;
}


/*!
 * Create the paremeter widgets and insert them into a container widget.
 * This is called by createEditorWidget and cmdSelectionChanged.
 * NOTE! This is designed for indernal use only.
 * NOTE! This will delete any existing parameter widgets first.
 * The widgets are inserted into cmdParamsBox.
 */
void CmdInstance::createParamWidgets()
{
	// Delete all existing parameter widgets ENTRIES.
	// Note that the actual elements are deleted with cmdEditorBox, so autoDelete
	// was NOT enabled, as the each element's contents is already gone.
	if ( cmdParamsBox != NULL ) // only delete the widgets if they are still alive
	{
		uint i;
		for ( i=0; i < cmdParamWidgets.count(); ++i ) {
			delete cmdParamWidgets.at(i);
		}
		delete cmdCloseBraceLabel; // delete this, as will add it again at the END
	}
	cmdParamWidgets.clear();

	// We can't proceed if no command has been selected!
	if ( myCmd == NULL ) {
		return;
	}
	
	// Create the parameter editor/input widgets
	QPtrList<CmdParam>* params = myCmd->params();
	if ( params != NULL )
	{
		uint i;

		// Create a suitable size policy that is to be used by the widgets
		QSizePolicy widgetSizePolicy( QSizePolicy::Expanding, 
		                              QSizePolicy::Expanding );
		widgetSizePolicy.setHorStretch( 3 );

		// WE NEED TO SKIP OVER THE INDIVIDUAL ELEMENTS OF ARRAYS
		// - so we use arrayOffsets for the displacement(s)...
		uint arrayOffsets = 0;
		for ( i=0; i < params->count(); ++i )
		{
			CmdParam* param = params->at( i );
			uint numExtVarElements = param->numExtVarElements();

			// WE NEED TO SKIP OVER THE INDIVIDUAL ELEMENTS OF ARRAYS
			// Below we subtract 1, as we get 1 for free, just for being a variable
			if ( numExtVarElements > 0 ) {
				arrayOffsets += numExtVarElements - 1;
			}

			QWidget* w = param->createWidget(
							cmdParamsBox, param->formalVarName(),
							this, SLOT( paramValueChanged() ) );

			// Apply the custom size policy to the widget
			w->setSizePolicy( widgetSizePolicy );
			
			if ( numExtVarElements > 0 ) {
				if ( myExternVarsMap.contains(i) && myExternVarsMap[i] != NULL ) {
					// Use the variable's name, not its element values
					param->setWidgetValue( w, QString(myExternVarsMap[i]->varName()) );
				}
			}
			else
			{	// Normal proceedure, just use the actual value directly
				param->setWidgetValue( w, myParamValues[i+arrayOffsets] );
			}
			cmdParamWidgets.append( w );
			QToolTip::add( w, param->typeName() + " " + param->formalVarName() );
			w->setMaximumWidth( param->maxWidgetWidth() );
			w->show();
		}
	}

	/* Create cmdCloseBraceLabel again, as we deleted it earlier, so that it
	 * could be positioned at the END of the widget.
	 * This helps to provide for a better layout, as it can work dirctly with
	 * the input widgets space/size.
	 */
	cmdCloseBraceLabel = new QLabel( "<b>);</b>", cmdParamsBox, "cmdCloseBraceLabel" );
	cmdCloseBraceLabel->show();

	// Apply a suitable custom size policy to cmdCloseBraceLabel
	QSizePolicy sp( QSizePolicy::Expanding, QSizePolicy::Expanding );
	sp.setHorStretch( 1 );
	cmdCloseBraceLabel->setSizePolicy( sp );
}



/*!
 * This is used to update the values of the parameter widgets in an existing
 * and active command instance editor (for this instance).
 *
 * It is particularly useful if you want to change the name of a variable
 * and then need the change to be reflected by the command instance editor.
 */
void CmdInstance::updateParamWidgets()
{
	QPtrList<CmdParam>* params = myCmd->params();

	if ( cmdEditorBox == NULL || cmdParamsBox == NULL || myCmd == NULL
		|| params == NULL || cmdParamWidgets.count() != params->count() ) {
		return;
	}

	// Block incomming signals for the following section, as WE will be changing
	// the values of widgets and we don't want to start a never-ending loop
	blockSignals( true );

	// Update the parameter editor/input widgets
	uint i;
	// WE NEED TO SKIP OVER THE INDIVIDUAL ELEMENTS OF ARRAYS
	// - so we use arrayOffsets for the displacement(s)...
	uint arrayOffsets = 0;
	for ( i=0; i < params->count(); ++i )
	{
		CmdParam* param = params->at( i );
		uint numExtVarElements = param->numExtVarElements();

		// WE NEED TO SKIP OVER THE INDIVIDUAL ELEMENTS OF ARRAYS
		// Below we subtract 1, as we get 1 for free, just for being a variable
		if ( numExtVarElements > 0 ) {
			arrayOffsets += numExtVarElements - 1;
		}

		// Check if the parameter is actually a variable, which contains one
		// or more elements (e.g. an array)
		if ( numExtVarElements > 0 ) {
			if ( myExternVarsMap.contains(i) && myExternVarsMap[i] != NULL ) {
				// Use the variable's name, not its element values
				param->setWidgetValue( cmdParamWidgets.at(i),
					myExternVarsMap[i]->varName() );
			}
		}
		else
		{	// Normal proceedure, just use the actual value directly
			param->setWidgetValue( cmdParamWidgets.at(i),
				myParamValues[i+arrayOffsets] );
		}
	}

	// We have finished changing values, so stop blocking signals
	blockSignals( false );
}


/*!
 * This method updates the command instance's enabled status. This includes
 * updating the private myEnabledStatus variable and  enabling/disabling the
 * editor widgets (bar the cmdEnabledCheckbox of course).
 * It also emits the cmdEnabledStatusChanged signal.
 */
void CmdInstance::cmdEnableCheckBoxToggled( bool on )
{
	myEnabledStatus = on;

	/* Only attempt to enable/disable the editor widgets if the core
	 * cmdEditorBox widget is alive. Hence, we assume is cmdEditorBox is alive,
	 * then so too are the editor widgets.
	 */
	if ( cmdEditorBox != NULL )
	{
		cmdSelectorButton->setEnabled( on );
		cmdOpenBraceLabel->setEnabled( on );
		cmdParamsBox->setEnabled( on );
		cmdCloseBraceLabel->setEnabled( on );
	}

	emit cmdEnabledStatusChanged( this );
}


/*!
 * Called when the command selection button is toggled.
 * This is connected to QPushButton's toggled( bool on ) signal.
 *
 * <strong>Precondition</strong>: <em>myCmdSelector</em> != 0.
 */
void CmdInstance::cmdSelectorClicked()
{
	if ( myCmdSelector == NULL ) {
		return;
	}

	// Set this instance's command as the current item in the command selector
	if ( myCmd != NULL ) {
		myCmdSelector->selectItem( myCmd );
	}

	// Popup the list and select the current command
	myCmdSelector->popup( cmdSelectorButton );
}


/*!
 * Change the command item for a given table command row.
 * It also emits the cmdInstanceChanged signal just before it returns.
 *
 * <strong>Precondition</strong>: <em>myCmdSelector</em> != 0 and <em>cmd</em> != 0.
 */
void CmdInstance::cmdSelectionChanged( CmdTreeNode* cmd )
{
	if ( myCmdSelector == NULL || cmd == NULL ) {
		return;
	}

	// If the same command was selected, then return, leaving the parameters
	// as they are. Note! We return WITHOUT emitting the cmdInstanceChanged signal.
	if ( myCmd != NULL
	     && cmd->name() == myCmd->name() ) {
		return;
	}

	// Change instance's the command and initialize its parameters
	setCmd( cmd );

	// Update the editor
	if ( cmdEditorBox != NULL )
	{
		if ( myCmd != NULL )
		{
			// update the button name and label...
			cmdSelectorButton->setText( myCmd->name() + "..." );
			QToolTip::add( cmdSelectorButton, myCmd->toString() );	// TOOLTIP: cmd's prototype
		}

		// Create the parameter widgets
		createParamWidgets();
	}

	emit cmdInstanceChanged( this );
}


/*!
 * Called when one of the parameter input widget's value is changed.
 * It also emits the cmdParamChanged signal just before it returns.
 */
void CmdInstance::paramValueChanged()
{
	QPtrList<CmdParam>* params = NULL;

	if ( myCmd == NULL ) {
		return; // error
	}

	if ( (params = myCmd->params()) == NULL ) {
		return;	// error
	}

	if ( cmdParamWidgets.count() != params->count() ) {
		return; // error OR simply the widgets are still being created and their
		        // text is being set, hence calling this prematurely.
				// either way, we don't want to continue
	}

	updateParamValues();

	emit cmdParamChanged( this );
}


/*!
 * Update that myParamValues string value list using values from both the
 * parameter widgets <em>and</em> the element widgets of arrays that being to
 * parameters.
 *
 * <strong>Note:</strong> An editor does NOT need to be active for this method
 * to be useful.
 * For example, this can be called when an external variable that belongs to one
 * of this instance's parameter's is updated.
 */
void CmdInstance::updateParamValues()
{
	// Extract values from ALL of the parameter widgets, as this method can
	// be called by ANY of these widgets on update.

	QPtrList<CmdParam>* params = NULL;

	if ( myCmd == NULL ) {
		return; // error
	}

	if ( (params = myCmd->params()) == NULL ) {
		return;	// error
	}

	// Get the new parameter values. Clear the existing ones first.
	QStringList oldParamValues = myParamValues;
	myParamValues.clear();
	uint i, arrayOffsets = 0;
	for ( i=0; i < params->count(); ++i )
	{
		CmdParam* param = params->at(i);
			
		// Check if the parameter is an array
		if ( param->numExtVarElements() > 0 )
		{
			// Get the variable's value(s)
			if ( myExternVarsMap.contains(i) && myExternVarsMap[i] != NULL ) {
				myParamValues += myExternVarsMap[i]->elementValues();
			}
			// Get value from the parameter input widget - update the variable's name
			if ( cmdParamWidgets.at(i) != NULL ) {
				myExternVarsMap[i]->setVarName(
					param->getWidgetValue(cmdParamWidgets.at(i)) );
			}

			arrayOffsets += param->numExtVarElements();
		}
		else {
			// Get value from the parameter input widget
			if ( cmdParamWidgets.at(i) != NULL ) {
				myParamValues.append( param->getWidgetValue(cmdParamWidgets.at(i)) );
			}
			else {
				myParamValues.append( oldParamValues[i+arrayOffsets] );
			}
		}
	}
}


/*!
 * This method cleans up this command instance when it editor is destroyed.
 *
 * This is called by cmdEditorBox's destroyed signal.
 */
void CmdInstance::editorWidgetCleanup()
{
	// Delete all existing parameter widgets ENTRIES.
	// Note that the actual elements are deleted with cmdEditorBox, so autoDelete
	// was NOT enabled, as the each element's contents is already gone.
	cmdParamWidgets.clear();

	if ( cmdEditorBox != NULL ) {
		cmdEditorBox->disconnect();     // disconnect editorWidgetCleanup slot
		cmdSelectorButton->disconnect();  // disconnect cmdSelectorClicked slot
	}

	if ( myCmdSelector != NULL ) {
		myCmdSelector->disconnect( this );  // disconnect cmdSelectionChanged slot
	}
	
	// Delete any existing editor, cmdEditorBox.
	// Note: this will automatically delete cmdEnabledCheckBox, cmdSelectorButton
	//       cmdOpenBraceLabel and CloseBraceLabel.
	cmdEditorBox = NULL;
	cmdParamsBox = NULL;
}


/*!
 * Set the background colour of the editor, that is if it is active.
 *
 * <strong>Note:</strong> There is a QtMac (Q_OS_MACX) specific section in this
 * method. It is used to set colours appropiately for specific widgets.
 */
void CmdInstance::setEditorBackgroundColour( const QColor& colour )
{
	if ( cmdEditorBox == NULL ) {
		return;
	}
	
	cmdEditorBox->setPaletteBackgroundColor( colour );
	cmdSpacerLabel->setPaletteBackgroundColor( colour );
	#if defined(Q_OS_MACX)
	cmdEnabledCheckBox->setPaletteBackgroundColor( colour );
	cmdSelectorButton->setPaletteBackgroundColor( colour );
	#endif // defined(Q_OS_MACX)
	cmdOpenBraceLabel->setPaletteBackgroundColor( colour );
	cmdCloseBraceLabel->setPaletteBackgroundColor( colour );

	// Make combo-boxes on the Mac look better, give them the background colour
	#if defined(Q_OS_MACX)
	if ( cmdParamsBox == NULL ) {
		return;
	}
	uint i;
	QWidget* w;
	for ( i=0; i < cmdParamWidgets.count(); ++i )
	{
		w = cmdParamWidgets.at(i);
		if ( w != NULL && typeid(*w) == typeid(QComboBox) ) {
			w->setPaletteBackgroundColor( colour );
		}
	}
	#endif // defined(Q_OS_MACX)
}
