/****************************************************************************

 File:    glproject.h
 Created: by Aidan Lane, December 01, 2003
 Updated: by Aidan Lane, February 15, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#ifndef __GLPROJECT_H__
#define __GLPROJECT_H__


#include <qdom.h>
#include <qstring.h>
#include <qptrlist.h>

#include "cmdtreenode.h"
#include "cmdinstance.h"

#define PROJECT_FILE_EXTENSION "glp" // GL Project


/*! \brief Encapsulates a Glitch project and provides file save and load facilities.
 *
 * A project consists primarily of a list of initialization command instances and
 * display command instances, both along with their own encapsulated data, including
 * parameter values and their variable instances (again, obviously with their data).
 *
 * A project also contains fields such as a title, a description and documentation.
 *
 * A project instance can easily be saved to disk via the GLProject::saveFile()
 * method (using the current filename). It can then be reloaded using the
 * GLProject::openFile() method.
 */

class GLProject
{
public:
	enum OpenResult { Success, DiskReadError, UnknownDocType,
	                  UnknownDocVersion, DocParseError };

	GLProject( const QString& title = QString::null,
		const QString& filename = QString::null );

	// getters
	QString title() const { return myTitle; }
	QString descText() const { return myDescText; }
	QString docText() const { return myDocText; }
	QString filename() const { return myFilename; }
	QPtrList<CmdInstance>* initCmds() { return &myInitCmds; }
	QPtrList<CmdInstance>* displayCmds() { return &myDisplayCmds; }

	// setters
	void setTitle( const QString& title ) { myTitle = title; }
	void setDescText( const QString& descText ) { myDescText = descText; }
	void setDocText( const QString& docText ) { myDocText = docText; }
	void setFilename( const QString& filename ) { myFilename = filename; }
	void setInitCmds( QPtrList<CmdInstance>& cmds ) { myInitCmds = cmds; }
	void setDisplayCmds( QPtrList<CmdInstance>& cmds ) { myDisplayCmds = cmds; }

	// file
	bool saveFile();
	OpenResult openFile( const QString& filename, QPtrList<CmdTreeNode>* tree );

	static QString fileExtension() { return PROJECT_FILE_EXTENSION; }

	// create a string representation of the project - used by saveFile
	// this should probably be private, but it is useful when it comes to debugging
	QString toString();

private:
	void insertCmds( QDomDocument &doc, QDomElement &root, QPtrList<CmdInstance>& cmds );
	QString getElementText( QDomElement e ) const;
	bool parseCmds( QDomElement &root, QPtrList<CmdInstance>& cmds, QPtrList<CmdTreeNode>* tree );

	QString myTitle;
	QString myDescText;		// description text (e.g. breif desc., author's name, etc...)
	QString myDocText;		// documention text
	QString myFilename;
	QPtrList<CmdInstance> myInitCmds;
	QPtrList<CmdInstance> myDisplayCmds;
};


#endif	// __GLPROJECT_H__
