/****************************************************************************

 File:    glscreenviewer.h
 Created: by Aidan Lane, January  26, 2004
 Updated: by Aidan Lane, February 17, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#ifndef __GLSCREENVIEWER_H__
#define __GLSCREENVIEWER_H__


#include "3rdparty/qglviewer_subset/qglviewer.h"
#include "glscene.h"


/*! \brief Inherits QGLViewer to provide a viewer that is more suited to simply displaying the contents of a scene directly.
 *
 * This class is used also to feed information to Glitch and any GLWorldViewer,
 * regarding the current matrix mode, and current model view and projection
 * matrix values. Where by current, we mean the state <em>immediately</em> after
 * the scene is drawn.
 *
 * <strong>Note:</strong> Display lists are NOT used to cache the initialization
 * and display commands, as they would enfore too many limitations. For example,
 * many OpenGL commands cannot be placed in display lists, such as those that
 * pass parameters by reference (as they may be out of scope). It would also
 * prevent the user from working with display lists themselves.
 */

class GLScreenViewer : public QGLViewer
{
public:
	GLScreenViewer( const GLScene* const s, QWidget* parent );
	GLScreenViewer( const GLScene* const s, const QGLFormat &format, QWidget* parent );

	/*! This tells us if the viewer will re-init itself on the next update. */
	bool willReinit() const { return reinitRequired; }
	/*! Tell the viewer that when it updates again, that it must re-init itself. */
	void setReinitRequired( bool required ) { reinitRequired = required; }

	GLenum matrixMode() const { return myMatrixMode; }	// matrix mode at the current step position
	// Note: I have purposely limited the matrix mode to either GL_MODELVIEW or GL_PROJECTION, as to simplify matters for the user
	const GLdouble* projectionMatrix() const { return myProjectionMatrix; } // projection matrix at the CURRENT STEP POSITION
	const GLdouble* modelviewMatrix() const { return myModelviewMatrix; } // modelview matrix at the CURRENT STEP POSITION

	// Auto glClear & glFLush and aspect 1:1 ratio are all enabled by default
	bool autoGLClearEnabled() const { return autoGLClear; }
	bool autoGLFlushEnabled() const { return autoGLFlush; }
	bool aspectRatioKept() const { return keepAspectRatio; }
	void setAutoGLClearEnabled( bool enable ) { autoGLClear = enable; }
	void setAutoGLFlushEnabled( bool enable ) { autoGLFlush = enable; }
	void setKeepAspectRatio( bool enable );

protected:
	void sharedInit();

	virtual void init();
	virtual void draw();
	/*!
	 * Ban QGLViewer from interfering with the pre-draw OpenGL state.
	 * We will look after things like glClear and the matrix modes and values.
	 */
	virtual void preDraw() {}
	/*!
	 * Ban QGLViewer from interfering with the post-draw OpenGL state.
	 * We don't need things like the axis, grid or visual hints to be drawn.
	 */
	virtual void postDraw() {}

protected slots:
	virtual void setGeometry( int x, int y, int w, int h );
	virtual void setGeometry( const QRect& r );

	/*! Reimplemented as {}, as to disable mouse interaction. */
	virtual void mousePressEvent( QMouseEvent* ) {}
	/*! Reimplemented as {}, as to disable mouse interaction. */
	virtual void mouseDoubleClickEvent( QMouseEvent* ) {}
	/*! Reimplemented as {}, as to disable mouse interaction. */
	virtual void mouseReleaseEvent( QMouseEvent* ) {}
	/*! Reimplemented as {}, as to disable mouse interaction. */
	virtual void mouseMoveEvent( QMouseEvent* ) {}
	/*! Reimplemented as {}, as to disable mouse interaction. */
	virtual void wheelEvent( QWheelEvent* ) {}

	/*! Reimplemented as {}, as to disable keyboard interaction.
	 * (Exception: mouse button modifiers) */
	virtual void keyPressEvent(QKeyEvent* ) {}

private:
	const GLScene* const myScene;

	GLenum myMatrixMode;
	GLdouble myProjectionMatrix[16];
	GLdouble myModelviewMatrix[16];

	bool reinitRequired;

	bool keepAspectRatio;
	bool autoGLClear;
	bool autoGLFlush;
};


#endif // __GLSCREENVIEWER_H__
