/****************************************************************************

 File:    glworldviewer.cpp
 Created: by Aidan Lane, January  26, 2004
 Updated: by Aidan Lane, February 22, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include "glworldviewer.h"

#include <limits.h>

using namespace qglviewer;


#define WORLD_SCENE_RADIUS 100.0	// a crazy large value for a scene


static const GLdouble identityMatrix[16] = {
	1.0, 0.0, 0.0, 0.0,
	0.0, 1.0, 0.0, 0.0,
	0.0, 0.0, 1.0, 0.0,
	0.0, 0.0, 0.0, 1.0 };


/*!
 * Constructor.
 *
 * See the class's detailed description for why we need a pointer to an instance
 * of GLScreenViewer.
 */
GLWorldViewer::GLWorldViewer( const GLScene* s, GLScreenViewer* screenViewer,
	QWidget* parent )
	: QGLViewer(parent), myScene(s), myScreenViewer(screenViewer)
{
	sharedInit();
}


/*!
 * Constructor.
 *
 * This constructor takes a QGLFormat parameter, so that special format options
 * such as double buffering can be enabled and disabled.
 *
 * See the class's detailed description for why we need a pointer to an instance
 * of GLScreenViewer.
 */

GLWorldViewer::GLWorldViewer( const GLScene* s, GLScreenViewer* screenViewer,
	const QGLFormat &format, QWidget* parent )
	: QGLViewer(format, parent), myScene(s), myScreenViewer(screenViewer)
{
	sharedInit();
}


/*!
 * This method is used by the constructors to perform tasks that are common to
 * all of them.
 */
void GLWorldViewer::sharedInit()
{
	reinitRequired = true;
	autoGLClear = true;
	autoGLFlush = true;

	myScreenViewersProjection = NULL;

	myBackgroundColour = Vec( 0.0, 0.0, 0.0 ); // black
	setBackgroundColor( myBackgroundColour  );

	extModifierKey = 0;

	resetCamera();
}


/*!
 * Stop the camera from spinning (if it is) and reposition it so that it is
 * located at (0.0, 0.0, 4.0), looking at (0.0, 0.0, 0.0) (i.e. the origin) and
 * has an up vector of (0.0, 1.0, 0.0).
 */
void GLWorldViewer::resetCamera()
{
	camera()->frame()->stopSpinning();
	camera()->setPosition( 0.0, 0.0, 4.0 );
	camera()->setUpVector( 0.0, 1.0, 0.0 );
	camera()->lookAt( sceneCenter() );
}


/*!
 * Initialize OpenGL for the context.
 *
 * This method is called by QGLViewer::initializeGL().
 *
 * The main reason that this has been implemented, is to undo the non-standard
 * modifications that QGLViewer::initializeGL() makes to the OpenGL initial state.
 */
void GLWorldViewer::init()
{
	glDisable( GL_LIGHTING );
	glDisable( GL_LIGHT0 );
	glDisable( GL_COLOR_MATERIAL );

	setSceneRadius( WORLD_SCENE_RADIUS );
}


/*!
 * Execute the OpenGL scene, though with some possible modifications, see below.
 *
 * <strong>Note:</strong> The scene is reinitialized EVERY time this is called,
 * as the camera has an effect on the initizliation commands and the most common
 * reason that this will be called is because the camera has been repositioned /
 * rotated, etc. This behaviour is unlike GLScreenViewer::draw()'s.
 *
 * <strong>Note:</strong> We prevent modifications to projection from being
 * executed by the scene, as we will be DRAWING the projection in the form of
 * its frustum instead in GLWorldViewer::postDraw().
 *
 * <strong>Note:</strong> OpenGL viewports are <em>NOTE</em> supported by this
 * viewer, as they would not make sense in this context. Hence glViewport
 * commands will be ignored.
 *
 * <strong>Note:</strong> This method will perfrom:
 * \code glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_ACCUM_BUFFER_BIT | GL_STENCIL_BUFFER_BIT ) \endcode
 * if and only if <em>autoGLClear</em> == true.
 *
 */
void GLWorldViewer::draw()
{
	glPushAttrib( GL_ALL_ATTRIB_BITS );
	glMatrixMode( GL_PROJECTION );
	glPushMatrix();
	glMatrixMode( GL_MODELVIEW );
	glPushMatrix();

		// We NEED depth testing throughout the world viewer
		glEnable( GL_DEPTH_TEST );

		/* Execute the initization commands part of the scene, where true =
		 * disable modifications to the projection.
		 * Hence, just output the scene's contents, don't execute its projection,
		 * as we will be drawing the projection in the form of its frustum later.
		 * Note: We execute the initialization commands BEFORE the auto glClear.
		 */
		myScene->execInitCmds( true, true, true );

		// Execute the automatic glClear call if required
		if ( autoGLClear ) {
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT
				| GL_ACCUM_BUFFER_BIT | GL_STENCIL_BUFFER_BIT );
		}

		/* Restore the previous post-display current matrix mode.
		 * This is needed, as to emulate the real conditions, where the
		 * initilzation function is executed only once, but then the display
		 * function will be called over-and-over, with the OpenGL state staying
		 * the same from the end of the display function to the start of it
		 * again, for the next iteration.
		 * Hence, due to the way the matrices are pushed & popped, the current
		 * matrix mode is affected, so we need to restore it. (Phew!)
		 */
		glMatrixMode( myScreenViewer->matrixMode() );

		/* Execute the display commands part of the scene, where true = disable
		 * modifications to the projection.
		 * Hence, just output the scene's contents, don't execute its projection,
		 * as we will be drawing the projection in the form of its frustum later.
		 */
		myScene->execDisplayCmds( true, true, true );


	glMatrixMode( GL_MODELVIEW );
	glPopMatrix();
	glMatrixMode( GL_PROJECTION );
	glPopMatrix();
	glMatrixMode( GL_MODELVIEW );

	glPopAttrib();
}


/*!
 * Prep the OpenGL context for drawing to.
 *
 * This method will make a call for its GLScreenViewer instance counterpart to
 * update itself (via an GLScreenViewer::updateGL() call).<br>
 * Hence if you want both this viewer and the screen space viewer to be updated,
 * you need ONLY call updateGL() on THIS viewer.<br>
 * (The updateGL() call on the screen view is needed to keep matrix synchronization)<br>
 * Getting the matrix values like this seems to be the most efficient, as the scene's
 * OpenGL commands need to be executed as to get them, so we get them from the screen
 * viewer, which we can rely on being a direct / pure representation of the scene.
 *
 * <strong>Note:</strong> This will delete any existing display lists.
 */
void GLWorldViewer::preDraw()
{
	// Update the screen viewer first, so that we can get the current projection
	// matrix values.
	myScreenViewer->updateGL();
	myScreenViewersProjection = myScreenViewer->projectionMatrix();


	// NOTE: This is VERY important, as it brings us back to OUR OpenGL context
	makeCurrent();


	// Delete any existing display lists
	glDeleteLists( 0, INT_MAX );

	// Ensure that the viewport is the same size as the widget
	glViewport( 0, 0, width(), height() );

	// Ensure that the correct background clear colour is used.
	glClearColor( myBackgroundColour[0],
		myBackgroundColour[1],
		myBackgroundColour[2], 1.0 );

	// GL_PROJECTION matrix
	camera()->loadProjectionMatrix();
	// GL_MODELVIEW matrix
	camera()->loadModelViewMatrix();

	glDisable( GL_LIGHTING );
	glDisable( GL_LIGHT0 );
	glDisable( GL_COLOR_MATERIAL );

}


/*!
 * Draw post-scene objects and visual hints, such as the axis, Z=0 grid,
 * projection frustum and possibly the contents of the Z-buffer.
 */
void GLWorldViewer::postDraw()
{
	GLdouble inverseP[16];

	// We NEED the screen viewer's post-scene projection matrix
	if ( myScreenViewersProjection == NULL ) {
		return;
	}

	// (Re-)create the grid and axis display lists
	// NOTE: We only do this now, AFTER the scene is drawn, as the display list
	// creation methods use the glGenList function to generate a unique display
	// list identifier.
	AxisDL_ = 0;
	GridDL_ = 0;
	createDisplayLists();

	glPushAttrib( GL_ALL_ATTRIB_BITS );
	glMatrixMode( GL_PROJECTION );
	glPushMatrix();
	glMatrixMode( GL_MODELVIEW );
	glPushMatrix();

		invert( myScreenViewersProjection, inverseP );

		// We NEED depth testing throughout the world viewer
  		glEnable( GL_DEPTH_TEST );

		if ( axisIsDrawn() || gridIsDrawn() )
		{
			glEnable( GL_LIGHTING );
			glEnable( GL_LIGHT0 );
			glEnable( GL_COLOR_MATERIAL );

			Vec c = foregroundColor();
			glColor3f( c[0], c[1], c[2] );

			if ( axisIsDrawn() ) drawAxis();
			if ( gridIsDrawn() ) drawGrid();

			glDisable( GL_LIGHTING );
			glDisable( GL_LIGHT0 );
			glDisable( GL_COLOR_MATERIAL );
		}


		/* PREPRARE the inverse matrix for the drawFrustum method.
		 * If the inverse projection matrix is the same as the identity matrix,
		 * then no projection transformations have been performed, hence, we need
		 * to negate the element at position [2,2] (0 indexed) = 10 to componsate.
		 */
		int i;
		bool diff = false;
		for ( i=0; i < 16; ++i ) {
			if ( inverseP[i] != identityMatrix[i] ) {
				diff = true;
				break;
			}
		}
		if ( ! diff ) { // inverse projection matrix == identity matrix
			inverseP[10] = inverseP[10] * -1;
		}


		// Draw the viewing frustum if required
		if ( frustumEnabled ) drawFrustum( inverseP );

		// Draw the visual hints LAST, ontop of everything
		drawVisualHints();

		// Display the Z-buffer if required
		if ( zBufferIsDisplayed() ) displayZBuffer();

	glPopAttrib();
	glMatrixMode( GL_MODELVIEW );
	glPopMatrix();
	glMatrixMode( GL_PROJECTION );
	glPopMatrix();
	glMatrixMode( GL_MODELVIEW );

	// Execute the automatic glFlush call if required
	if ( autoGLFlush ) {
		glFlush();
	}
}


/*!
 * Draw the viewing frustum for a given projection matrix.
 *
 * <strong>Note:</strong> This method takes the <em>INVERSE</em> of the
 * projection matrix as its parameter.
 */
void GLWorldViewer::drawFrustum( GLdouble* inverseProjectionMatrix )
{
	glPushAttrib( GL_ALL_ATTRIB_BITS );
	glMatrixMode( GL_PROJECTION );
	glPushMatrix();
	glMatrixMode( GL_MODELVIEW );
	glPushMatrix();

		glMatrixMode( GL_MODELVIEW );
		glMultMatrixd( inverseProjectionMatrix );

		glDisable( GL_LIGHTING );

	    /* draw the viewing frustum */
		// FAR PLANE
		glColor3f(0.2, 0.2, 0.2);
	    glBegin(GL_QUADS);
	    glVertex3i(1, 1, 1);
    	glVertex3i(-1, 1, 1);
	    glVertex3i(-1, -1, 1);
    	glVertex3i(1, -1, 1);
	    glEnd();

		// LINES
    	glColor3ub(128, 196, 128);
	    glBegin(GL_LINES);
    	glVertex3i(1, 1, -1);
	    glVertex3i(1, 1, 1);
    	glVertex3i(-1, 1, -1);
	    glVertex3i(-1, 1, 1);
	    glVertex3i(-1, -1, -1);
    	glVertex3i(-1, -1, 1);
	    glVertex3i(1, -1, -1);
    	glVertex3i(1, -1, 1);
	    glEnd();
    
		// NEAR PLANE
	    glEnable(GL_BLEND);
    	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	    glColor4f(0.2, 0.2, 0.4, 0.5);
    	glBegin(GL_QUADS);
	    glVertex3i(1, 1, -1);
    	glVertex3i(-1, 1, -1);
	    glVertex3i(-1, -1, -1);
    	glVertex3i(1, -1, -1);
	    glEnd();
    	glDisable(GL_BLEND);

	glPopAttrib();
	glMatrixMode( GL_MODELVIEW );
	glPopMatrix();
	glMatrixMode( GL_PROJECTION );
	glPopMatrix();
	glMatrixMode( GL_MODELVIEW );
}


/*!
 * Used to extend Qt's mouse button modifier keys to include the "1", "2" and
 * "3" number keys.
 * This method will handle a key press event and filter it to check if one of
 * these extended modifier keys has been pressed. If one has been, then the
 * current extended modifier key variable will be set to the Qt key code.
 *
 * This reimplementation also disables QGLViewer's key bindings.
 */
void GLWorldViewer::keyPressEvent( QKeyEvent* e )
{
	// Note how we filter for just what we care about, other key presses can come
	// and go without interupting us (maybe interupt is the wrong word to use?)
	switch ( e->key() )
	{
		case Qt::Key_1:
			extModifierKey = Qt::Key_1;
			break;

		case Qt::Key_2:
			extModifierKey = Qt::Key_2;
			break;

		case Qt::Key_3:
			extModifierKey = Qt::Key_3;
			break;

		default:
			break;
	}
}


/*!
 * Used to extend Qt's mouse button modifier keys to include "1", "2" and
 * "3" number keys.
 * This method will handle a key release event and filter it to check if one of
 * these extended modifier keys has been released. If one has been, then the
 * current extended modifier key variable will be set to = 0 (no key).
 *
 * This reimplementation also disables QGLViewer's key bindings.
 */
void GLWorldViewer::keyReleaseEvent( QKeyEvent* e )
{
	// If it makes no difference to us, then just ignore it!
	if ( extModifierKey != e->key() ) {
		return;
	}

	switch ( e->key() )
	{
		case Qt::Key_1:
			extModifierKey = 0;
			break;

		case Qt::Key_2:
			extModifierKey = 0;
			break;

		case Qt::Key_3:
			extModifierKey = 0;
			break;

		default:
			break;
	}
}


/*! Re-implemented, as to use the mouse event translator.
 * See GLWorldViewer::mouseEventTranslator() for details. */
void GLWorldViewer::mousePressEvent( QMouseEvent* e )
{
	mouseEventTranslator( e, QEvent::MouseButtonPress );
}


/*! Re-implemented, as to use the mouse event translator.
 * See GLWorldViewer::mouseEventTranslator() for details. */
void GLWorldViewer::mouseReleaseEvent( QMouseEvent* e )
{
	mouseEventTranslator( e, QEvent::MouseButtonRelease );
}


/*! Re-implemented, as to use the mouse event translator.
 * See GLWorldViewer::mouseEventTranslator() for details. */
void GLWorldViewer::mouseDoubleClickEvent( QMouseEvent* e )
{
	mouseEventTranslator( e, QEvent::MouseButtonDblClick );
}


/*! Re-implemented, as to use the mouse event translator.
 * See GLWorldViewer::mouseEventTranslator() for details. */
void GLWorldViewer::mouseMoveEvent( QMouseEvent* e )
{
	mouseEventTranslator( e, QEvent::MouseMove );
}


/*!
 * This is used to translate mouse events, as to support all of the control
 * features of a 3-button mouse by adding the "1", "2" and "3" number keys as
 * modifiers to the left mouse button.
 *
 * Translated events:
 * <ul>
 * <li>Key 1 + Left Mouse Button => Middle Mouse Button (reposition camera's z position)</li>
 * <li>Key 2 + Left Mouse Button => Right Mouse Button (reposition camera's x & y position)</li>
 * <li>Key 3 + Left Mouse Button => Left + Middle Mouse Button (rotate camera on z-axis)</li>
 * </ul>
 *
 * <strong>Note:</strong> The ordering of the modifiers, where key 1 was chosen
 * for the middle button, followed by key 2 for the right mouse button, was
 * implemented, as that is the same ordering of the buttons on a mouse (right-hand setup).
 */
void GLWorldViewer::mouseEventTranslator( QMouseEvent* e,
	QEvent::Type mouseEventType )
{
	QMouseEvent* transE = NULL;

	// Modifier keys are only avaliable here for the LEFT mouse button
	// If you have more than one button, then use the actual physical
	// buttons to perform the different actions, instead of using these.
	if ( (e->button() & Qt::MouseButtonMask) == Qt::LeftButton )
	{
		switch( extModifierKey )
		{
			case Qt::Key_1:
				// translate to MIDDLE mouse button - zoom in/out
				transE = new QMouseEvent( mouseEventType, e->pos(),
					Qt::MidButton, Qt::NoButton );
				break;

			case Qt::Key_2:
				// translate to RIGHT mouse button - move camera
				transE = new QMouseEvent( mouseEventType, e->pos(),
					Qt::RightButton, Qt::NoButton );
				break;

			case Qt::Key_3:
				// translate to LEFT+MIDDLE mouse buttons - rotate on z-axis
				transE = new QMouseEvent( mouseEventType, e->pos(),
					Qt::LeftButton|Qt::MidButton, Qt::NoButton );
				break;
		}
	}

	// Choose the appropiate mouse event to emit
	if ( transE != NULL ) {
		e = transE;
	}
	
	// Pass the possibly translated event to the next level... QGLViewer
	switch ( mouseEventType )
	{
		case QEvent::MouseButtonPress:
			QGLViewer::mousePressEvent( e );
			break;

		case QEvent::MouseButtonRelease:
			QGLViewer::mouseReleaseEvent( e );
			break;

		case QEvent::MouseButtonDblClick:
			QGLViewer::mouseDoubleClickEvent( e );
			break;

		case QEvent::MouseMove:
			QGLViewer::mouseMoveEvent( e );
			break;

		default:
			break;
	}
}


/*!
 * This method is used to get the inverse of a 4x4 matrix.
 *
 * It actually came from from Nate Robins' OpenGL Tutors.<br>
 * I contacted him via email (nate@pobox.com) on Sunday 25, January 2004
 * and got the OK from him to use it before I went ahead and used it.
 *
 * Beginning of the source code file projection.c:
 * \code
 *     projection.c
 *     Nate Robins, 1997
 *
 *     Tool for teaching about OpenGL projections.
 * \endcode
 */
GLboolean GLWorldViewer::invert( const GLdouble src[16], GLdouble inverse[16] )
{
    double t;
    int i, j, k, m, swap;
    GLdouble tmp[4][4];
    
	// Modification by Aidan Lane, February 01, 2004
	// identity(inverse);
	// Copy the identity matrix into the `inverse' matrix
	for ( m=0; m < 16; ++m ) {
		inverse[m] = identityMatrix[m];
	}
    
    for (i = 0; i < 4; i++) {
        for (j = 0; j < 4; j++) {
            tmp[i][j] = src[i*4+j];
        }
    }
    
    for (i = 0; i < 4; i++) {
        /* look for largest element in column. */
        swap = i;
        for (j = i + 1; j < 4; j++) {
            if (fabs(tmp[j][i]) > fabs(tmp[i][i])) {
                swap = j;
            }
        }
        
        if (swap != i) {
            /* swap rows. */
            for (k = 0; k < 4; k++) {
                t = tmp[i][k];
                tmp[i][k] = tmp[swap][k];
                tmp[swap][k] = t;
                
                t = inverse[i*4+k];
                inverse[i*4+k] = inverse[swap*4+k];
                inverse[swap*4+k] = t;
            }
        }
        
        if (tmp[i][i] == 0) {
        /* no non-zero pivot.  the matrix is singular, which
           shouldn't happen.  This means the user gave us a bad
            matrix. */
            return GL_FALSE;
        }
        
        t = tmp[i][i];
        for (k = 0; k < 4; k++) {
            tmp[i][k] /= t;
            inverse[i*4+k] /= t;
        }
        for (j = 0; j < 4; j++) {
            if (j != i) {
                t = tmp[j][i];
                for (k = 0; k < 4; k++) {
                    tmp[j][k] -= tmp[i][k]*t;
                    inverse[j*4+k] -= inverse[i*4+k]*t;
                }
            }
        }
    }
    return GL_TRUE;
}
