/****************************************************************************

 File:    glworldviewer.h
 Created: by Aidan Lane, January  26, 2004
 Updated: by Aidan Lane, February 26, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#ifndef __GLWORLDVIEWER_H__
#define __GLWORLDVIEWER_H__


#include "3rdparty/qglviewer_subset/qglviewer.h"
#include "glscene.h"
#include "glscreenviewer.h"


/*! \brief Inherits QGLViewer to provide an extended  viewer that is more suited to simply displaying the contents of a world scene.
 *
 * <strong>Note:</strong> This viewer relies upon / requires a pointer to a 
 * valid GLScreenViewer instance. This is because it requires its post-draw
 * projection matrix values, so that it can draw the projection frustum. The
 * reason that we don't know these values is because we deliberately prevent
 * modifications to be made to the projection when we draw the scene, as we want
 * to DRAW the projection (as the frustum), not EXECUTE it.
 *
 * <strong>Note:</strong> The scene is reinitialized EVERY time
 * GLWorldViewer::draw() is called, as the camera has an effect on the
 * initizliation commands and the most common reason that GLWorldViewer::draw()
 * will be called is because the camera has been repositioned / rotated, etc.
 * This behavior is unlike GLScreenViewer::draw()'s.
 *
 * <strong>Note:</strong> OpenGL viewports are <em>NOT</em> supported by this
 * viewer, as they would not make sense in this context. Hence glViewport
 * commands will be ignored when the scene is executed/drawn.
 *
 * <strong>Note:</strong> Display lists are NOT used to cache the initialization
 * and display commands, as they would enfore too many limitations. For example,
 * many OpenGL commands cannot be placed in display lists, such as those that
 * pass parameters by reference (as they may be out of scope). It would also
 * prevent the user from working with display lists themselves.
 *
 * <strong>Note:</strong> The GLWorldViewer::invert() method, used to get the
 * inverse of a 4x4 matrix actually came from Nate Robins' OpenGL Tutors.<br>
 * I contacted him via email (nate@pobox.com) on Sunday 25, January 2004
 * and got the OK from him to use it before I went ahead and used it.<br>
 * Beginning of the source code file projection.c:
 * \code
 *     projection.c
 *     Nate Robins, 1997
 *
 *     Tool for teaching about OpenGL projections.
 * \endcode
 */
class GLWorldViewer : public QGLViewer
{
public:
	GLWorldViewer( const GLScene* s, GLScreenViewer* screenViewer,
		QWidget* parent );
	GLWorldViewer( const GLScene* s, GLScreenViewer* screenViewer,
		const QGLFormat &format, QWidget* parent );

	void resetCamera();

	// The glClear, glFlush and the frustum are all enabled by default
	bool autoGLClearEnabled() const { return autoGLClear; }
	bool autoGLFlushEnabled() const { return autoGLFlush; }
	bool frustumIsDrawn() const { return frustumEnabled; }
	void setDrawFrustum( bool enable ) { frustumEnabled = enable; }
	void setAutoGLClearEnabled( bool enable ) { autoGLClear = enable; }
	void setAutoGLFlushEnabled( bool enable ) { autoGLFlush = enable; }


	/*! Reimplemented, due to GLWorldViewer::setBackgroundColor() having
	 * to be reimplemented.
	 */
	qglviewer::Vec backgroundColor() const { return myBackgroundColour; };
	/*! Same as backgroundColor(), but result is returned as three floats. */
	void getBackgroundColor( float& r, float& g, float& b ) const
	{
		r = myBackgroundColour[0];
		g = myBackgroundColour[1];
		b = myBackgroundColour[2];
	}
	/*! Same as backgroundColor(), but result is returned as a QColor attribute. */
	void getBackgroundColor( QColor& c ) const
	{
		c.setRgb(
			(int)(myBackgroundColour[0]*255.0),
			(int)(myBackgroundColour[1]*255.0),
			(int)(myBackgroundColour[2]*255.0) );
	};

	/*! Reimplemented, as to gain more control over when the glClearColor call
	 * is made (so that the user can override it if they call it) and also to
	 * overcome a bug in QGLViewer, where a makeCurrent() call is not made
	 * before glClearColor is called, hence any viewer could otherwise end-up
	 * being effected.
	 */
	void setBackgroundColor( const qglviewer::Vec& color )
		{ myBackgroundColour = color; };
	/*! Same as setBackgroundColor(), but with float parameters. */
	void setBackgroundColor( float r, float g, float b )
		{ setBackgroundColor(qglviewer::Vec(r, g ,b)); };
	/*! Same as setBackgroundColor, but with a QColor attribute. */
	void setBackgroundColor( const QColor &c ) {
		setBackgroundColor( (float)c.red() / 255.0,
		                    (float)c.green() / 255.0,
		                    (float)c.blue() / 255.0 );
	};


protected :
	void sharedInit();

	void drawFrustum( GLdouble* inverseProjectionMatrix );

	virtual void init();
	virtual void draw();
	virtual void preDraw();
	virtual void postDraw();

	virtual void keyPressEvent( QKeyEvent* e );
	virtual void keyReleaseEvent( QKeyEvent* e );
	
	virtual void mousePressEvent( QMouseEvent* e );
	virtual void mouseReleaseEvent( QMouseEvent* e );
	virtual void mouseDoubleClickEvent( QMouseEvent* e );
	virtual void mouseMoveEvent( QMouseEvent* e );

	void mouseEventTranslator( QMouseEvent* e, QEvent::Type mouseEventType );

	GLboolean invert( const GLdouble src[16], GLdouble inverse[16] );

	const GLScene* myScene;
	GLScreenViewer* myScreenViewer;
	const GLdouble* myScreenViewersProjection;

	bool reinitRequired;

	qglviewer::Vec myBackgroundColour;
	
	bool autoGLClear;
	bool autoGLFlush;
	bool frustumEnabled;

	int extModifierKey;
};


#endif // __GLWORLDVIEWER_H__
