/****************************************************************************

 File:    superspinbox.cpp
 Created: by Aidan Lane, January  02, 2004
 Updated: by Aidan Lane, February 16, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include <qvalidator.h>
#include <qfontmetrics.h>
#include <qsizepolicy.h>

#include "superspinbox.h"


/* NOTES ABOUT HOW THIS REIMPLEMENTATION WORKS:
 * The double and integer min, max, linestep and values can be (and almost
 * always will be) TOTALLY DIFFERENT.
 * The double system could range [0.2-0.45], have a step of 0.05 and a value
 * of 0.3. While the integer system will always range [0-2], and have a step
 * of 1, but the value can be variable within this range.
 * See below... the definitions of INT_MIN_POS, INT_MID_POS, etc...
 *
 * The following allow for control over the up/down buttons:
 * If we call:
 *   QSpinBox::setValue( INT_MIN_POS ), then the DOWN button will be DISABLED
 *   QSpinBox::setValue( INT_MID_POS ), then BOTH of the buttons will be ENABLED
 *   QSpinBox::setValue( INT_MAX_POS ), then the UP button will be DISABLED
 * Note: the up/down buttons will be enabled/disabled upon the next updateDisplay() call
 */
#define INT_MIN_POS   0
#define INT_MID_POS   1
#define INT_MAX_POS   2
#define INT_LINE_STEP 1


/*!
 * Constructor.
 *
 * This will initialize the double's min, max and line step values to their
 * defaults of 0.0, 99.0 and 1.0 (respectively).
 */
SuperSpinBox::SuperSpinBox( QWidget* parent, const char* name )
	: QSpinBox( parent, name )
{
	d_minValue = DEFAULT_MIN_VALUE;
	d_maxValue = DEFAULT_MAX_VALUE;
	d_lineStep = DEFAULT_STEP_VALUE;

	sharedInit();
}


/*!
 * Constructor.
 *
 * This constructor takes the double's min, max and line step values as
 * parameters, instead of using defaults.
 */
// Note: there is NO need to pass the integer equivalents to QSpinBox
SuperSpinBox::SuperSpinBox( double minValue, double maxValue, double step,
	QWidget* parent, const char* name )
	: QSpinBox( parent, name ),
		d_minValue(minValue), d_maxValue(maxValue), d_lineStep(step)
{
	sharedInit();
}


/*!
 * This method is used by the constructors to perform tasks that are common to
 * all of them.
 */
// initialize d_minValue, d_maxValue and d_lineStep BEFORE this!
void SuperSpinBox::sharedInit()
{
	d_value = DEFAULT_VALUE;

	// Initialize the double validator
	validator = new QDoubleValidator( this );
	validator->setBottom( d_minValue );
	validator->setTop( d_maxValue );
	setValidator( validator );

	// The following allow for control over the up/down buttons
	QSpinBox::setRange( INT_MIN_POS, INT_MAX_POS );
	QSpinBox::setLineStep( INT_LINE_STEP );
	QSpinBox::setValue( INT_MID_POS );

	// Set the spin box to expdand to fill its entire region
	setSizePolicy(
		QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding, 1, 1) );
}


/*!
 * Reimplemented as to update the double min value and the editor's validator.
 */ 
void SuperSpinBox::setMinValue( double min )
{
	d_minValue = min;
	validator->setBottom( d_minValue );
	QSpinBox::setMinValue( INT_MIN_POS );
}


/*!
 * Reimplemented as to update the double max value and the editor's validator.
 */ 
void SuperSpinBox::setMaxValue( double max )
{
	d_maxValue = max;
	validator->setTop( d_maxValue );
	QSpinBox::setMaxValue( INT_MAX_POS );
}


/*!
 * Reimplemented as to update the double's line step (increment/decrement) value.
 */ 
void SuperSpinBox::setLineStep( double step )
{
	d_lineStep = step;
	QSpinBox::setLineStep( INT_LINE_STEP );
}


/*!
 * Reimplemented as to update the double value and to update the editor widget.
 */ 
void SuperSpinBox::setValue( double value )
{
	if ( value < d_minValue || value > d_maxValue ) {
		return;	// out of range
	}

	d_value = value;

	// NOTE: valueChange will look after bounds checks and enabling / disabling
	// the  up / down buttons
	valueChange();	// explicitly called as the integer value has NOT changed
}


/*!
 * Increment the double value by <em>d_lineStep</em> and update the editor widget.
 */
void SuperSpinBox::stepUp()
{
	// Ensure that we have the latest, freshest, most correct value.
	// This is needed, as the math errors (e.g. rounding) from step inc/decs
	// can add-up to lots of error.
	bool ok;
	mapTextToValue( &ok );
	if ( !ok )
		return;

	d_value += d_lineStep;

	// NOTE: valueChange will look after bounds checks and enabling / disabling
	// the  up / down buttons
	valueChange();	// explicitly called as the integer value has NOT changed
}


/*!
 * Decrement the double value by <em>d_lineStep</em> and update the editor widget.
 */
void SuperSpinBox::stepDown()
{
	// Ensure that we have the latest, freshest, most correct value.
	// This is needed, as the math errors (e.g. rounding) from step inc/decs
	// can add-up to lots of error.
	bool ok;
	mapTextToValue( &ok );
	if ( !ok )
		return;

	d_value -= d_lineStep;

	// NOTE: valueChange will look after bounds checks and enabling / disabling
	// the  up / down buttons
	valueChange();	// explicitly called as the integer value has NOT changed
}


/*!
 * Reimplemented as to get the lastest DOUBLE value from the editor widget.
 */
int SuperSpinBox::mapTextToValue( bool* ok )
{
	int pos;
	QString text = cleanText();

	if ( validator->validate(text,pos) != QValidator::Invalid )
	{
		*ok = true;
		d_value = text.toDouble();
		valueChange();
		return QSpinBox::value();
	}

	*ok = false;
	return 0;
}


/*!
 * Map the value of the double value to its actual string representation.
 */
QString SuperSpinBox::mapValueToText( int )
{
	QString str = QString::number( d_value );

	// Ensure precision to atleast decimal point is shown
	// (i.e. append "." to integers)
	if ( str.find('.') == -1 && str.find('e') == -1 ) {
		str += ".0";
	}

	return str;
}


/*!
 * This looks after things that need to be updated when the value is changed.
 *
 * It will look after bounds checks and enabling / disabling the  up and down
 * buttons, update the text display and emit the SuperSpinBox::valueChanged()
 * signals.
 *
 * <strong>Note:</strong> This does not call the parent's valueChange, as we
 * don't want to emit the integer-based signals.
 */
void SuperSpinBox::valueChange()
{
	if ( d_value <= d_minValue ) {
		QSpinBox::setValue( INT_MIN_POS );	// disable the down button - we hit the min value
		d_value = d_minValue;
	}
	else if ( d_value >= d_maxValue ) {
		QSpinBox::setValue( INT_MAX_POS );	// disable the up button - we hit the max value
		d_value = d_maxValue;
	}
	else {
		QSpinBox::setValue( INT_MID_POS );	// enable both the up and down buttons
	}
	
	QSpinBox::updateDisplay();

	emit valueChanged( d_value );
	emit valueChanged( QString::number(d_value) );
}


/*!
 * Reimplemented as to update the editor's validator and to enable/disable the
 * up and down arrow buttons.
 */
void SuperSpinBox::rangeChange()
{
	if ( d_value < d_minValue ) {
		QSpinBox::setValue( INT_MIN_POS );	// disable the down button - we hit the min value
		d_value = d_minValue;
	}
	else if ( d_value > d_maxValue ) {
		QSpinBox::setValue( INT_MAX_POS );	// disable the up button - we hit the max value
		d_value = d_maxValue;
	}
	else {
		QSpinBox::setValue( INT_MID_POS );	// enable both the up and down buttons
	}
	
	validator->setBottom( d_minValue );
	validator->setTop( d_maxValue );

	QSpinBox::rangeChange();
}
