/****************************************************************************

 File:    glitch.cpp
 Created: by Aidan Lane, November 19, 2003
 Updated: by Aidan Lane, February 27, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include "glitch.h"

#include <qlayout.h>
#include <qtabwidget.h>
#include <qsizepolicy.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qstringlist.h>
#include <qstatusbar.h>
#include <qlineedit.h>
#include <qtextbrowser.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qaction.h>
#include <qiconset.h>
#include <qfile.h>
#include <qcolordialog.h>
#include <qsettings.h>
#include <qmenubar.h>
#include <qpopupmenu.h>

#include <limits.h>

#include "allcmdtrees.h"
#include "aboutdialog.h"
#include "propertiesdialog.h"

#include "cmdtreetitles.h"


#define WINDOW_CAPTION "glitch"

#define TAB_VARS_INDEX 0
#define TAB_INIT_INDEX 1
#define TAB_DISP_INDEX 2

#define TABLE_ROW_HEIGHT 29 // 25 + 2(for top margin) + 2(for bottom margin)
#define TABLE_CELL_MARGIN 2
#define TABLE_COMMAND_INDEX 0	// column number (0-indexed) to place the command in
#define TABLE_COMMAND_LABEL "Command"
#define TABLE_VARIABLE_INDEX 0	// Note: this is NOT used with TABLE_COMMAND_INDEX
#define TABLE_VARIABLE_LABEL "Variable"

#define CMD_SELECTOR_WIDTH 570
#define CMD_SELECTOR_HEIGHT 240

#define INIT_DEFAULT_CMD "glEnable"
#define DISPLAY_DEFAULT_CMD "glBegin"

// The following macros are just for setting the DEFAULTS
#define VAR_SELECTED_QCOLOR QColor(198,230,124)  // green
#define CMD_SELECTED_QCOLOR QColor(134,191,220)  // blue
#define CMD_STEP_POS_QCOLOR QColor(249,224,137)  // orange

#define MODELVIEW_MATRIX_ID 0	// the matrixSelectionGroup groupbox ID
#define PROJECTION_MATRIX_ID 1
#define MODELVIEW_MATRIX_LABEL "Model View"
#define PROJECTION_MATRIX_LABEL "Projection"
#define UNKNOWN_MATRIX_LABEL "Unknown Matrix"

#define PROJECT_FILENAME_UNTITLED "Untitled." + GLProject::fileExtension()
#define PROJECT_FILENAME_FILTERS "Glitch Projects (*." + GLProject::fileExtension() + ");;All Files (*)"

#define SETTINGS_GLITCH_ORGANIZATION   "GlitchOrg"
#define SETTINGS_GLITCH_ROOT           "Glitch"
#define SETTINGS_OPTIONS_GROUP         "/Options"
#define SETTINGS_WORLDVIEW_GROUP       "/WorldView"
#define SETTINGS_SCREENVIEW_GROUP      "/ScreenView"
#define SETTINGS_DRAWAXES              "/DrawAxes"
#define SETTINGS_DRAWGRID              "/DrawGrid"
#define SETTINGS_DRAWFRUSTUM           "/DrawFrustum"
#define SETTINGS_DOUBLEBUFFER          "/DoubleBuffer"
#define SETTINGS_DISPLAYZBUFFER        "/DisplayZBuffer"
#define SETTINGS_ORTHOPROJECTION       "/OrthoProjection"
#define SETTINGS_PERSPECTIVEPROJECTION "/PerspectiveProjection"
#define SETTINGS_FOREGROUNDCOLOUR      "/ForegroundColour"
#define SETTINGS_BACKGROUNDCOLOUR      "/BackgroundColour"
#define SETTINGS_KEEPASPECTRATIO       "/KeepAspectRatio"
#define SETTINGS_AUTOGLCLEAR           "/AutoGLClear"
#define SETTINGS_AUTOGLFLUSH           "/AutoGLFlush"
#define SETTINGS_AUTOGLEND             "/AutoGLEnd"
#define SETTINGS_FILTERCMDLISTS        "/FilterCmdLists"
#define SETTINGS_FULLSCREEN            "/FullScreen"
#define SETTINGS_SAVEOPTIONSONEXIT     "/SaveOptionsOnExit"

#define DEFAULTS_DRAWAXES              true
#define DEFAULTS_DRAWGRID              true
#define DEFAULTS_DRAWFRUSTUM           true
#define DEFAULTS_DOUBLEBUFFER          true
#define DEFAULTS_DISPLAYZBUFFER        false
#define DEFAULTS_ORTHOPROJECTION       false
#define DEFAULTS_PERSPECTIVEPROJECTION true
#define DEFAULTS_FOREGROUNDCOLOUR      "#969696" // light grey
#define DEFAULTS_BACKGROUNDCOLOUR      "#000000" // black
#define DEFAULTS_KEEPASPECTRATIO       true
#define DEFAULTS_AUTOGLCLEAR           true
#define DEFAULTS_AUTOGLFLUSH           true
#define DEFAULTS_AUTOGLEND             true
#define DEFAULTS_FILTERCMDLISTS        true
#define DEFAULTS_FULLSCREEN            false
#define DEFAULTS_SAVEOPTIONSONEXIT     false


/*!
 * Default constructor.
 *
 * If <em>filename</em> is not empty, then Glitch will attempt to open it as the
 * initial project.
 */
Glitch::Glitch( QString filename )
{
	// Initialize variables
	currInitRow = currDispRow = currVarRow = 0;
	filterCmdLists = true;
	inStepperMode = false;
	stepperPos = oldStepperPos = -1;
	varSelectedColour = VAR_SELECTED_QCOLOR;
	cmdSelectedColour = CMD_SELECTED_QCOLOR;
	cmdStepPosColour = CMD_STEP_POS_QCOLOR;
	matrixSelection = MODELVIEW_MATRIX_ID;

	// Create the OpenGL scene, which will be used by the viewers
	scene = new GLScene();

	// Create a OpenGL widget and put it inside it screenSpaceView groupbox
	screenView = new GLScreenViewer( scene, screenSpaceView );
	QVBoxLayout *slayout = new QVBoxLayout( screenSpaceView, 1, 1, "screenViewLayout");
	slayout->addWidget( screenView, 1 );


	// Create a OpenGL widget and put it inside it worldSpaceView groupbox
	// Note: share OpenGL with screenView and render extra details
	worldView = new GLWorldViewer( scene, screenView, worldSpaceView );
	QVBoxLayout *wlayout = new QVBoxLayout( worldSpaceView, 1, 1, "worldViewLayout");
	wlayout->addWidget( worldView, 1 );

	// Initialize the tables
	// Make the table highlight colour the same as its row selection colour
	QPalette p = palette();
	QColorGroup activeGroup = p.active();
	QColorGroup inactiveGroup = p.inactive();
	QColorGroup disabledGroup = p.disabled();
	// Prep the highlight colour for the variables table
	activeGroup.setColor( QColorGroup::Highlight, varSelectedColour );
	inactiveGroup.setColor( QColorGroup::Highlight, varSelectedColour );
	disabledGroup.setColor( QColorGroup::Highlight, varSelectedColour );
	p.setActive( activeGroup );
	p.setInactive( inactiveGroup );
	p.setDisabled( disabledGroup );
	// -- VARIABLES TABLE
	varTable->horizontalHeader()->setLabel( TABLE_VARIABLE_INDEX, TABLE_VARIABLE_LABEL );
	varTable->setSelectionMode( QTable::SingleRow );
	varTable->verticalHeader()->hide();
	varTable->setLeftMargin( 0 );
	varTable->setColumnStretchable( TABLE_VARIABLE_INDEX, true );
	varTable->setFocusStyle( QTable::FollowStyle );
	varTable->setPalette( p );
	// Prep the highlight colour for the command tables
	activeGroup.setColor( QColorGroup::Highlight, cmdSelectedColour );
	inactiveGroup.setColor( QColorGroup::Highlight, cmdSelectedColour );
	disabledGroup.setColor( QColorGroup::Highlight, cmdSelectedColour );
	p.setActive( activeGroup );
	p.setInactive( inactiveGroup );
	p.setDisabled( disabledGroup );
	// -- INIT TABLE
	initCmdsTable->horizontalHeader()->setLabel( TABLE_COMMAND_INDEX, TABLE_COMMAND_LABEL );
	initCmdsTable->setSelectionMode( QTable::SingleRow );
	initCmdsTable->verticalHeader()->hide();
	initCmdsTable->setLeftMargin( 0 );
	initCmdsTable->setColumnStretchable( TABLE_COMMAND_INDEX, true );
	initCmdsTable->setFocusStyle( QTable::FollowStyle );
	initCmdsTable->setPalette( p );
	// -- DISPLAY TABLE
	dispCmdsTable->horizontalHeader()->setLabel( TABLE_COMMAND_INDEX, TABLE_COMMAND_LABEL );
	dispCmdsTable->setSelectionMode( QTable::SingleRow );
	dispCmdsTable->verticalHeader()->hide();
	dispCmdsTable->setLeftMargin( 0 );
	dispCmdsTable->setColumnStretchable( TABLE_COMMAND_INDEX, true );
	dispCmdsTable->setFocusStyle( QTable::FollowStyle );
	dispCmdsTable->setPalette( p );
	// Setup the table pointer list - allows for generic table manipulation code
	// They are indexed by their TAB PAGE position - very helpful!
	tabTables.insert( TAB_VARS_INDEX, varTable );
	tabTables.insert( TAB_INIT_INDEX, initCmdsTable );
	tabTables.insert( TAB_DISP_INDEX, dispCmdsTable );

	// Construct / initialize the command trees
	allCmdTrees = new AllCmdTrees;
	initTrees = new QPtrList<CmdTreeNode>;
	dispTrees = new QPtrList<CmdTreeNode>;
	constructCmdTrees();	// append the items from allCmdTrees into the lists

	// Initialize the command selector widget
	// Note: the command tree will be set by the tab changed slot (called on launch)
	myCmdSelector = new CmdSelector( this, "myCmdSelector",
	                                 CMD_SELECTOR_WIDTH, CMD_SELECTOR_HEIGHT );

	// Initialize the project
	project = NULL;
	fileNew(); // Note: This will also tell the OpenGL render boxes about the command lists.

	// Hide the statusbar
	statusBar()->hide();

	// Set / load the options, as appropiate
	loadOptionSettings();
	
	// If the filename is not empty (i.e != ""), then attempt to open it
	if ( !filename.isEmpty() ) {
		openProject( filename );
	}

	// If this is a MacOSX build, then tweek its menus
	#if defined(Q_OS_MACX)
	tweekMacMenus();
	#endif	// defined(Q_OS_MACX)

	// Update the scene viewers, both the screen and world space views
	// NOTE: see glworldview.cpp for why it calls screenView->updateGL() for us
	screenView->setReinitRequired( true );
	worldView->updateGL();
}


/*!
 * Destructor - deallocate memory.
 *
 * Option settings will be saved it required and things such as the command
 * trees and the project are deallocated.
 *
 * <strong>Note:</strong> Qt will look after the clean-up of the
 * <em>screenView</em>, <em>worldView</em> and <em>myCmdSelector</em>
 * (the command selector pop-up widget), along with all of the other Qt widgets,
 * as they all inherit from QObject, which has automatic de-allocation facilities.
 */
Glitch::~Glitch()
{
	// Save the options if required
	if ( saveOptionsOnExit ) {
		saveOptionSettings();
	}
	else
	{
		// NOTE: The "Save Options On Exit" must ALWAYS be saved!
		QSettings settings;
		settings.setPath( SETTINGS_GLITCH_ORGANIZATION, SETTINGS_GLITCH_ROOT );
		bool s_optionsSaveOptionsOnExit = optionsSaveOptionsOnExitAction->isOn();
		settings.beginGroup( SETTINGS_OPTIONS_GROUP );
			settings.writeEntry( SETTINGS_SAVEOPTIONSONEXIT, s_optionsSaveOptionsOnExit );
		settings.endGroup();
	}

	/* Clean-up the command trees
	 * Note: initTrees and dispTrees have autoDelete set already, so their
	 * elements will automatically be deleted too.
	 * This is not needed for allCmdTrees, as its elements belong to
	 * initTrees and dispTrees.
	 */
	delete allCmdTrees;
	delete initTrees;
	delete dispTrees;

	delete project;
}


/*!
 * Construct the initialization and display command trees, using the command
 * tree list <em>allCmdTrees</em>. The trees are identified via macros such as
 * "INITTREE_TITLE" ("Initialization"), which are defined by cmdtreetitles.h.
 *
 * See AllCmdTrees for why the command trees headers were not directly included
 * into this class and hence why we must work on a string basis, not a direct
 * object basis.
 */
void Glitch::constructCmdTrees()
{
	if ( allCmdTrees == NULL ) {
		return;
	}

	if ( initTrees != NULL )
	{
		appendTree( initTrees, allCmdTrees, ATTRIBSTREE_TITLE );
		appendTree( initTrees, allCmdTrees, FOGTREE_TITLE );
		appendTree( initTrees, allCmdTrees, INITTREE_TITLE );
		appendTree( initTrees, allCmdTrees, LIGHTTREE_TITLE );
		appendTree( initTrees, allCmdTrees, MATERIALTREE_TITLE );
		appendTree( initTrees, allCmdTrees, MATRIXTREE_TITLE );
		appendTree( initTrees, allCmdTrees, PROJECTIONTREE_TITLE );
	}

	if ( dispTrees != NULL )
	{
		appendTree( dispTrees, allCmdTrees, ATTRIBSTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, COLOURTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, CAMERATREE_TITLE );
		appendTree( dispTrees, allCmdTrees, DISPTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, DISPLISTTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, FOGTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, GLUTTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, INITTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, LIGHTTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, MATERIALTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, MATRIXTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, PRIMITSTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, PROJECTIONTREE_TITLE );
		appendTree( dispTrees, allCmdTrees, TRANSTREE_TITLE );
	}
}


/*!
 * This is a simple utility method that searches for a specific node in a
 * command tree list (by name), and if found and if it is not NULL, it adds to
 * the specified <em>to</em> list.
 *
 * <strong>Assumption:</strong> This method does <strong>NOT</strong> check to
 * see if the `to' or `trees' paremeters are NULL or not. Ensure that they are
 * not NULL BEFORE calling this method. This was done, as this method will be
 * called many times on the same `to' and `trees' lists.
 *
 * <strong>Note:</strong> This method has a request to be inlined.
 */
inline void Glitch::appendTree( QPtrList<CmdTreeNode>* to,
	QPtrList<CmdTreeNode>* trees, const QString& name )
{
	CmdTreeNode* node = CmdTreeNode::findNode(trees, name);

	if ( node != NULL ) {
		to->append( node );
	}
}


/*!
 * Attempt to load the option settings, using QSettings.
 *
 * From the Qt documentation, regarding QSettings:<br>
 * <em>
 * The QSettings class provides persistent platform-independent application
 * settings.<br>
 * On Unix systems, QSettings uses text files to store settings. On Windows
 * systems, QSettings uses the system registry. On Mac OS X, QSettings uses
 * the Carbon preferences API.
 * </em>
 *
 * If any of the settings are not found, then defaults will be used.
 *
 * <strong>Note:</strong> Regarding boolean settings for actions: The methods
 * to actually update the states of the* program will ONLY be called (by
 * actions) IF the setting is different to the initial setting for the actions.
 * Hence, for settings such as worldViewDoubleBuffer, worldViewPerspectiveProjection,
 * screenViewKeepAspectRatio, optionsAutoGLClear, optionsAutoGLFlush and
 * optionsAutoGLEnd that are ON by default, the actual actions have also been
 * set to ON by default (which is in the UI file).
 *
 * See Glitch::saveOptionSettings().
 */
void Glitch::loadOptionSettings()
{
	QSettings settings;
	settings.setPath( SETTINGS_GLITCH_ORGANIZATION, SETTINGS_GLITCH_ROOT );

	// Attempt to read the settings
	// NOTE! The second parameter in the readEntry method is the DEFAULT value,
	//       which is used if a stored vaue cannot be found.

	settings.beginGroup( SETTINGS_OPTIONS_GROUP );

		settings.beginGroup( SETTINGS_WORLDVIEW_GROUP );
			bool s_worldViewDrawAxes = settings.readBoolEntry(
				SETTINGS_DRAWAXES, DEFAULTS_DRAWAXES );
			bool s_worldViewDrawGrid = settings.readBoolEntry(
				SETTINGS_DRAWGRID, DEFAULTS_DRAWGRID );
			bool s_worldViewDrawFrustum = settings.readBoolEntry(
				SETTINGS_DRAWFRUSTUM, DEFAULTS_DRAWFRUSTUM );
			bool s_worldViewDoubleBuffer = settings.readBoolEntry(
				SETTINGS_DOUBLEBUFFER, DEFAULTS_DOUBLEBUFFER );
			bool s_worldViewDisplayZBuffer = settings.readBoolEntry(
				SETTINGS_DISPLAYZBUFFER, DEFAULTS_DISPLAYZBUFFER );
			bool s_worldViewOrthoProjection = settings.readBoolEntry(
				SETTINGS_ORTHOPROJECTION, DEFAULTS_ORTHOPROJECTION );
			bool s_worldViewPerspectiveProjection = settings.readBoolEntry(
				SETTINGS_PERSPECTIVEPROJECTION, DEFAULTS_PERSPECTIVEPROJECTION );
			QString s_worldViewForegroundColour = settings.readEntry(
				SETTINGS_FOREGROUNDCOLOUR, DEFAULTS_FOREGROUNDCOLOUR );
			QString s_worldViewBackgroundColour = settings.readEntry(
				SETTINGS_BACKGROUNDCOLOUR, DEFAULTS_BACKGROUNDCOLOUR );
		settings.endGroup();

		settings.beginGroup( SETTINGS_SCREENVIEW_GROUP );
			bool s_screenViewKeepAspectRatio = settings.readBoolEntry(
				SETTINGS_KEEPASPECTRATIO, DEFAULTS_KEEPASPECTRATIO );
		settings.endGroup();

		bool s_optionsAutoGLClear = settings.readBoolEntry(
			SETTINGS_AUTOGLCLEAR, DEFAULTS_AUTOGLCLEAR );
		bool s_optionsAutoGLFlush = settings.readBoolEntry(
			SETTINGS_AUTOGLFLUSH, DEFAULTS_AUTOGLFLUSH );
		bool s_optionsAutoGLEnd = settings.readBoolEntry(
			SETTINGS_AUTOGLEND, DEFAULTS_AUTOGLEND );
		bool s_optionsFilterCmdLists = settings.readBoolEntry(
			SETTINGS_FILTERCMDLISTS, DEFAULTS_FILTERCMDLISTS );
		bool s_optionsSaveOptionsOnExit = settings.readBoolEntry(
			SETTINGS_SAVEOPTIONSONEXIT, DEFAULTS_SAVEOPTIONSONEXIT );
		bool s_optionsFullScreen = settings.readBoolEntry(
			SETTINGS_FULLSCREEN, DEFAULTS_FULLSCREEN );

	settings.endGroup();

	// Apply the settings to the actions
	worldViewDrawAxesAction->setOn( s_worldViewDrawAxes );
	worldViewDrawGridAction->setOn( s_worldViewDrawGrid );
	worldViewDrawFrustumAction->setOn( s_worldViewDrawFrustum );
	worldViewDoubleBufferAction->setOn( s_worldViewDoubleBuffer );
	worldViewDisplayZBufferAction->setOn( s_worldViewDisplayZBuffer );
	worldViewOrthoProjectionAction->setOn( s_worldViewOrthoProjection );
	worldViewPerspectiveProjectionAction->setOn( s_worldViewPerspectiveProjection );
	screenViewKeepAspectRatioAction->setOn( s_screenViewKeepAspectRatio );
	optionsAutoGLClearAction->setOn( s_optionsAutoGLClear );
	optionsAutoGLFlushAction->setOn( s_optionsAutoGLFlush );
	optionsAutoGLEndAction->setOn( s_optionsAutoGLEnd );
	optionsFilterCmdListsAction->setOn( s_optionsFilterCmdLists );
	optionsFullScreenAction->setOn( s_optionsFullScreen );
	optionsSaveOptionsOnExitAction->setOn( s_optionsSaveOptionsOnExit );
	saveOptionsOnExit = s_optionsSaveOptionsOnExit; // insurance!

	// Apply other settings
	QColor c;
	c.setNamedColor( s_worldViewForegroundColour );
	worldView->setForegroundColor(
		(float)c.red()/255.0, (float)c.green()/255.0, (float)c.blue()/255.0 );
	c.setNamedColor( s_worldViewBackgroundColour );
	worldView->setBackgroundColor( c );
}


/*!
 * Attempt to save the option settings, using QSettings.
 *
 * Please see Glitch::loadOptionSettings() for more information.
 */
void Glitch::saveOptionSettings()
{
	QSettings settings;
	settings.setPath( SETTINGS_GLITCH_ORGANIZATION, SETTINGS_GLITCH_ROOT );

	// Get the settings from the actions
	bool s_worldViewDrawAxes = worldViewDrawAxesAction->isOn();
	bool s_worldViewDrawGrid = worldViewDrawGridAction->isOn();
	bool s_worldViewDrawFrustum = worldViewDrawFrustumAction->isOn();
	bool s_worldViewDoubleBuffer = worldViewDoubleBufferAction->isOn();
	bool s_worldViewDisplayZBuffer = worldViewDisplayZBufferAction->isOn();
	bool s_worldViewOrthoProjection = worldViewOrthoProjectionAction->isOn();
	bool s_worldViewPerspectiveProjection = worldViewPerspectiveProjectionAction->isOn();
	bool s_screenViewKeepAspectRatio = screenViewKeepAspectRatioAction->isOn();
	bool s_optionsAutoGLClear = optionsAutoGLClearAction->isOn();
	bool s_optionsAutoGLFlush = optionsAutoGLFlushAction->isOn();
	bool s_optionsAutoGLEnd = optionsAutoGLEndAction->isOn();
	bool s_optionsFilterCmdLists = optionsFilterCmdListsAction->isOn();
	bool s_optionsFullScreen = optionsFullScreenAction->isOn();
	bool s_optionsSaveOptionsOnExit = optionsSaveOptionsOnExitAction->isOn();

	// Get other settings
	float r, g, b;
	QColor c;
	worldView->getForegroundColor( r, g, b );
	c.setRgb( (int)(r*255.0), (int)(g*255.0), (int)(b*255.0) );
	QString s_worldViewForegroundColour = c.name();
	worldView->getBackgroundColor( c );
	QString s_worldViewBackgroundColour = c.name();
	

	// Attempt to write the settings

	settings.beginGroup( SETTINGS_OPTIONS_GROUP );
	
		settings.beginGroup( SETTINGS_WORLDVIEW_GROUP );
			settings.writeEntry( SETTINGS_DRAWAXES, s_worldViewDrawAxes );
			settings.writeEntry( SETTINGS_DRAWGRID, s_worldViewDrawGrid );
			settings.writeEntry( SETTINGS_DRAWFRUSTUM, s_worldViewDrawFrustum );
			settings.writeEntry( SETTINGS_DOUBLEBUFFER, s_worldViewDoubleBuffer );
			settings.writeEntry( SETTINGS_DISPLAYZBUFFER, s_worldViewDisplayZBuffer );
			settings.writeEntry( SETTINGS_ORTHOPROJECTION, s_worldViewOrthoProjection );
			settings.writeEntry( SETTINGS_PERSPECTIVEPROJECTION, s_worldViewPerspectiveProjection );
			settings.writeEntry( SETTINGS_FOREGROUNDCOLOUR, s_worldViewForegroundColour );
			settings.writeEntry( SETTINGS_BACKGROUNDCOLOUR, s_worldViewBackgroundColour );
		settings.endGroup();

		settings.beginGroup( SETTINGS_SCREENVIEW_GROUP );
			settings.writeEntry( SETTINGS_KEEPASPECTRATIO, s_screenViewKeepAspectRatio );
		settings.endGroup();

		settings.writeEntry( SETTINGS_AUTOGLCLEAR, s_optionsAutoGLClear );
		settings.writeEntry( SETTINGS_AUTOGLFLUSH, s_optionsAutoGLFlush );
		settings.writeEntry( SETTINGS_AUTOGLEND, s_optionsAutoGLEnd );
		settings.writeEntry( SETTINGS_FILTERCMDLISTS, s_optionsFilterCmdLists );
		settings.writeEntry( SETTINGS_FULLSCREEN, s_optionsFullScreen );
		settings.writeEntry( SETTINGS_SAVEOPTIONSONEXIT, s_optionsSaveOptionsOnExit );

	settings.endGroup();
}


/*!
 * This slot is called when current tab in cmdListTab is changed
 * (currentChanged signal emitted), so that it can perform the required
 * housekeeping.
 *
 * It performs things such as the following:
 * <ul>
 * <li>Enables and disables buttons such as "Remove", "Up" and "Down".</li>
 * <li>Selects the appropiate command trees for the command selector.</li>
 * <li>Sets up the current table row instance for editing for the given tab</li>
 * </ul>
 */
void Glitch::cmdListTabChanged( QWidget* w )
{
	bool cmdsPresent = false;
	int tabIndex = cmdListTabs->currentPageIndex();
	SuperTable* table = tabTables.at( tabIndex );
	CmdInstance* instance = NULL;
	int currentRow = 0;

	// Determine the tab that was selected
	switch ( cmdListTabs->indexOf(w) )
	{
		case TAB_VARS_INDEX:
			// Just disable cmdRowManipulatorButtons, select the row and return.
			// Nothing else in this method needs to be performed.
			cmdRowManipulatorButtons->setEnabled( false );
			varTableCurrentChanged( currVarRow, TABLE_VARIABLE_INDEX );
			return;
			break;
			
		case TAB_INIT_INDEX:
			currentRow = currInitRow;
			instance = project->initCmds()->at( currInitRow );
			myCmdSelector->setCmdTree( filterCmdLists
			                           ? initTrees
			                           : allCmdTrees );
			break;

		case TAB_DISP_INDEX:
			currentRow = currDispRow;
			instance = project->displayCmds()->at( currDispRow );
			myCmdSelector->setCmdTree( filterCmdLists
			                           ? dispTrees
			                           : allCmdTrees );
			break;
	}
	
	// Select the current row for editing
	cmdTableCurrentChanged( currentRow, TABLE_COMMAND_INDEX );

	// Enable/disable the pushbuttons as appropiate
	cmdRowManipulatorButtons->setEnabled( true ); // enable atleast the "insert" button
	cmdsPresent = (table->numRows() > 0);
	removeCmdButton->setEnabled( cmdsPresent );
	moveCmdUpButton->setEnabled( cmdsPresent );
	moveCmdDownButton->setEnabled( cmdsPresent );
}


/*!
 * Update the table's editor widgets accordingly to the currenly selected row.
 *
 * This is called/connected to QTable's currentChanged signal for BOTH
 * initCmdsTable and dispCmdsTable and is also called by several methods.
 */
void Glitch::cmdTableCurrentChanged( int row, int )
{
	int tabIndex = cmdListTabs->currentPageIndex();
	SuperTable* table = tabTables.at( tabIndex );
	CmdInstance* instance = NULL;
	QHBox* cmdEditor = NULL;

	// Safety - bounds checks
	if ( row < 0
		|| table == NULL
		|| row >= table->numRows() ) {
		return;
	}

	// Remove the command instance editor from the previously selected row.
	// AND set the cell's text to the instance's string representation.
	// Note: DONT DO THIS IF THE ROW DOES NOT EXIST (SUCH AS IT BEING DELETED).
	// This will also delete the cmdInstanceChanged and cmdParamChanged connections.
	if ( (tabIndex == TAB_INIT_INDEX && currInitRow < (uint)table->numRows())
	     || (tabIndex == TAB_DISP_INDEX && currDispRow < (uint)table->numRows()) )
	{
		uint oldCmdRow = 0;

		// Get/find the old command instance and its table row
		if ( tabIndex == TAB_INIT_INDEX ) {
			oldCmdRow = currInitRow;
			instance = project->initCmds()->at( oldCmdRow );
		}
		else if ( tabIndex == TAB_DISP_INDEX ) {
			oldCmdRow = currDispRow;
			instance = project->displayCmds()->at( oldCmdRow );
		}

		// Note: oldCmdRow is an unsigned integer, hence no need to check if < 0
		if ( instance != NULL && oldCmdRow < (uint)table->numRows() )
		{
			if ( instance->getEditorWidget() != NULL )
			{
				// Disconnect the cmdInstanceChanged, cmdEnabledStatusChanged
				// and cmdParamChanged signals
				// NOTE: BUT _NOT_ the varInstanceCreated signal
				// NOTE: The following method of attack is used, as the signals
				//       that we want to ensure are disconnected may not actually
				//       be connected! This way we can ensure that only the
				//       varInstanceCreated signal is connected.
				instance->disconnect( this );
				connect( instance, SIGNAL( varInstanceCreated(CmdInstance*,VarInstance*) ),
				                   SLOT( varInstanceCreated(CmdInstance*,VarInstance*) ) );

				// Clear / delete the OLD command instance editor from the
				// previously selected row.
				table->clearCellWidget( oldCmdRow, TABLE_COMMAND_INDEX );
			}

			// If the command instance has been disabled, then show it as so
			table->setRowDisabledLook( oldCmdRow, !instance->isEnabled() );

			table->setText( oldCmdRow, TABLE_COMMAND_INDEX, instance->toString() );
		}
	}
	
	// Get/find the current command instance and also update what currInitRow or
	// currDispRow refers to as the current row.
	if ( tabIndex == TAB_INIT_INDEX ) {
		currInitRow = row;
		instance = project->initCmds()->at(row);
	}
	else if ( tabIndex == TAB_DISP_INDEX ) {
		currDispRow = row;
		instance = project->displayCmds()->at(row);
	}
	
	// Create and insert a command instance editor widget into the new current
	// row's command cell and make the required signal/slot connections.
	// Also, clear the text from this newly selected cell.
	if ( instance != NULL )
	{
		cmdEditor = instance->createEditorWidget( table );
		cmdEditor->setMargin( TABLE_CELL_MARGIN );

		// If we are in stepping mode, then run the stepper update proceedure,otherwise
		// simply set the background of the selection widgets to the selected colour.
		// NOTE: this IS needed for the initialization selection too.
		if ( inStepperMode ) {
			updateStepperWidgets();
		} else {
			instance->setEditorBackgroundColour( cmdSelectedColour );
		}
		
		// NOTE! The command instance has the signals, NOT the editor widget
		connect( instance, SIGNAL( cmdInstanceChanged(CmdInstance*) ),
	                       SLOT( cmdInstanceUpdated(CmdInstance*) ) );
		connect( instance, SIGNAL( cmdEnabledStatusChanged(CmdInstance*) ),
	                       SLOT( cmdInstanceUpdated(CmdInstance*) ) );
		connect( instance, SIGNAL( cmdParamChanged(CmdInstance*) ),
	                       SLOT( cmdInstanceUpdated(CmdInstance*) ) );
						   
		table->setCellWidget( row, TABLE_COMMAND_INDEX, cmdEditor );
	}
}


/*!
 * Insert a new command row into the current table AFTER the current selection and select it.
 */
void Glitch::insertCommand()
{
	int tabIndex = cmdListTabs->currentPageIndex();
	SuperTable* table = tabTables.at( tabIndex );
	int row = table->currentRow();

	CmdTreeNode* cmd = NULL;

	// Select the command that was chosen last, OTHERWISE choose default.
	// If no commands have previously been selected (i.e. new project).
	// Note: "row" contains the number of the previously selected row.
	if ( tabIndex == TAB_INIT_INDEX )
	{
		if ( project->initCmds()->count() >= 1 ) {
			cmd = project->initCmds()->at(row)->cmd();
		} else {
			cmd = CmdTreeNode::findNode( initTrees, INIT_DEFAULT_CMD );
		}
	}
	else if ( tabIndex == TAB_DISP_INDEX )
	{
		if ( project->displayCmds()->count() >= 1 ) {
			cmd = project->displayCmds()->at(row)->cmd();
		} else {
			cmd = CmdTreeNode::findNode( dispTrees, DISPLAY_DEFAULT_CMD );
		}
	}

	// Create the new command instance (with params set to their defaults)
	CmdInstance* instance = new CmdInstance( cmd );

	// Make a connection that ensures that Glitch is told straight away when a
	// the new command instance creates a new variable instance.
	connect( instance, SIGNAL( varInstanceCreated(CmdInstance*,VarInstance*) ),
	                      SLOT( varInstanceCreated(CmdInstance*,VarInstance*) ) );

	instance->setCmdSelector( myCmdSelector );

	// Insert a new element into the command instance list
	if ( tabIndex == TAB_INIT_INDEX ) {
		project->initCmds()->insert( row+1, instance );
	}
	else if ( tabIndex == TAB_DISP_INDEX ) {
		project->displayCmds()->insert( row+1, instance );
	}

	// Insert 1 new row into the table at the current position and select it
	table->insertRows( row+1, 1 );
	table->setRowHeight( row+1, TABLE_ROW_HEIGHT );
	table->setCurrentCell( row+1, TABLE_COMMAND_INDEX );

	// Update things that rely on the commands, like the OpenGL scene
	updateSceneDependents();
	updateCmdInstanceVars( instance );

	// There now must be command rows to delete - enable the remove button
	removeCmdButton->setEnabled( true );
	moveCmdUpButton->setEnabled( true );
	moveCmdDownButton->setEnabled( true );

	// Make note that the project has been modified
	unsavedChangesExist = true;
}


/*!
 * Remove / delete the currently selected command row from the table.
 *
 * This will perform all required clean-ups, such as destroying any related
 * variable instances.
 *
 * This will also disable the "Remove", "Up" and "Down" buttons if the current
 * command table does not contain any entries after the removal and then finally
 * update any command dependents (such as the OpenGL scene and its dependents).
 */
void Glitch::removeCommand()
{
	int tabIndex = cmdListTabs->currentPageIndex();
	SuperTable* table = tabTables.at( tabIndex );
	int row = table->currentRow();
	CmdInstance* instance = NULL;

	// Determine the command instance to be deleted
	if ( tabIndex == TAB_INIT_INDEX ) {
		instance = project->initCmds()->at( row );
	} else if ( tabIndex == TAB_DISP_INDEX ) {
		instance = project->displayCmds()->at( row );
	}

	// The instance is about to be destroyed, so make sure that it does not talk
	// to ANYONE and make any trouble for us.
	// Disconnects its cmdInstanceChanged, cmdEnabledStatusChanged,
	// cmdParamChanged and varInstanceCreated signals.
	instance->disconnect();

	// Delete any existing variable entries and their table rows.
	// NOTE! This __MUST__ be done __BEFORE__ the actual command instance is
	//       deleted, as varTableCurrentChanged will be called, which needs to
	//       talk to the variable instance, which is encapsulated by the
	//       command instance. (Hehe!)
	if ( varCmdInfoMap.contains( instance ) )
	{
		QPtrListIterator<VarInstance> o_it( varCmdInfoMap[instance].vars );
		VarInstance* varInst;
		while ( (varInst = o_it.current()) != NULL ) {
			++o_it;
			varInstanceDestroyedCleanup( varInst );
			
		}
		varCmdInfoMap.erase( instance ); // remove the mapping for this instance
	}
	
	// Remove the command instance from the command instance list
	if ( tabIndex == TAB_INIT_INDEX ) {
		project->initCmds()->remove( row );
	} else if ( tabIndex == TAB_DISP_INDEX ) {
		project->displayCmds()->remove( row );
	}

	// Remove the row from the table
	table->removeRow( row );

	// SELECT THE NEW CURRENT ROW - SINCE WE JUST REMOVED ONE
	// If the top item was remove then select the new top item,
	// otherwise select the the item above
	// Note: this can only be done if there is >0 rows left in the table
	if ( table->numRows() > 0 )
	{
		table->setCurrentCell( ( (row==0)
		                         ? 0
		                         : row-1 ),
		                       TABLE_COMMAND_INDEX );
	}
	else
	{
		// There are no command rows to delete - disable the remove button
		removeCmdButton->setEnabled( false );
		moveCmdUpButton->setEnabled( false );
		moveCmdDownButton->setEnabled( false );
	}

	// If we are in stepping mode and we are removing the current last render
	// command, then update the stepper pos to be the one above the last
	if ( inStepperMode
	     && stepperPos >= 0
	     && stepperPos == (int)project->displayCmds()->count() ) {
		stepperPos--;
	}

	// Update things that rely on the commands, like the OpenGL scene
	updateSceneDependents();

	// Make note that the project has been modified
	unsavedChangesExist = true;
}


/*!
 * Move the currently selected command in the currently selected list/tab up,
 * if possible (e.g. you can't move the upper-most command up any further).
 *
 * See Glitch::moveCommand() for more details.
 */
void Glitch::moveCmdUp()
{
	int tabIndex = cmdListTabs->currentPageIndex();
	SuperTable* table = tabTables.at( tabIndex );
	int currentRow = table->currentRow();
	
	moveCommand( tabIndex, currentRow, currentRow-1 );

	// Make note that the project has been modified
	unsavedChangesExist = true;
}


/*!
 * Move the currently selected command in the currently selected list/tab down,
 * if possible (e.g you can't move the bottom-most command down any further).
 *
 * See Glitch::moveCommand() for more details.
 */
void Glitch::moveCmdDown()
{
	int tabIndex = cmdListTabs->currentPageIndex();
	SuperTable* table = tabTables.at( tabIndex );
	int currentRow = table->currentRow();
	
	moveCommand( tabIndex, currentRow, currentRow+1 );

	// Make note that the project has been modified
	unsavedChangesExist = true;
}

/*!
 * Move a command instance from the <em>fromPos</em> to the <em>toPos</em> in
 * the table in tab <em>tabIndex</em>.
 *
 * This looks after bounds checks, selection, visibility and updating of
 * dependents.
 *
 * This method is used by Glitch::moveCmdUp() and Glitch::moveCmdDown(), as it
 * abstracts their core functionality.
 */
void Glitch::moveCommand( int tabIndex, int fromPos, int toPos )
{
	SuperTable* table = tabTables.at( tabIndex );

	// Can't move command if it is the first or the last in the list
	// AND Can't move the bottom-most item any lower
	if ( toPos < 0 || toPos >= table->numRows() ) {
		table->ensureCellVisible( fromPos, TABLE_COMMAND_INDEX );
		return;
	}
	
	// Take the ABOVE command instance out of the list and reinsert it below the
	// current instance. The command list we do this too depends on tabIndex.
	if ( tabIndex == TAB_INIT_INDEX )
	{
		CmdInstance* instance = project->initCmds()->take( toPos );
		project->initCmds()->insert( fromPos, instance );
	}
	else if ( tabIndex == TAB_DISP_INDEX )
	{
		CmdInstance* instance = project->displayCmds()->take( toPos );
		project->displayCmds()->insert( fromPos, instance );
	}
	
	// Re-select the row (it has now moved up 1 position in the table)
	table->clearSelection();
	table->selectRow( toPos );
	table->setCurrentCell( toPos, TABLE_COMMAND_INDEX );
	table->ensureCellVisible( toPos, TABLE_COMMAND_INDEX );
	cmdTableCurrentChanged( toPos, TABLE_COMMAND_INDEX );

	// Update things that rely on the commands, like the OpenGL scene
	updateSceneDependents();
}


/*!
 * This slot is connected to the cmdInstanceChanged and cmdParamChanged signals.
 *
 * Hence, this is called when one of the command instance's parameter input
 * widget's value is changed and when a command instance's actual command is
 * changed.
 *
 * This ensures that the required housekeeping is performed, such as updating
 * the OpenGL scene and its dependents and any associated external variables.
 */
void Glitch::cmdInstanceUpdated( CmdInstance* instance )
{
	updateSceneDependents();
	updateCmdInstanceVars( instance );

	// Make note that the project has been modified
	unsavedChangesExist = true;
}


/*!
 * Update the OpenGL scene dependents, such as stepper widgets and matrix values.
 */
void Glitch::updateSceneDependents()
{
	if ( cmdListTabs->currentPageIndex() == TAB_VARS_INDEX )
	{
		CmdInstance* instance = varToCmdInstMap[ orderedVarList.at(currVarRow) ];
		if ( instance != NULL ) {
			instance->updateParamValues();
		}
	}

	// NOTE: The following will also make the updateGL() and updateMatrixValues calls
	updateStepperWidgets();

	// Make note that the project has been modified
	unsavedChangesExist = true;
}


/*!
 * This method looks after the creation, updating and deletion of the external
 * variables from the external variables table.
 *
 * It also ensures that variable instance names are kept unique.
 *
 * <strong>Note:</strong> This method can and IS used for command instances that
 * don't or longer contain variable instances. This is because it is in charge
 * the deletion of variable listings from the variable table and also from
 * management variables (such as varCmdInfoMap, varToCmdInstMap, uniqueVarNames
 * and orderedVarList).
 */
void Glitch::updateCmdInstanceVars( CmdInstance* instance )
{
	// This whole method requires a command instace, don't proceed without it
	if ( instance == NULL ) {
		return;
	}

	QMap<uint,VarInstance*> externVarsMap = instance->externVarsMap();

	/* Only proceed with the contents of the IF block if the command instance
	 * already exists with external variables, but the actual command tree node
	 * command prototype it is representing and hence variables has changed.
	 * __OR__ If the command instance has never owned external variables before.
	 * Hence, NOT if the instance and the actual command has stayed the same,
	 * as in this case, we jump to the ELSE, where a simple update is performed.
	 */
	if ( (varCmdInfoMap.contains(instance) && instance->cmd() != varCmdInfoMap[instance].cmd)
		|| varCmdInfoMap.contains(instance) == false )
	{
		uint row;
		VarInstance* varInst;

		// UPDATE INSTANCE'S COMMAND MAPPING
		
		// copy the current command pointer
		varCmdInfoMap[instance].cmd = instance->cmd();
		
		// Delete any existing variable entries and their table rows
		// -- this should NEVER occur (thanks to each variable instance's
		//    destroyed signal being tapped onto), BUT just in case something
		//    goes wrong, it is.
		QPtrListIterator<VarInstance> o_it( varCmdInfoMap[instance].vars );
		while ( (varInst = o_it.current()) != NULL ) {
			++o_it;
			varInstanceDestroyedCleanup( varInst );
		}

		// clear then copy the current variable instance mappings
		varCmdInfoMap[instance].vars.clear();
		QMap<uint,VarInstance*>::Iterator c_it;
		for ( c_it = externVarsMap.begin(); c_it != externVarsMap.end(); ++c_it ) {
			varCmdInfoMap[instance].vars.append( (*c_it) );
		}

		// create new variable entries and their table rows
		QPtrListIterator<VarInstance> n_it( varCmdInfoMap[instance].vars );
		while ( (varInst = n_it.current()) != NULL )
		{
			++n_it;

			// Ensure unique variable names
			if ( makeVarNameUnique(varInst) ) { // name changed, so update
				instance->updateParamWidgets();
			}

			row = varTable->numRows(); // new row number - 1 beyond the last
			orderedVarList.append( varInst );	// at the end
			varTable->insertRows( row );	// at the end
			varTable->setRowHeight( row, TABLE_ROW_HEIGHT );
			varTable->setText( row, TABLE_COMMAND_INDEX, varInst->toString() );
			varToCmdInstMap[varInst] = instance; // used for fast-lookups

			// Ensure that we are told when a variable instance is being destroyed
			connect( varInst, SIGNAL( destroyed(QObject*) ),
		    	              SLOT( varInstanceDestroyedCleanup(QObject*) ) );
		}
	}
	else
	{
		// Simple update - for example when a variable's name is changed
		uint row;
		VarInstance* varInst;
		QPtrListIterator<VarInstance> it( varCmdInfoMap[instance].vars );
		while ( (varInst = it.current()) != NULL )
		{
			++it;

			// Ensure unique variable names
			if ( makeVarNameUnique(varInst) ) { // name changed, so update
				instance->updateParamWidgets();
			}

			row = orderedVarList.find( varInst ); // find the variable instance

			// Depending on whether or not the variable is currently selected
			// editing, update the editor or just simply the table's text.
			if ( row == currVarRow ) {
				// Update editor
				varTableCurrentChanged( currVarRow, TABLE_VARIABLE_INDEX );
			}
			else {
				// Update just the text
				varTable->setText( row, TABLE_COMMAND_INDEX, varInst->toString() );
			}
		}
	}	
}


/*!
 * This is called by a command instance when it creates a new variable instance.
 *
 * This will look after the creation / update of records that track the
 * variables, insertion of a new row in the variables table, ensuring that the
 * variable name is unique and connects a slot to the variable instance's
 * destroyed() signal.
 */
void Glitch::varInstanceCreated( CmdInstance* instance, VarInstance* varInst )
{
	int row;

	varCmdInfoMap[instance].cmd = instance->cmd();
	varCmdInfoMap[instance].vars.append( varInst );
		
	row = varTable->numRows(); // new row number - 1 beyond the last
	orderedVarList.append( varInst );	// at the end
	varTable->insertRows( row );	// at the end
	varTable->setRowHeight( row, TABLE_ROW_HEIGHT );
	varTable->setText( row, TABLE_COMMAND_INDEX, varInst->toString() );
	varToCmdInstMap[varInst] = instance; // used for fast-lookups

	// Ensure that we are told when a variable instance is being destroyed
	connect( varInst, SIGNAL( destroyed(QObject*) ),
	                  SLOT( varInstanceDestroyedCleanup(QObject*) ) );

	// Ensure unique variable names
	if ( makeVarNameUnique(varInst) ) { // name changed, so update
		instance->updateParamWidgets();
	}

	// Set the project as not been modified yet
	unsavedChangesExist = false;
}


/*!
 * This is connected to variable instances destroyed() signal.
 *
 * This will look after the removal of records that track the variables,
 * removal of variables table row that displays the variable and makes other
 * misc. updates.
 *
 * <strong>Note:</strong> The object's virtual table is already degenerated at
 * this point, it is not safe to call any of its functions.
 */
void Glitch::varInstanceDestroyedCleanup( QObject *obj )
{
	uint row;
	VarInstance* varInst = (VarInstance*)obj;

	// The variable instance is about to be destroyed, so make sure that it does
	// not talk to ANYONE and make any trouble for us.
	// Disconnects its varElementChanged, destroyed, etc. signals.
	varInst->disconnect();

	QString name;

	// get data.. then delete it...
	CmdInstance* instance = varToCmdInstMap[varInst];
	varCmdInfoMap[instance].vars.remove( varInst );
	
	
	// get data.. then delete it...
	QValueList<QString> keys = uniqueVarNames.keys();
	uint i;
	for (i=0; i<keys.count(); ++i )
	{
		if ( uniqueVarNames[keys[i]] == varInst )
		{
			name = keys[i];
		}
	}
	uniqueVarNames.erase( name );


	// get info... then delete it...
	row = orderedVarList.find( varInst ); // find the variable instance position
	orderedVarList.remove( row );

	varToCmdInstMap.erase(varInst); // used for fast-lookups

	/* If currVarRow > 0 __AND__ currVarRow > varRow,
	 * then the CURRENT row position will WILL change when the variable
	 * row is removed, as changes to the table occur BEFORE it.
	 * This following puts things back into sync! - TRUST ME ON THIS!
	 */
	if ( currVarRow > 0 && currVarRow > row ) {
		currVarRow--;
	}

	varTable->removeRow( row );

	// Set the project as not been modified yet
	unsavedChangesExist = false;
}


/*!
 * Take a variable instance and make sure that its name unique, when compared
 * to the rest of the variable instances.
 *
 * If the variable's name is not unique, then new names with a number appended
 * to end of original will be tried, in the order of 2, 3, 4, ..., UINT_MAX.
 *
 * <strong>Limitation:</strong> If the user is wants to AND is able to create
 * variables named varName0 all the way to varName[UINT_MAX], then a unique
 * name will NOT be returned (they deserve to be able to have un-unique
 * variable names!).

 * <strong>Note:</strong> If an empty string is passed (""), then "untitled"
 * will be the base of the new name.
 * *
 * Returns true if a modification was made, false otherwise.
 */
bool Glitch::makeVarNameUnique( VarInstance* varInst )
{
	QString varName = varInst->varName();
	bool modified = false;

	if ( varName.isEmpty() ) { // don't allow ""
		varName = "untitled";
		modified = true;
	}

	VarInstance* existInst = ( uniqueVarNames.contains(varName) )
	                           ? uniqueVarNames[varName]
	                           : NULL;

	// Remove the existing / old variable name
	QMap<QString,VarInstance*>::Iterator it;
	for ( it = uniqueVarNames.begin(); it != uniqueVarNames.end(); ++it ) {
		if ( it.data() == varInst ) {
			uniqueVarNames.erase( it.key() );
		}
	}

	// Check if variable name DOES exist, but belongs to another variable instance
	if ( existInst != NULL && existInst != varInst )
	{
		// NOTE: a limitation: if the user is wants to AND is able to create
		// varName0 to varName[UINT_MAX], then they deserve to be able to have
		// un-unique variable names...
		uint i;
		for( i=2; i <= UINT_MAX; ++i )
		{
			// Check if the variable name does NOT exist
			if ( ! uniqueVarNames.contains(varName+QString::number(i)) ) {
				varName = varName + QString::number(i);
				modified = true;
				break;
			}
		}
	}

	// Perform updates
	varInst->setVarName( varName );
	uniqueVarNames[varName] = varInst;

	return modified;
}


/*!
 * Update the table's editor widgets accordingly to the currenly selected row.
 *
 * It handles the change of the selected row (remove old/existing variable
 * instance editor and create and insert a new one for the newly seleted row),
 * and updates the current row's editor when the same row is re-selected (such
 * as when a variable's name is changed).
 *
 * This is called/connected to QTable's currentChanged signal for varTable and
 * is also called by several methods.
 */
void Glitch::varTableCurrentChanged( int row, int )
{
	uint oldVarRow = currVarRow;
	currVarRow = row;
	VarInstance* oldVarInst = orderedVarList.at( oldVarRow );
	VarInstance* varInst = orderedVarList.at( row );

	// Note: currVarRow is an unsigned integer, therefore it is always > 0
	if ( currVarRow >= (uint)varTable->numRows() ) {
		return;
	}

	// If the row being selected is already the current row AND the row already
	// has an editor in it, then only UPDATE the editor, don't destroy and recreate it.
	// This situation occurs when a command's variable name is modified, as
	// it calls varTableCurrentChanged, with row set to currVarRow.
	if ( oldVarRow == currVarRow
	     && oldVarInst != NULL && oldVarInst->getEditorWidget() != NULL )
	{
		oldVarInst->updateEditorWidget();
		return;
	}


	// Remove the editor from the previous row (if it exists)
	if ( oldVarInst != NULL )
	{
		// Ensure that the variable instance's varElementChanged signal IS NOT
		// connected to us, BUT ALSO ensure that its destroyed IS connected to
		// us still.
		oldVarInst->disconnect( this );
		connect( oldVarInst, SIGNAL( destroyed(QObject*) ),
		                     SLOT( varInstanceDestroyedCleanup(QObject*) ) );

		if ( oldVarInst->getEditorWidget() != NULL )
		{
			// Clear / remove the old, existing variable editor widget
			varTable->clearCellWidget( oldVarRow, TABLE_VARIABLE_INDEX );
		}
		varTable->setText( oldVarRow, TABLE_VARIABLE_INDEX, oldVarInst->toString() );
	}

	// Create and insert the new editor into the current row
	if ( varInst != NULL )
	{
		QHBox* varEditor = varInst->createEditorWidget( varTable );
		varEditor->setMargin( TABLE_CELL_MARGIN );
		varInst->setEditorBackgroundColour( varSelectedColour );

		// Connect the variable instance's varElementChanged event to
		// updateSceneDependents, so that changes to the elements will update
		// the scene event if the command instance is not selected.			
		connect( varInst, SIGNAL( varElementChanged() ),
		                  SLOT( updateSceneDependents() ) );

		varTable->setCellWidget( currVarRow, TABLE_VARIABLE_INDEX, varEditor );
	}
}


/*!
 * Start a new project. If one is currently open, then offer the user the chance
 * to save it.
 *
 * This is called when the program is launched to start a new project.
 *
 * <strong>Note:</strong> This will also tell the OpenGL render box about the
 * command lists.
 */
void Glitch::fileNew()
{
	// Check if the project has been modfied, if it has, then give the user the
	// to save it first, discard it, or simply cancel doing anything
	if ( project != NULL && unsavedChangesExist )
	{
		switch( QMessageBox::information( this, WINDOW_CAPTION,
				"The project contains unsaved changes\n"
				"Do you want to save the changes first?",
				"&Save", "&Discard", "&Cancel",
				0, 2 ) )	// default=Save, escape=Cancel
		{
			case 0: // Save clicked or Alt+S pressed or Enter pressed.
				fileSave();
				break;

			case 1: // Discard clicked or Alt+D pressed
				// don't save
				break;
		
			case 2:	// Cancel clicked or Alt+C pressed or Escape pressed
				// don't save and stop going any further down this path
				return;
				break;
		}

		delete project;
	}

	// Create a new project
	project = new GLProject( "Untitled Project" );
	project->setDescText( "Anonymous Author" );
	project->setDocText( "<h3>Untitled Project</h3>\n"
						"Tutorial / project documentation can be written in here.<br>\n"
						"<br>\n"
						"To edit this documentation text, select \"File\" from the menubar "
						"and then click \"Properties...\". You will see a text box entitled "
						"\"Documentation\", where you will see the source of <i>this</i> text." );

	// Tell the OpenGL scene viewers about the command lists
	scene->setInitCmds( project->initCmds() );
	scene->setDisplayCmds( project->displayCmds() );

	// Update the GUI widgets
	// Note: we don't need to handle the case where there is no project title (see above)
	setCaption( project->title() + QString(" - ") + WINDOW_CAPTION );
	docTextBrowser->setText( project->docText() ); // Update the documentation text browser's text
	currInitRow = currDispRow = currVarRow = 0;
	initCmdsTable->setNumRows( 0 );
	dispCmdsTable->setNumRows( 0 );
	varTable->setNumRows( 0 );
	initCmdsTable->setCurrentCell( 0, TABLE_COMMAND_INDEX );
	dispCmdsTable->setCurrentCell( 0, TABLE_COMMAND_INDEX );
	cmdListTabs->setCurrentPage( TAB_INIT_INDEX );	// prep the init page
	cmdListTabs->setCurrentPage( TAB_DISP_INDEX );	// show the display page
	leftSideTabs->setCurrentPage( 0 );	// select the "Documentation" page tab
	stepDisplayCheckBox->setChecked( false );	// disable stepper mode
	modelViewButton->setChecked( true );        // show the model view matrix by default

	// Look after the variable instance variables
	varCmdInfoMap.clear();
	varToCmdInstMap.clear();
	orderedVarList.clear();
	uniqueVarNames.clear();

	// Restore the OpenGL contexts for the viewers (destroy and re-create them)
	optionsRestoreOpenGLContexts();

	// Set the project as not been modified yet
	unsavedChangesExist = false;
}


/*!
 * Open an existing project / tutorial
 */
void Glitch::fileOpen()
{
	QString filename;

	// Check if the project has been modfied, if it has, then give the user the
	// to save it first, discard it, or simply cancel doing anything
	if ( project != NULL && unsavedChangesExist )
	{
		switch( QMessageBox::information( this, WINDOW_CAPTION,
				"The project contains unsaved changes\n"
				"Do you want to save the changes first?",
				"&Save", "&Discard", "&Cancel",
				0, 2 ) )	// default=Save, escape=Cancel
		{
			case 0: // Save clicked or Alt+S pressed or Enter pressed.
				fileSave();
				break;

			case 1: // Discard clicked or Alt+D pressed
				// don't save
				break;
		
			case 2:	// Cancel clicked or Alt+C pressed or Escape pressed
				// don't save and stop going any further down this path
				return;
				break;
		}
	}

	filename = QFileDialog::getOpenFileName(
					project->filename(),
					PROJECT_FILENAME_FILTERS,
					this,
					NULL,
					QString( "Open File - " ) + WINDOW_CAPTION );

	// Check if the user clicked "Cancel" in the dialog... empty filename string ("")
	// Note: The dialog will NOT allow the user to click "OK" without
	//       specifying a filename.
	if ( filename.isEmpty() ) {
		return;
	}

	// Attempt to open the file
	openProject( filename );

	// Restore the OpenGL contexts for the viewers (destroy and re-create them)
	optionsRestoreOpenGLContexts();
}


/*!
 * Save the current project / tutorial
 */
void Glitch::fileSave()
{
	// If the project has not yet got a filename ( via Open or SaveAs ) then run
	// the SaveAs routine instead.
	if ( project->filename().isEmpty() )
	{
		fileSaveAs();
	}
	else
	{
		// Attempt to save the project using its current filename
		saveCurrentProject( project->filename() );
	}

	// The project now has no unsaved changes
	unsavedChangesExist = false;
}


/*!
 * Save the current project / tutorial to a new / different file
 */
void Glitch::fileSaveAs()
{
	QString filename;
	bool filenameReady = false;

	while ( ! filenameReady )
	{
		QString selectFilename = ( project->filename().isEmpty() )
		                         ? PROJECT_FILENAME_UNTITLED
		                         : project->filename(); // current project filename

		filename = QFileDialog::getSaveFileName(
						selectFilename,
						PROJECT_FILENAME_FILTERS,
						this,
						NULL,
						QString( "Save File - " ) + WINDOW_CAPTION );

		// Check if the user clicked "Cancel" in the dialog (filename is empty)
		// Note: The dialog will NOT allow the user to click "OK" without
		//       specifying a filename.
		if ( filename.isEmpty() ) {
			return;
		}

		// If the filename does not already exist, then break loop and save
		if ( ! QFile::exists(filename) ) {
			filenameReady = true;
			break;
		}

		// File name exists... warn user, give them a chnage to change it
		switch ( QMessageBox::warning( this,
						QString( "Warning - " ) + WINDOW_CAPTION,
						QString("A file with this name already exists\n")
						+ QString("Do you wish to overwrite it?"),
						"&Yes", "&No", "&Cancel",
						0, 2 ) )	// default=Yes, escape=Cancel
		{
			case 0:	// Yes clicked or Enter pressed
				// Break the loop and save
				filenameReady = true;
				break;

			case 1:	// No clicked
				// Go back through the look, presenting the save dialog again
				break;

			case 2:	// Cancel clicked or Escape pressed
				// Cancel the save, return from this method immediately
				return;
				break;
		}
	}

	// Attempt to save the project
	saveCurrentProject( filename );

	// The project now has no unsaved changes
	unsavedChangesExist = false;
}


/*!
 * Attempt to open the a project from disk.
 *
 * This does NOT look after save dialogs and so forth, that sould be performed
 * by the caller, if required. This just simply does what it says, open a project.
 */
bool Glitch::openProject( const QString& filename )
{
	uint i;
	CmdInstance* instance;
	GLProject* fileProject = new GLProject;
	
	// Attempt to open the project
	// Note: that it will erase the current initCmds and displayCmds lists
	//       -- but we still need to clear the tables here
	GLProject::OpenResult result;
	while ( (result = fileProject->openFile(filename, allCmdTrees))
			!= GLProject::Success )
	{
		switch (result)
		{
			case GLProject::Success:
				// NOTE! This case will NEVER occur, as control can ONLY flow
				//       into this block if result != GLProject::Success.
				//       Hence, this case is only here to keep the compiler happy!
				break;

			case GLProject::DiskReadError:
				// Continue on in this block, show error msg and possibly re-loop
				break;
				
			case GLProject::UnknownDocType:
				QMessageBox::warning( this, QString("Sorry - ") + WINDOW_CAPTION,
				                      "Unknown / unsupported file type.\n"
				                      "This is not a Glitch project file.",
				                      QMessageBox::Ok, QMessageBox::NoButton,
				                      QMessageBox::NoButton );
				return false;
				break;

			case GLProject::UnknownDocVersion:
				QMessageBox::warning( this, QString("Sorry - ") + WINDOW_CAPTION,
				                      "Unknown project file version.",
				                      QMessageBox::Ok, QMessageBox::NoButton,
				                      QMessageBox::NoButton );
				return false;
				break;

			case GLProject::DocParseError:
				QMessageBox::warning( this, QString("Sorry - ") + WINDOW_CAPTION,
				                      "An error was encountered while parsing "
				                      "the project file.\n"
				                      "The file seems to be corrupt.",
				                      QMessageBox::Ok, QMessageBox::NoButton,
				                      QMessageBox::NoButton );
				return false;
				break;
		}
		
		switch( QMessageBox::warning( this, QString("Sorry - ") + WINDOW_CAPTION,
					"You do not have permission to read \"" + filename + "\".",
					"&Retry", "&Cancel", QString::null,
					0, 1 ) )  // default=Retry, escape=Cancel
		{
			case 0: // Retry clicked or Enter pressed
				break;
			case 1: // Cancel clicked or Escape pressed
				return false;	// failed to open file - return out of the function
				break;
		}
	}


	// The project seems to have loaded fine...
	delete project;
	project = fileProject;

	// Tell the OpenGL scene viewers about the command lists
	// NOTE: The project as been deleted and recreated, hence POINTERS HAVE CHANGED
	scene->setInitCmds( project->initCmds() );
	scene->setDisplayCmds( project->displayCmds() );

	// Clear the tables now that the project "must" have loaded
	currInitRow = currDispRow = currVarRow = 0;
	initCmdsTable->setNumRows( 0 );
	dispCmdsTable->setNumRows( 0 );
	varTable->setNumRows( 0 );
	
	// Look after the variable instance variables
	varCmdInfoMap.clear();
	varToCmdInstMap.clear();
	orderedVarList.clear();
	uniqueVarNames.clear();

	// INITIALIZATION
	// Create rows for the commands and set text of each cell to their string representation
	initCmdsTable->setNumRows( project->initCmds()->count() );	// create the rows for the commands
	for ( i=0; i < project->initCmds()->count(); ++i )
	{
		instance = project->initCmds()->at( i );

		// If the command instance has been disabled, then show it as so
		initCmdsTable->setRowDisabledLook( i, !instance->isEnabled() );
		
		initCmdsTable->setText( i, TABLE_COMMAND_INDEX, instance->toString() );
		initCmdsTable->setRowHeight( i, TABLE_ROW_HEIGHT );
		instance->setCmdSelector( myCmdSelector );

		// Make a connection that ensures that Glitch is told straight away when a
		// the new command instance creates a new variable instance.
		connect( instance, SIGNAL( varInstanceCreated(CmdInstance*,VarInstance*) ),
	                       SLOT( varInstanceCreated(CmdInstance*,VarInstance*) ) );

		updateCmdInstanceVars( instance );	// setup the instance's variables (if any)
	}

	// DISPLAY
	// Create rows for the commands and set text of each cell to their string representation
	dispCmdsTable->setNumRows( project->displayCmds()->count() );	// create the rows for the commands
	for ( i=0; i < project->displayCmds()->count(); ++i )
	{
		instance = project->displayCmds()->at( i );

		// If the command instance has been disabled, then show it as so
		dispCmdsTable->setRowDisabledLook( i, !instance->isEnabled() );

		dispCmdsTable->setText( i, TABLE_COMMAND_INDEX, instance->toString() );
		dispCmdsTable->setRowHeight( i, TABLE_ROW_HEIGHT );
		instance->setCmdSelector( myCmdSelector );

		// Make a connection that ensures that Glitch is told straight away when a
		// the new command instance creates a new variable instance.
		connect( instance, SIGNAL( varInstanceCreated(CmdInstance*,VarInstance*) ),
	                       SLOT( varInstanceCreated(CmdInstance*,VarInstance*) ) );

		updateCmdInstanceVars( instance );	// setup the instance's variables (if any)
	}

	// Set the tab widget to show the initialization page.
	// Note: this will signal the cmdListTabChanged slot, enabling or disabling
	//       the "Remove", "Up" and "Down" pushbuttons as appropiate.
	cmdListTabs->setCurrentPage( TAB_INIT_INDEX );	// prep the init page
	initCmdsTable->ensureCellVisible( 0, TABLE_COMMAND_INDEX );
	cmdListTabs->setCurrentPage( TAB_DISP_INDEX );	// show the display page
	dispCmdsTable->ensureCellVisible( 0, TABLE_COMMAND_INDEX );

	// The following two lines are VERY IMPORTANT, as currentRow is called is
	// called many times and the value returned must be >=0.
	initCmdsTable->setCurrentCell( 0, TABLE_COMMAND_INDEX );
	dispCmdsTable->setCurrentCell( 0, TABLE_COMMAND_INDEX );

	// All "seems" to be good... update the widgets
	setCaption( (project->title().isEmpty()
	             ? QString::null
	             : project->title() + " - ")
	           + WINDOW_CAPTION );
	docTextBrowser->setText( project->docText() ); // Update the documentation text browser's text
	leftSideTabs->setCurrentPage( 0 );	// select the "Documentation" page tab
	stepDisplayCheckBox->setChecked( false );	// disable stepper mode
	modelViewButton->setChecked( true );        // show the model view matrix by default

	// Set the project as not been modified yet
	unsavedChangesExist = false;

	return true;
}


/*!
 * Save the current project to disk.
 * This does NOT look after save dialogs and so forth...
 * The project and its filename must ALREADY be defined
 * POST: on sucess, returns true AND updates the project's filename
 *       on failure, returens false
 */
bool Glitch::saveCurrentProject( const QString& filename )
{
	// Try the given filename
	project->setFilename( filename );

	// Attempt to save the project - allow the user to retry if it the process fails
	while ( !project->saveFile() )
	{
		switch( QMessageBox::warning( this, QString("Sorry - ") + WINDOW_CAPTION,
					"The project could not be saved as \"" + filename + "\".\n"
					"Please check if you have write permission.",
					"&Retry", "&Cancel", QString::null,
					0, 1 ) )	 // default=Retry, escape=Cancel
		{
			case 0: // The user clicked the Retry again button or pressed Enter
				// try again
				break;
			case 1: // The user clicked the Cancel button or pressed Escape
				// exit
				project->setFilename( QString::null );	// bad filename - don't use it
				return false;
				break;
		}
	}

	return true;
}


/*!
 * Display the project properties dialog (modal).
 * This is called by the menu item "File" -> "Properties...".
 */
void Glitch::fileProperties()
{
	PropertiesDialog *dlg = new PropertiesDialog;

	// Initialize the line and text edit boxes
	dlg->titleEdit->setText( project->title() );
	dlg->descTextEdit->setText( project->descText() );
	dlg->docTextEdit->setText( project->docText() );

	// If the use clicked "OK" (accepted), then save the text values
	if ( dlg->exec() == QDialog::Accepted )
	{
		// Update the project
		project->setTitle( dlg->titleEdit->text() );
		project->setDescText( dlg->descTextEdit->text() );
		project->setDocText( dlg->docTextEdit->text() );

		// Update the window caption and documentation text browser's text
		docTextBrowser->setText( project->docText() );
		setCaption( (project->title().isEmpty()
		             ? QString::null
	    	         : project->title() + " - ")
	        	   + WINDOW_CAPTION );

		// Make note that the project has been modified
		unsavedChangesExist = true;
	}

	delete dlg;
}


/*!
 * Reimplemented, as to add the ability to question the user upon their request
 * to exit the program, whether or not they want to save changes, discard them
 * or simply not exit anymore.
 *
 * Obviously, this only occurs if there actually are unsaved changes.
 */
void Glitch::closeEvent( QCloseEvent* e )
{
	bool ignoreEvent = false;

	// Check if the project has been modfied, if it has, then give the user the
	// to save it first, discard it, or simply cancel doing anything
	if ( project != NULL && unsavedChangesExist )
	{
		switch( QMessageBox::information( this, WINDOW_CAPTION,
				"The project contains unsaved changes\n"
				"Do you want to save the changes first?",
				"&Save", "&Discard", "&Cancel",
				0, 2 ) )	// default=Save, escape=Cancel
		{
			case 0: // Save clicked or Alt+S pressed or Enter pressed.
				fileSave();
				break;

			case 1: // Discard clicked or Alt+D pressed
				// don't save
				break;
		
			case 2:	// Cancel clicked or Alt+C pressed or Escape pressed
				// don't save and stop going any further down this path
				// NOTE: THIS IS VERY IMPORTANT - we must IGNORE the event
				ignoreEvent = true;
				break;
		}
	}

	if ( ignoreEvent ) {
		e->ignore();
	} else {
		QMainWindow::closeEvent( e );
	}
}


/*!
 * Reset the world viewer's camera back to its original position. This will also
 * stop the camera from spinning if it is currently doing so.
 */
void Glitch::worldViewResetCamera()
{
	worldView->resetCamera();
	worldView->updateGL();
}


/*!
 * Toggle the rendering of the axes (X, Y and Z) in the world view.
 */
void Glitch::worldViewDrawAxes( bool enable )
{
	worldView->setDrawAxis( enable );
	worldView->updateGL();
}


/*!
 * Toggle the rendering of the grid on the Z=0 plane in the world view.
 */
void Glitch::worldViewDrawGrid( bool enable )
{
	worldView->setDrawGrid( enable );
	worldView->updateGL();
}


/*!
 * Toggle the rendering of the projection frustum in the world view.
 */
void Glitch::worldViewDrawFrustum( bool enable )
{
	worldView->setDrawFrustum( enable );
	worldView->updateGL();
}


/*!
 * Toggle whether or not the world view OpenGL viewer should be double buffered.
 *
 * This method will only proceed if this option is toggled to a setting other
 * than what is already currently chosen. In which case, the world viewer is
 * actually destroyed and recreated with the new display format setting.
 */
void Glitch::worldViewDoubleBuffer( bool enable )
{
	// Don't do anything if the world viewer is already in the right mode
	if ( enable == worldView->format().doubleBuffer() ) {
		return;
	}

	qglviewer::Vec foregroundColour = worldView->foregroundColor();
	qglviewer::Vec backgroundColour = worldView->backgroundColor();

	// Initialize the format for the new world viewer object
	QGLFormat format;
	format.setDoubleBuffer( enable );

	// Destroy the current world viewer
	delete worldSpaceView->child( "worldViewLayout" );
	delete worldView;

	// Re-create the world viewer and put it inside it worldSpaceView groupbox.
	// Note: share OpenGL with screenView and render extra details
	worldView = new GLWorldViewer( scene, screenView, format, worldSpaceView );
	QVBoxLayout *wlayout = new QVBoxLayout( worldSpaceView, 2, 2, "worldViewLayout" );
	wlayout->addWidget( worldView, 1 );
	worldView->show();

	// Ensure that the world viewer's settings are re-initialized to be the
	// same as before.
	worldView->setDrawAxis( worldViewDrawAxesAction->isOn() );
	worldView->setDrawGrid( worldViewDrawGridAction->isOn() );
	worldView->setDrawFrustum( worldViewDrawFrustumAction->isOn() );
	worldView->setDisplayZBuffer( worldViewDisplayZBufferAction->isOn() );
	worldView->camera()->setType(
		worldViewPerspectiveProjectionAction->isOn()
		? qglviewer::Camera::PERSPECTIVE
		: qglviewer::Camera::ORTHO );
	worldView->setForegroundColor( foregroundColour );
	worldView->setBackgroundColor( backgroundColour );
}


/*!
 * Toggle whether or not the world view OpenGL viewer should show its Z / depth
 * buffer instead of the normal output.
 */
void Glitch::worldViewDisplayZBuffer( bool enable )
{
	worldView->setDisplayZBuffer( enable );
	worldView->updateGL();
}


/*!
 * Toggle whether or not the world view OpenGL viewer should use an orthographic
 * projection for its camera, or a perspective one.
 */
void Glitch::worldViewOrthoProjection( bool enable )
{
	worldView->camera()->setType(
		enable
		? qglviewer::Camera::ORTHO
		: qglviewer::Camera::PERSPECTIVE );
	worldView->updateGL();
}


/*!
 * Toggle whether or not the world view OpenGL viewer should use a perspective
 * projection for its camera, or simply an orthographic one.
 */
void Glitch::worldViewPerspectiveProjection( bool enable )
{
	worldView->camera()->setType(
		enable
		? qglviewer::Camera::PERSPECTIVE
		: qglviewer::Camera::ORTHO );
	worldView->updateGL();
}


/*!
 * Get the user to set the world view's foreground colour.
 *
 * The user will be presented with a colour selection dialog that is native to
 * the user's system.
 *
 * This colour is used when drawing things like the grid and the X, Y and Z
 * axes labels.
 */
void Glitch::worldViewSetForegroundColour()
{
	QColor c;

	// Get the world viewer's background colour
	float r, g, b;
	worldView->getForegroundColor( r, g, b );
	c.setRgb( (int)(r*255.0), (int)(g*255.0), (int)(b*255.0) );
	
	// Get the NEW background colour
	c = QColorDialog::getColor();

	// Update the world viewer's background colour
	worldView->setForegroundColor(
		(float)c.red()/255.0, (float)c.green()/255.0, (float)c.blue()/255.0 );
	worldView->updateGL();
}


/*!
 * Get the user to set the world view's background colour.
 *
 * The user will be presented with a colour selection dialog that is native to
 * the user's system.
 *
 * This colour will be used when the world viewer makes an internal
 * glClearColor call.
 *
 * <strong>Note:</strong> The user can override this setting in their Glitch
 * project, for example:
 * \code
 * glClearColor( 0.5, 0.5, 0.5, 1.0 );
 * glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
 * \endcode
 */
void Glitch::worldViewSetBackgroundColour()
{
	QColor c;

	// Get the NEW background colour, using the current one as a reference
	worldView->getBackgroundColor( c );
	c = QColorDialog::getColor( c );

	// Update the world viewer's background colour
	worldView->setBackgroundColor( c );
	worldView->updateGL();
}


/*!
 * Toggle the OpenGL screen space view to keep a 1:1 aspect ratio or not.
 */
void Glitch::screenViewKeepAspectRatio( bool enable )
{
	screenView->setKeepAspectRatio( enable );
}


/*!
 * This is used to restore the OpenGL contexts that are used by the world and
 * screen viewers, in the case where they are messed-up / confused, due to some
 * truly bad sequence of some truly bad code.
 *
 * To do this, the viewers are actually destroyed and recreated.
 */
void Glitch::optionsRestoreOpenGLContexts()
{
	/* W A R N I N G
	 * The screen viewer must be destroyed and recreated FIRST, as the world
	 * viewer needs to talk to it and obviously needs to be talking to the
	 * latest version.
	 */
	
	// * * *   S C R E E N   V I E W E R   * * *
	
	// Destroy the current screen viewer
	delete screenSpaceView->child( "screenViewLayout" );
	delete screenView;

	// Re-create the screen viewer and put it inside it screenSpaceView groupbox.
	// Note: share OpenGL with screenView and render extra details
	screenView = new GLScreenViewer( scene, screenSpaceView );
	QVBoxLayout *slayout = new QVBoxLayout( screenSpaceView, 2, 2, "screenViewLayout" );
	slayout->addWidget( screenView, 1 );
	screenView->show();

	// Ensure that the screen viewer's settings are re-initialized to be the
	// same as before.
	screenView->setKeepAspectRatio( screenViewKeepAspectRatioAction->isOn() );


	// * * *   W O R L D   V I E W E R   * * *

	// Initialize the format for the new world viewer object
	QGLFormat format;
	format.setDoubleBuffer( worldViewDoubleBufferAction->isOn() );

	qglviewer::Vec foregroundColour = worldView->foregroundColor();
	qglviewer::Vec backgroundColour = worldView->backgroundColor();

	// Destroy the current world viewer
	delete worldSpaceView->child( "worldViewLayout" );
	delete worldView;

	// Re-create the world viewer and put it inside it worldSpaceView groupbox.
	// Note: share OpenGL with screenView and render extra details
	worldView = new GLWorldViewer( scene, screenView, format, worldSpaceView );
	QVBoxLayout *wlayout = new QVBoxLayout( worldSpaceView, 2, 2, "worldViewLayout" );
	wlayout->addWidget( worldView, 1 );
	worldView->show();

	// Ensure that the world viewer's settings are re-initialized to be the
	// same as before.
	worldView->setDrawAxis( worldViewDrawAxesAction->isOn() );
	worldView->setDrawGrid( worldViewDrawGridAction->isOn() );
	worldView->setDrawFrustum( worldViewDrawFrustumAction->isOn() );
	worldView->setDisplayZBuffer( worldViewDisplayZBufferAction->isOn() );
	worldView->camera()->setType(
		worldViewPerspectiveProjectionAction->isOn()
		? qglviewer::Camera::PERSPECTIVE
		: qglviewer::Camera::ORTHO );
	worldView->setForegroundColor( foregroundColour );
	worldView->setBackgroundColor( backgroundColour );
}


/*!
 * Toggle an automatic glClear() OpenGL call for the OpenGL viwers screenView
 * and worldView.
 *
 * This option is on by default, as most of the time, you just want to get on
 * with teaching a specific part of OpenGL, not bombarding the user with all of
 * OpenGL's little management necessities.
 */
void Glitch::optionsAutoGLClear( bool enable )
{
	screenView->setAutoGLClearEnabled( enable );
	worldView->setAutoGLClearEnabled( enable );

	screenView->setReinitRequired( true );

	// Update the scene viewers, both the screen and world space views
	// NOTE: see glworldview.cpp for why it calls screenView->updateGL() for us
	worldView->updateGL();
}


/*!
 * Toggle an automatic glFlush() OpenGL call for the OpenGL viwers screenView
 * and worldView.
 *
 * This option is on by default, as most of the time, you just want to get on
 * with teaching a specific part of OpenGL, not bombarding the user with all of
 * OpenGL's little management necessities.
 */

void Glitch::optionsAutoGLFlush( bool enable )
{
	screenView->setAutoGLFlushEnabled( enable );
	worldView->setAutoGLFlushEnabled( enable );

	screenView->setReinitRequired( true );

	// Update the scene viewers, both the screen and world space views
	// NOTE: see glworldview.cpp for why it calls screenView->updateGL() for us
	worldView->updateGL();
}


/*!
 * Toggle an automatic glEnd() OpenGL call, which is needed to keep OpenGL
 * working in the case where glBegin is called but has not been followed through
 * with a glEnd call.
 *
 * This happens for example, when in stepper mode and you have stepped past a
 * glBegin but not yet to its matching glEnd call.
 *
 * It fixes the display and ALSO the ability to get the matrix values.
 */
void Glitch::optionsAutoGLEnd( bool enable )
{
	scene->setAutoGLEndEnabled( enable );

	screenView->setReinitRequired( true );

	// Update the scene viewers, both the screen and world space views
	// NOTE: see glworldview.cpp for why it calls screenView->updateGL() for us
	worldView->updateGL();
}


/*!
 * Toggle the option to either filter or show all commands in the initilzation
 * and display sections. Hence, if disabled, it would this allow the user to
 * insert any of the commands into either of the sections.
 */
void Glitch::optionsFilterCmdLists( bool enable )
{
	filterCmdLists = enable;
	// Update the current command selector by re-selecting the current tab
	// (As cmdListTabChanged looks after setting up myCmdSelector's groups)
	if ( cmdListTabs != NULL ) {
		cmdListTabChanged( cmdListTabs->currentPage() );
	}
}


/*!
 * Set this mainwindow to go into fullscreen mode or not. In fullscreen mode,
 * the title bar will be removed, the window will be maximized and any task
 * bar / kicker / dock / etc. should be hidden.
 */
void Glitch::optionsFullScreen( bool enable )
{
	if ( enable ) {
		showFullScreen();
	} else {
		showNormal();
		// Ensure that the correct icon is shown - XWindows can lose it
		setIcon( QPixmap::fromMimeSource( "glitch128.png" ) );
	}
}


/*!
 * Toggle whether or not the option settings should be remembered from this
 * session for the next session (excluding this option, as it is ALWAYS saved).
 *
 * This method is known internally as "save options on exit", as that is what
 * it really does.
 */
void Glitch::optionsSaveOptionsOnExit( bool enable )
{
	saveOptionsOnExit = enable;
}


/*!
 * Call this to restore/revet the option settings back to their defaults.
 *
 * The user will be prompted first, as to check if they really want to proceed.
 *
 * As the fore- and background colours must also be reset, the OpenGL scene
 * viewers are also updated.
 *
 * <strong>Note:</strong> If the user wishes to proceed, then
 * Glitch::optionsRestoreOpenGLContexts() will be called, as it seems fitting
 * to clean everything up (I think so anyway!). The camera will be reset by this.
 */
void Glitch::optionsRestoreDefaults()
{
	switch ( QMessageBox::warning( this,
					QString( "Warning - " ) + WINDOW_CAPTION,
					QString("Are you sure that you want to restore "
					"the default options?\n"),
					"&Yes", "&No", QString::null,
					0, 1 ) )	// default=Yes, escape=No
	{
		case 0:	// Yes clicked or Enter pressed
			// continue on to complete this process...
			break;

		case 1:	// No clicked or Escape pressed
			return;	// DO NOT CONTINUE
			break;
	}

	// Apply the settings to the actions
	worldViewDrawAxesAction->setOn( DEFAULTS_DRAWAXES );
	worldViewDrawGridAction->setOn( DEFAULTS_DRAWGRID );
	worldViewDrawFrustumAction->setOn( DEFAULTS_DRAWFRUSTUM );
	worldViewDoubleBufferAction->setOn( DEFAULTS_DOUBLEBUFFER );
	worldViewDisplayZBufferAction->setOn( DEFAULTS_DISPLAYZBUFFER );
	worldViewOrthoProjectionAction->setOn( DEFAULTS_ORTHOPROJECTION );
	worldViewPerspectiveProjectionAction->setOn( DEFAULTS_PERSPECTIVEPROJECTION );
	screenViewKeepAspectRatioAction->setOn( DEFAULTS_KEEPASPECTRATIO );
	optionsAutoGLClearAction->setOn( DEFAULTS_AUTOGLCLEAR );
	optionsAutoGLFlushAction->setOn( DEFAULTS_AUTOGLFLUSH );
	optionsAutoGLEndAction->setOn( DEFAULTS_AUTOGLEND );
	optionsFilterCmdListsAction->setOn( DEFAULTS_FILTERCMDLISTS );
	optionsFullScreenAction->setOn( DEFAULTS_FULLSCREEN );
	optionsSaveOptionsOnExitAction->setOn( DEFAULTS_SAVEOPTIONSONEXIT );
	saveOptionsOnExit = DEFAULTS_SAVEOPTIONSONEXIT; // insurance!

	// Apply other settings
	QColor c;
	c.setNamedColor( DEFAULTS_FOREGROUNDCOLOUR );
	worldView->setForegroundColor(
		(float)c.red()/255.0, (float)c.green()/255.0, (float)c.blue()/255.0 );
	c.setNamedColor( DEFAULTS_BACKGROUNDCOLOUR );
	worldView->setBackgroundColor( c );


	// Restore the OpenGL contexts for the viewers (destroy and re-create them)
	optionsRestoreOpenGLContexts();

	// Update the scene viewers, both the screen and world space views
	// NOTE: see glworldview.cpp for why it calls screenView->updateGL() for us
	//worldView->updateGL();
}


/*!
 * Set the render mode to step through the display commands one at a time.
 * _OR_ Set the render mode to loop over the display commands continuously.
 */
void Glitch::stepDisplay( bool enable )
{
	// Ensure we are looking at the display page
	if ( cmdListTabs->currentPageIndex() != TAB_DISP_INDEX )
		cmdListTabs->setCurrentPage( TAB_DISP_INDEX );

	if ( enable )  // step display
	{
		inStepperMode = true;
		// Ensure that the step display widgets are enabled
		stepDisplayGroupBox->setEnabled( true );
		scene->setStepModeEnabled( true );
		stepperReset();	// NOTE: this will make the updateStepperWidgets() call
	}
	else           // loop display
	{
		inStepperMode = false;
		// Ensure that the step display widgets are disabled
		stepDisplayGroupBox->setEnabled( false );
		scene->setStepModeEnabled( false );
		updateStepperWidgets();
		// Un-highlight the OLD stepper row position
		dispCmdsTable->setRowBackground( stepperPos, false );
	}
}


/*!
 * When in the stepper mode, this will reset the state back to the start of the
 * display commands list.
 */
void Glitch::stepperReset()
{
	// Ensure we are looking at the display page
	if ( cmdListTabs->currentPageIndex() != TAB_DISP_INDEX )
		cmdListTabs->setCurrentPage( TAB_DISP_INDEX );

	stepperPos = -1;

	updateStepperWidgets();

	// Do this last - less flicker!
	if ( ! project->displayCmds()->isEmpty() ) {
		dispCmdsTable->ensureCellVisible( stepperPos, TABLE_COMMAND_INDEX );
	}
}


/*!
 * When in the stepper mode, this will step back to the previous command
 * in the display list.
 *
 * Note: you can't step back any further than 1 position before the first command.
 */
void Glitch::stepperPrev()
{
	// Ensure we are looking at the display page
	if ( cmdListTabs->currentPageIndex() != TAB_DISP_INDEX )
		cmdListTabs->setCurrentPage( TAB_DISP_INDEX );

	if ( ! project->displayCmds()->isEmpty()
	     && stepperPos >= 0 ) {
		stepperPos--;
	}

	updateStepperWidgets();

	// Do this last - less flicker!
	if ( ! project->displayCmds()->isEmpty() ) {
		dispCmdsTable->ensureCellVisible( stepperPos, TABLE_COMMAND_INDEX );
	}
}


/*!
 * When in the stepper mode, this will step forwards to the next command
 * in the display list.
 *
 * Note: you can't step forwards any further past the last command.
 */
void Glitch::stepperNext()
{
	// Ensure we are looking at the display page
	if ( cmdListTabs->currentPageIndex() != TAB_DISP_INDEX )
		cmdListTabs->setCurrentPage( TAB_DISP_INDEX );

	if ( ! project->displayCmds()->isEmpty()
	     && stepperPos+1 < (int)project->displayCmds()->count() ) {
		stepperPos++;
	}

	updateStepperWidgets();

	// Do this last - less flicker!
	if ( ! project->displayCmds()->isEmpty() ) {
		dispCmdsTable->ensureCellVisible( stepperPos, TABLE_COMMAND_INDEX );
	}
}



/*!
 * Update widgets that are affected by the stepper, such as the display commands
 * table, current matrix mode label, matrix values, etc...
 *
 * <strong>Note:</strong> This CAN be and IS called even if the stepper mode is
 * NOT active, so that if the user just switched off stepper mode, then the
 * command row editor's background will be updated to be the selection colour.
 *
 * <strong>Note:</strong> This WILL make updateGL calls, as it needs to get the
 * latest matrix information for the current stepper position.
 */
void Glitch::updateStepperWidgets()
{
	int tabIndex = cmdListTabs->currentPageIndex();

	// Update the OpenGL boxes
	if ( inStepperMode ) {
		scene->setStepModePos( stepperPos );
	}

	uint count = project->displayCmds()->count();

	// get the background colour
	QColor bgColour = ( inStepperMode
	                    && tabIndex == TAB_DISP_INDEX
	                    && (int)currDispRow == stepperPos )
	                  ? cmdStepPosColour
	                  : cmdSelectedColour;

	// Highlight the currently selected row if it is also the stepper row
	CmdInstance* instance = NULL;
	if ( tabIndex == TAB_INIT_INDEX ) {
		instance = project->initCmds()->at(currInitRow);
	} else if ( tabIndex == TAB_DISP_INDEX ) {
		instance = project->displayCmds()->at(currDispRow);
	}
	if ( instance != NULL ) {
		instance->setEditorBackgroundColour( bgColour );
	}

	// Update the prgress text and background background colours
	if ( inStepperMode && ! project->displayCmds()->isEmpty() )
	{
		// Update the stepper progress text
		stepperProgressText->setText(
				QString::number(stepperPos+1) + " / " + QString::number(count) );

		// Un-highlight the OLD stepper row position
		dispCmdsTable->setRowBackground( oldStepperPos, false );

		// Highlight the stepper row position
		dispCmdsTable->setRowBackground( stepperPos, true, cmdStepPosColour );

		oldStepperPos = stepperPos;
	}
	else {
		stepperProgressText->setText( "0 / 0" );
	}

	// Update the scene viewers, both the screen and world space views
	// NOTE: see glworldview.cpp for why it calls screenView->updateGL() for us
	screenView->setReinitRequired( true );
	worldView->updateGL();

	// Update the current matrix identifier label, according to what it is now
	// Note: we do this AFTER the updateGL calls, so the info is up to date
	GLenum matrixMode = screenView->matrixMode();
	if ( matrixMode == GL_MODELVIEW ) {
		currentMatrixLabel->setText( MODELVIEW_MATRIX_LABEL );
	} else if ( matrixMode == GL_PROJECTION ) {
		currentMatrixLabel->setText( PROJECTION_MATRIX_LABEL );
	} else {
		currentMatrixLabel->setText( UNKNOWN_MATRIX_LABEL );
	}

	// Update the values in the 4x4 matrix
	// If NOT in stepper mode, then clear the matrix values
	updateMatrixValues( !inStepperMode );
}


/*!
 * Update which matrix is to be shown by Glitch::updateMatrixValues().
 *
 * This is called when the user changes (via the GUI) which matrix they want to
 * be shown.
 */
void Glitch::matrixSelectionClicked( int id )
{
	matrixSelection = id;
	updateMatrixValues();
}


/*!
 * Update the values in the 4x4 matrix. The values are dependent on which matrix
 * is to be shown, either Model View or Projection.
 *
 * <strong>Note:</strong> Setting clearValuesInstead to true will set each cell
 * to "0.0", instead of outputting the real matrix data. This (non-default)
 * option allows the matrix element widgets to be zeroed-out when the stepped
 * mode is disabled.
 */
void Glitch::updateMatrixValues( bool clearValuesInstead )
{
	const GLdouble* m;
	GLdouble z[16];

	if ( clearValuesInstead )
	{
		int i;
		for ( i=0; i < 16; i++ ) {
			z[i] = 0.0;
		}
		m = z;
	}
	else
	{
		// Get the matrix values corresponding to the matrix selection
		// NOTE! screenView is used and NOT worldView, as it does not use
		//       interaction and hence the matrices have not already been modified.
		switch ( matrixSelection )
		{
			case MODELVIEW_MATRIX_ID:
				m = screenView->modelviewMatrix();
				break;

			case PROJECTION_MATRIX_ID:
				m = screenView->projectionMatrix();
				break;

			default:
				return;	// unknown matrix selected
		}
	}

	// Output the matrix values
	ctmBox1_1->setText( doubleToStr(m[0]) );
	ctmBox1_2->setText( doubleToStr(m[1]) );
	ctmBox1_3->setText( doubleToStr(m[2]) );
	ctmBox1_4->setText( doubleToStr(m[3]) );
	ctmBox2_1->setText( doubleToStr(m[4]) );
	ctmBox2_2->setText( doubleToStr(m[5]) );
	ctmBox2_3->setText( doubleToStr(m[6]) );
	ctmBox2_4->setText( doubleToStr(m[7]) );
	ctmBox3_1->setText( doubleToStr(m[8]) );
	ctmBox3_2->setText( doubleToStr(m[9]) );
	ctmBox3_3->setText( doubleToStr(m[10]) );
	ctmBox3_4->setText( doubleToStr(m[11]) );
	ctmBox4_1->setText( doubleToStr(m[12]) );
	ctmBox4_2->setText( doubleToStr(m[13]) );
	ctmBox4_3->setText( doubleToStr(m[14]) );
	ctmBox4_4->setText( doubleToStr(m[15]) );
}


/*!
 * This utility method is used to get the string representation of a number of
 * type `double'. It ensures that any number that could be represented exactly
 * by an integer still has a `.0' at the end of it (e.g. `4' -> `4.0').
 */
QString Glitch::doubleToStr( double n ) const
{
	QString str = QString::number( n );

	// Ensure precision to atleast decimal point is shown
	// (i.e. append "." to integers)
	if ( str.find('.') == -1 && str.find('e') == -1 ) {
		str += ".0";
	}

	return str;
}


/*!
 * Display the about dialog (modal).
 *
 * This is called by the menu item "Help" -> "About".
 */
void Glitch::helpAbout()
{
	AboutDialog dlg;
	dlg.exec();
}


/*!
 * Tweek the menus for MacOS X - keep it looking nice!
 *
 * This is obviously a QtMac (Q_OS_MACX) specific section.
 */
#if defined(Q_OS_MACX)
void Glitch::tweekMacMenus()
{
	// Remove the icons from the main menubar item actions on MacOSX
	QIconSet emptySet = QIconSet();
	fileNewAction->setIconSet( emptySet );
	fileOpenAction->setIconSet( emptySet );
	fileSaveAction->setIconSet( emptySet );
	fileSaveAsAction->setIconSet( emptySet );
	filePropertiesAction->setIconSet( emptySet );
	fileExitAction->setIconSet( emptySet );
	helpAboutAction->setIconSet( emptySet );
}
#endif // defined(Q_OS_MACX)
