/****************************************************************************

 File:    glscreenviewer.cpp
 Created: by Aidan Lane, January  26, 2004
 Updated: by Aidan Lane, February 17, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include "glscreenviewer.h"

using namespace qglviewer;


static const GLdouble identityMatrix[16] = {
	1.0, 0.0, 0.0, 0.0,
	0.0, 1.0, 0.0, 0.0,
	0.0, 0.0, 1.0, 0.0,
	0.0, 0.0, 0.0, 1.0 };


/*!
 * Constructor.
 */
GLScreenViewer::GLScreenViewer(const GLScene* const s, QWidget* parent)
  : QGLViewer(parent), myScene(s)
{
	sharedInit();
}


/*!
 * Constructor.
 *
 * This constructor takes a QGLFormat parameter, so that special format options
 * such as double buffering can be enabled and disabled.
 */
GLScreenViewer::GLScreenViewer(const GLScene* const s, const QGLFormat &format, QWidget* parent)
  : QGLViewer(format, parent), myScene(s)
{
	sharedInit();
}


/*!
 * This method is used by the constructors to perform tasks that are common to
 * all of them.
 */
void GLScreenViewer::sharedInit()
{
	// Forbid rotation
	WorldConstraint* constraint = new WorldConstraint();
	constraint->setTranslationConstraintType( AxisPlaneConstraint::FORBIDDEN );
	constraint->setRotationConstraintType( AxisPlaneConstraint::FORBIDDEN );
	camera()->frame()->setConstraint( constraint );

	// Initialize the matrix values to the identity matrix values
	int i;
	for ( i=0; i < 16; ++i ) {
		myProjectionMatrix[i] = identityMatrix[i];
		myModelviewMatrix[i] = identityMatrix[i];
	}

	myMatrixMode = GL_MODELVIEW;

	autoGLClear = true;
	autoGLFlush = true;
	keepAspectRatio = true;
}


/*!
 * Initialize OpenGL for the context.
 *
 * This method is called by QGLViewer::initializeGL().
 *
 * The main reason that this has been implemented, is to undo the non-standard
 * modifications that QGLViewer::initializeGL() makes to the OpenGL initial state.
 */
void GLScreenViewer::init()
{
	setBackgroundColor( 0.0, 0.0, 0.0 ); // black

	// Ensure that the scene is (re-)initialized
	reinitRequired = true;
	
	/* The following commands save the current OpenGL state (well most of it
	 * anyway).
	 * NOTE: This IS needed, as reinitRequired is set to true (just above),
	 * which immediately pops off the `currently saved' pre-init state, as to
	 * make a fresh start when (re-)initializing the scene (INC. attributes).
	 */
	glPushAttrib( GL_ALL_ATTRIB_BITS );
	glMatrixMode( GL_PROJECTION );
	glPushMatrix();
	glMatrixMode( GL_MODELVIEW );
	glPushMatrix();
}


/*!
 * Execute the OpenGL scene, in its purist form.
 *
 * We also get and store the post-scene matrix mode and the model view and
 * projection matrix values, which will be used by Glitch and GLWorldViewer.
 * <strong>Note:</strong> This method will perfrom:
 *
 * \code glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_ACCUM_BUFFER_BIT | GL_STENCIL_BUFFER_BIT ) \endcode
 * if and only if <em>autoGLClear</em> == true. Note that this will be called
 * after any initialization commands and before any display ones.
 *
 * <strong>Note:</strong> If <em>reinitRequired</em> = true, then this will
 * restore the OpenGL state, delete any existing display lists and make
 * a few other cleanups so that it can then (re-)initialize the scene, by
 * executing the scene's initializetion commands.
 */
void GLScreenViewer::draw()
{
	// Run the initialization process IFF required
	if ( reinitRequired )
	{
		// Restore the `currently saved' pre-init OpenGL state
		glPopAttrib();
		glMatrixMode( GL_MODELVIEW );
		glPopMatrix();
		glMatrixMode( GL_PROJECTION );
		glPopMatrix();
		glMatrixMode( GL_MODELVIEW );

		// Save the new pre-init OpenGL state.
		glPushAttrib( GL_ALL_ATTRIB_BITS );
		glMatrixMode( GL_PROJECTION );
		glPushMatrix();
		glMatrixMode( GL_MODELVIEW );
		glPushMatrix();

		// Delete any existing display lists
		glDeleteLists( 0, INT_MAX );

		glDisable( GL_LIGHTING );
		glDisable( GL_LIGHT0 );
		glDisable( GL_DEPTH_TEST );
		glDisable( GL_COLOR_MATERIAL );

		// CLEANUP - FRESH START
		glMatrixMode( GL_PROJECTION );
		glLoadIdentity();
		glMatrixMode( GL_MODELVIEW );
		glLoadIdentity();

		// Ensure that the viewport is the same size as the widget
		glViewport( 0, 0, width(), height() );

		// *** Execute the initialization commands part of the scene
		myScene->execInitCmds();

		// Update MY RECORD of the current matrix mode to the post-init one
		GLint postInitMatrixMode;
		glGetIntegerv( GL_MATRIX_MODE, &postInitMatrixMode );
		myMatrixMode = postInitMatrixMode;
		
		reinitRequired = false;
	}
	else
	{
		// Ensure that the viewport is the same size as the widget
		glViewport( 0, 0, width(), height() );
	}

	// Note: Extra commands have been added to this OpenGL state save, as to
	//       ensure that the current matrix mode stays the same.
	GLint preSaveMatrixMode;
	glGetIntegerv( GL_MATRIX_MODE, &preSaveMatrixMode );
	glPushAttrib( GL_ALL_ATTRIB_BITS );
	glMatrixMode( GL_PROJECTION );
	glPushMatrix();
	glMatrixMode( GL_MODELVIEW );
	glPushMatrix();
	glMatrixMode( preSaveMatrixMode );

		// Execute the automatic glClear call if required
		if ( autoGLClear ) {
			glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT
				| GL_ACCUM_BUFFER_BIT | GL_STENCIL_BUFFER_BIT );
		}

		/* Restore the previous post-display current matrix mode.
		 * This is needed, as to emulate the real conditions, where the
		 * initilzation function is executed only once, but then the display
		 * function will be called over-and-over, with the OpenGL state staying
		 * the same from the end of the display function to the start of it
		 * again, for the next iteration.
		 * Hence, due to the way the matrices are pushed & popped, the current
		 * matrix mode is affected, so we need to restore it. (Phew!)
		 *
		 * NOTE: Don't do this in stepper mode, as we need to be able step
		 * backwards and this proceedure would stuff things up.
		 */
		if ( ! myScene->stepModeIsEnabled() ) {
			glMatrixMode( myMatrixMode );
		}

		// *** Execute the display commands part of the scene
		myScene->execDisplayCmds();

		// Note: I have purposely limited the matrix mode to either GL_MODELVIEW
		// or GL_PROJECTION, as to simplify matters for the user.
		GLint mode;
		glGetIntegerv( GL_MATRIX_MODE, &mode );
		if ( mode == GL_MODELVIEW || mode == GL_PROJECTION ) {
			myMatrixMode = (GLenum)mode;
		}
	
		glGetDoublev( GL_PROJECTION_MATRIX, myProjectionMatrix );
		glGetDoublev( GL_MODELVIEW_MATRIX, myModelviewMatrix );

	glPopAttrib();
	glMatrixMode( GL_MODELVIEW );
	glPopMatrix();
	glMatrixMode( GL_PROJECTION );
	glPopMatrix();
	glMatrixMode( GL_MODELVIEW );

	// Execute the automatic glFlush call if required
	if ( autoGLFlush ) {
		glFlush();
	}
}


/*!
 * Request that this widget resizes itself within its parent, so that it keeps
 * a perfect 1:1 width-to-height aspect ratio.
 *
 * <strong>Note:</strong> This method requires that the parent for this widget
 * is valid (i.e. != NULL).
 */
void GLScreenViewer::setKeepAspectRatio( bool enable )
{
	QWidget* parent;

	keepAspectRatio = enable;

	if ( (parent=parentWidget()) == NULL) {
		return;
	}

	setGeometry( 0, 0, parent->width(), parent->height() );
}


/*!
 * This method looks after actually resizing this widget within its parent,
 * so that it keeps a perfect 1:1 width-to-height aspect ratio.
 *
 * This is called by both GLScreenViewer::setKeepAspectRatio() and just as
 * importantly Qt. Qt wil call this when the parent is resized, hence asking
 * for this widget to update its size.
 */
void GLScreenViewer::setGeometry( const QRect& r )
{
	int w = r.width();
	int h = r.height();

	if ( keepAspectRatio )
	{
		int min = ( w < h ) ? w : h;

		setUpdatesEnabled( false );

		resize( min, min );
		move( QPoint((w-min)/2,(h-min)/2) );

		setUpdatesEnabled( true );
		repaint();
	}
	else
	{
		setUpdatesEnabled( false );

		resize( w, h );
		move( 0, 0 );

		setUpdatesEnabled( true );
		repaint();
	}
}


/*!
 * This is an overloaded member function, provided for convenience. It behaves
 * essentially like the above function.
 *
 * This corresponds to setGeometry( QRect(<em>x</em>, <em>y</em>, <em>w</em>,
 * <em>h</em>) )
 */
void GLScreenViewer::setGeometry( int x, int y, int w, int h )
{
	setGeometry( QRect(x,y,w,h) );
}
