/*
 *  loadObj.c
 *  This program loads an object in the Wavefront .obj format
 *  It uses code from Nate Robins OpenGL tutors software.
 *
 * You can use this code to help with assignment 2 rather than use
 * the mid-point displacement algorithm to generate the landscape.
 *
 *  Jon McCormack, August 2007
 *  Version 1.1, September 24 2007 -- added display lists and height function
 *  		This version returns the height and normal for a specifed (x,z)
 * 			position on the island.
 *
 */

#include "glm.h"
#include "height.h"
#include <stdlib.h>

/* island.obj is the island model - the dimensions 
 * are 2000 (x) x 400 (y) x 2000 (z)
 * the up direction is in +ive y
 */
#define MODEL_FILE		"data/island.obj"
#define ROTATE_INC		0.1

/* global variables */
GLMmodel* pmodel = NULL;	/* pointer to the model read in by glm */
GLuint gModelID = 0;		/* island display list */
GLfloat g_rotate, g_rotate_inc;	/* global rotation value and animation increment */
GLfloat gXpos, gZpos;		/* global position to show height */
int g_w, g_h;				/* viewport dimensions */

/*
 * loadmodel
 *
 * reads the mode and returns a display list for it
 * This version uses vertex normals for smoothing
 *
 */
GLuint loadmodel(char * modelfile)
{
    if (pmodel)
		glmDelete(pmodel);
    pmodel = glmReadOBJ(modelfile);
    if (!pmodel) exit(0);
    glmFacetNormals(pmodel);
    glmVertexNormals(pmodel, 179.0);

    return glmList(pmodel, GLM_SMOOTH | GLM_MATERIAL);
}


void display(void)
{
	 GLfloat height, normal[3] = {0,1,0};

	 /* clear the frame buffer and depth buffer */
	 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT); 

	glPushMatrix();
		glRotatef(g_rotate, 0, 1, 0); /* rotate the island */
		glEnable(GL_LIGHTING);

		glCallList(gModelID);		  /* call the island display list */

		/* draw a line at the current position pointing up 
		 * the line points in the direction of the vertex normal
		 */
		glDisable(GL_LIGHTING);
		height = getHeight(pmodel, gXpos, gZpos, normal);
		glColor3f(1,0,0);
		glBegin(GL_LINES);
		 glVertex3f(gXpos, height, gZpos);
		 glVertex3f(gXpos + normal[0]*100.0, height + normal[1] * 100.0, gZpos + normal[2] * 100.0);
		glEnd();

	glPopMatrix();

	glutSwapBuffers();
}

void resize(int w, int h)
{
	g_w = w, g_h = h;
	glViewport(0, 0, g_w, g_h);

 	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
	gluPerspective(60.0, (GLfloat)g_w / g_h, 1.0, 4000.0);
	glMatrixMode (GL_MODELVIEW);
}

void init()
{
	/* light source position */
	GLfloat light_pos[] = {300.0, 900.0, 300.0};

	/* set up a perspective projection in the projection matrix */
	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
	gluPerspective(70.0, (GLfloat)g_w / g_h, 1.0, 10000.0);

	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity();
	/* set loght position before camera transformation */
	glLightfv(GL_LIGHT0, GL_POSITION, light_pos);

	/* position camera to see the island */
	gluLookAt(1400.0, 500.0, 1400.0,  /* eye point */
			     0.0, 10.0,    0.0,  /* at point */
				 0.0,  1.0,    0.0); /* up vector */

    glClearColor(0.3, 0.3, 0.3, 0.3);
	glColor3f(1.0,1.0,1.0);

	/* turn on depth buffering and lighting */
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_LIGHTING);
	glShadeModel(GL_SMOOTH);
	glEnable(GL_LIGHT0);
}

void keyboard(unsigned char key, int x, int y)
{
	/* use the space bar to start and stop the island rotation */
	switch(key) {
	case ' ':
		if (g_rotate_inc > 0) g_rotate_inc = 0.0;
		else g_rotate_inc = 0.1;
		break;
	}
}

void specialKeys(int key, int x, int y)
{
	/* use the arrow keys to move the current line position around the island */
	switch(key) {
	case GLUT_KEY_UP:
		gXpos += 20;
		break;
	case GLUT_KEY_DOWN:
		gXpos -= 20;
		break;
	case GLUT_KEY_RIGHT:
		gZpos += 20;
		break;
	case GLUT_KEY_LEFT:
		gZpos -= 20;
		break;
	}
	glutPostRedisplay();
}

void idle()
{
	/* the idle func just increments the global island rotation and
	 * forces a redisplay.
	 */
	g_rotate += g_rotate_inc;
	if (g_rotate >= 360.0f) g_rotate = 0.0f;
	glutPostRedisplay();
}

int main(int argc, char** argv)
{

	g_w = 800, g_h = 600;
	g_rotate = 0.0;
	g_rotate_inc = ROTATE_INC;
	gXpos = 0, gZpos = 0;	/* set position */

	/* Initialize mode and open a window in upper left corner of screen */

	/* You must call glutInit before any other OpenGL/GLUT calls */
	glutInit(&argc,argv); 

	/* initialise double buffering and depth buffering */
	glutInitDisplayMode (GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH);  
	glutInitWindowSize(g_w,g_h);
	glutInitWindowPosition(0,0); 
	glutCreateWindow("load obj"); 
	glutDisplayFunc(display);
	glutReshapeFunc(resize);
	glutKeyboardFunc(keyboard);
	glutSpecialFunc(specialKeys);
	glutIdleFunc(idle);
	init();
	/* load the model into a display list */
	gModelID = loadmodel(MODEL_FILE);
	setHeights(pmodel);
	glutMainLoop();
	return 0;
}
