/****************************************************************************

 File:    cmdselector.cpp
 Created: by Aidan Lane, November 25, 2003
 Updated: by Aidan Lane, February 15, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include "cmdselector.h"

#include <qstringlist.h>
#include <qlayout.h>
#include <qheader.h>
#include <qrect.h>
#include <qpoint.h>


/*!
 * Default constructor.
 */
CmdSelector::CmdSelector( QWidget* parent, const char* name,
	int width, int height )
	: QFrame( parent, name, WType_Popup )
{
	setFrameStyle( WinPanel | Raised );

	if ( width > 0 && height > 0 ) {
		resize( width, height );
	}

	cmdList = new QListView( this );
	cmdList->header()->hide();
	cmdList->setRootIsDecorated( true );
	cmdList->addColumn( "Command" );

	QBoxLayout * l = new QVBoxLayout( this );
	l->addWidget( cmdList );

	connect( cmdList, SIGNAL( clicked(QListViewItem*) ),
	                  SLOT( listItemClicked(QListViewItem*) ) );

	currentCmd = NULL;
}


/*!
 * Pop-up the command selector widget, enabling the user to select a command.
 *
 * <strong>Note:</strong>This will try to position the pop-up window in a
 * suitable location on the screen.
 */
void CmdSelector::popup( QWidget* parent)
{
	popupParent = parent;

	// Make sure that this widget is *completely* visible on the screen
	if ( popupParent != NULL )
	{
		QDesktopWidget *d = QApplication::desktop();
		int dw = d->width();     // returns desktop width
		int dh = d->height();    // returns desktop height
		QRect r = popupParent->rect();
		QPoint pp = popupParent->mapToGlobal( popupParent->rect().bottomLeft() );

		// Determine if the popup can fit below the parent (pushbutton)
		if ( pp.y()+height() <= dh )
		{
			// Determine if the popup can fit left-aligned with the parent (pushbutton)
			if ( pp.x()+width() <= dw )
			{
				// CAN fit height AND width
				move( popupParent->mapToGlobal( r.bottomLeft() ) );
			}
			else
			{
				// CAN fit height, BUT NOT width
				QPoint p = popupParent->mapToGlobal( r.bottomRight() );
				p.setX( p.x() - width() );
				move( p );
			}
		}
		else
		{
			// Determine if the popup can fit left-aligned with the parent (pushbutton)
			if ( pp.x()+width() <= dw )
			{
				// CAN'T fit height, BUT CAN fit width
				QPoint p = popupParent->mapToGlobal( r.topLeft() );
				p.setY( p.y() - height() );
				move( p );
			}
			else
			{
				// CAN'T fit height OR width
				QPoint p = popupParent->mapToGlobal( r.topRight() );
				p.setX( p.x() - width() );
				p.setY( p.y() - height() );
				move( p );
			}

		}
	}

	show();
}



/*!
 * Set the tree of commands that will be shown in the listview.
 * Existing list view items and mappings will be cleared first.
 */
void CmdSelector::setCmdTree( QPtrList<CmdTreeNode>* tree )
{
	// Clear any existing list view items and mappings
	cmdList->clear();
	listItemToCmd.clear();
	cmdToListItem.clear();

	insertSubtree<QListView>( cmdList, tree );
}


/*!
 * Recursively inserts items into the treeview
 * Templated as the parent could be of type QListView OR QListViewItem.
 */
template<class ParentT>
void CmdSelector::insertSubtree( ParentT* parent, QPtrList<CmdTreeNode>* tree )
{
	uint i;
	for ( i=0; i < tree->count(); i++ )
	{
		CmdTreeNode* node = tree->at( i );

		// check if the node is a sub-tree
		if ( node->subTree() != NULL )
		{
			// use the sub-tree's name
			QListViewItem *subparent = new QListViewItem( parent, node->name() );
			insertSubtree<QListViewItem>( subparent, node->subTree() );
		}
		else {
			// use the command's string representation - its prototype
			QListViewItem* listItem = new QListViewItem( parent, node->toString() );
			// only command leaves get mappings
			listItemToCmd[listItem] = node;
			cmdToListItem[node] = listItem;
		}
	}
}


/*!
 * Change which item is currently selected, then close the pop-up window.
 */
void CmdSelector::listItemClicked( QListViewItem* item )
{
	// Note: signal a change only when a command leaf is selected
	//       -- ONLY command leaves exist in the item mapping
	if ( listItemToCmd.contains(item) )
	{
		currentCmd = listItemToCmd[item];
		hide();
		emit cmdSelected( currentCmd );
	}
}


/*!
 * Select a given item if it is in the list.
 * Reference: QListViewItem*
 */
void CmdSelector::selectItem( QListViewItem* item )
{
	if ( listItemToCmd.contains(item) )
	{
		// Note: the default behaviour is SINGLE selection mode
		currentCmd = listItemToCmd[item];
		cmdList->setSelected( item, true );
		cmdList->ensureItemVisible( item );
	}
}


/*!
 * Select a given command if it is in the list.
 * Reference: CmdTreeNode*
 */
void CmdSelector::selectItem( CmdTreeNode* cmd )
{
	if ( cmdToListItem.contains(cmd) )
	{
		// Note: the default behaviour is SINGLE selection mode
		currentCmd = cmd;
		cmdList->setSelected( cmdToListItem[cmd], true );
		cmdList->ensureItemVisible( cmdToListItem[cmd] );
	}
}
