/****************************************************************************

 File:    cmdtreenode.cpp
 Created: by Aidan Lane, November 25, 2003
 Updated: by Aidan Lane, February 10, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include "cmdtreenode.h"


CmdTreeNode::CmdTreeNode( const QString& name, QPtrList<CmdTreeNode>* subTree,
	void (*funcPtr) (const QStringList&), SuperPtrList<CmdParam>* params )
	: myName(name), mySubTree(subTree), myFuncPtr(funcPtr), myParams(params)
{
	if ( mySubTree ) {
		mySubTree->setAutoDelete( true );
	}

	if ( myParams ) {
		myParams->setAutoDelete( true );
	}

}


CmdTreeNode::~CmdTreeNode()
{
	// QPtrList's auto-delete is enabled, hence all of the elements in the
	// following lists will automatically deleted too.
	delete mySubTree;
	delete myParams;
}


/*!
 * Get the string representation of the command - its prototype.
 */
QString CmdTreeNode::toString() const
{
	// Build the string representation of the parameters
	QString text = myName + "(";
	if ( myParams != NULL )
	{
		text += " ";
		uint i;
		for ( i=0; i < myParams->count(); i++ )
		{
			text += myParams->at(i)->typeName();
			text += " ";
			text += myParams->at(i)->formalVarName();

			// If this is not the last parameter, then insert a comma
			if ( i != myParams->count()-1 ) {
				text += ", ";
			}
		}
		text += " ";
	}
	text += ")";


	return text;
}


/*!
 * Find an an item by its textual name in a given command group.
 *
 * This is designed to be used by the insert command operation,
 * where a certian command may need to be found for initialization.
 *
 * <strong>Note:</strong> This method is recursive.
 *
 * <strong>Note:</strong> This will only return the FIRST occurance.
 *
 * <strong>Note:</strong> This is static - so that it can be used directly on
 * an object of type QPtrList<CmdTreeNode>
 */
CmdTreeNode* CmdTreeNode::findNode( QPtrList<CmdTreeNode>* tree, QString name )
{
	if ( tree == NULL ) { // prevent looking in an uninitialized tree
		return NULL;
	}

	uint i;
	for ( i=0; i < tree->count(); ++i )
	{
		CmdTreeNode* node = tree->at( i );
		QPtrList<CmdTreeNode>* itemSubTree = node->subTree();

		// Check if the current node is a match
		// If not, try looking in the node's subtree, if one exists
		if ( node->name() == name ) {
			return node;	// node found
		}
		else if ( itemSubTree != NULL )
		{
			CmdTreeNode* retNode = findNode( itemSubTree, name );
			if ( retNode != NULL ) {
				return retNode;	// node found
			}
		}
	}

	return NULL;	// node NOT found
}
