/****************************************************************************

 File:    doubleparam.h
 Created: by Aidan Lane, November 26, 2003
 Updated: by Aidan Lane, February 26, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#ifndef __DOUBLEPARAM_H__
#define __DOUBLEPARAM_H__


#include <float.h>

#include "cmdparam.h"
#include "superspinbox.h"

#define DOUBLE_MIN -DBL_MAX
#define DOUBLE_MAX DBL_MAX
#define DOUBLE_STEP 0.1
#define DOUBLE_MAX_WIDGET_WIDTH 100


/*! \brief Represents a parameter of type <em>double</em>, used both by commmands and arrays.
 *
 * This class inherits CmdParam to provide a parameter that has a formal
 * variable name, a default value and an exact type name, among other fields.
 *
 * As an editor widget, it provides a SuperSpinBox, which is an extension of a
 * standard integer QSpinBox, but instead provides support of doubles and floats.
 */

class DoubleParam : public CmdParam {

	Q_OBJECT	// make QT's signal/slot system work

public:
	DoubleParam( const char* formalVarName, const char* defaultValue,
		const char* exactTypeName = "GLdouble",
		double minValue = DOUBLE_MIN, double maxValue = DOUBLE_MAX,
		double stepValue = DOUBLE_STEP,
		int maxWidgetWidth = DOUBLE_MAX_WIDGET_WIDTH )
		: CmdParam( formalVarName, defaultValue, exactTypeName ),
			myMinValue(minValue), myMaxValue(maxValue), myStepValue(stepValue),
			myMaxWidgetWidth(maxWidgetWidth) {}

	// widgets
	virtual QWidget* createWidget( QWidget* parent, const char* name,
		const QObject *slotOwner, const char* member )
	{
		SuperSpinBox *w = new SuperSpinBox( myMinValue, myMaxValue,
		                                    myStepValue, parent, name );
		connect( w, SIGNAL( valueChanged(double) ), slotOwner, member );
		return (QWidget*)w;
	}

	virtual void setWidgetValue( QWidget* widget, const QString& value ) {
		if ( widget != NULL )
			((SuperSpinBox*)widget)->setValue( value.toDouble() );
	}

	virtual QString getWidgetValue( QWidget* widget ) {
		return ( widget != NULL )
			? QString::number( ((SuperSpinBox*)widget)->value() )
			: QString::null;
	}

	/*!
	 * This utility method is used to get the string representation of a number of
	 * type `double'. It ensures that any number that could be represented exactly
	 * by an integer still has a `.0' at the end of it (e.g. `4' -> `4.0').
	 */
	virtual QString getStringRep( const QString& value ) const
	{
		QString str( value );

		// Ensure precision to atleast decimal point is shown
		// (i.e. append "." to integers)
		if ( str.find('.') == -1 && str.find('e') == -1 ) {
			str += ".0";
		}

		return str;
	}

	virtual int maxWidgetWidth() const {
		return myMaxWidgetWidth;
	}

private:
	double myMinValue;
	double myMaxValue;
	double myStepValue;
	int    myMaxWidgetWidth;
};


#endif	// __DOUBLEPARAM_H__
