/****************************************************************************

 File:    glscene.cpp
 Created: by Aidan Lane, January  26, 2003
 Updated: by Aidan Lane, February 19, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include "glscene.h"

#include <qgl.h>


#define GL_VIEWPORT_NAME "glViewport"
#define GL_LOAD_IDENTITY_NAME "glLoadIdentity"


/*!
 * >Constructor.
 *
 * This will initialize the object's members to the following<ul>
 * <li>initCmds = NULL</li>
 * <li>displayCmds = NULL</li>
 * <li>stepMode = false</li>
 * <li>stepModePos = 0</li>
 * <li>autoGLEnd = true</li>
 * </ul>
 */
GLScene::GLScene()
{
	initCmds = NULL;
	displayCmds = NULL;

	stepMode = false;
	stepModePos = 0;

	autoGLEnd = true;
}


/*!
 * Execute the initialization commands part of the scene.
 *
 * Please see GLScene::execCmdIntanceList() for details.
 */
void GLScene::execInitCmds( bool disableProjectionMods,
	bool disableViewports,
	bool origStateAsIdentity ) const
{
	execCmdIntanceList( initCmds, false, origStateAsIdentity,
		disableProjectionMods, disableViewports );
}


/*!
 * Execute the display commands part of the scene.
 *
 * Please see GLScene::execCmdIntanceList() for details.
 */
void GLScene::execDisplayCmds( bool disableProjectionMods,
	bool disableViewports,
	bool origStateAsIdentity ) const
{
	execCmdIntanceList( displayCmds, true, origStateAsIdentity,
		disableProjectionMods, disableViewports );
}


/*!
 * Execute the commands from one of the scene's command instance lists,
 * which is either initialization or display, set by <em>cmdInstances</em>.
 *
 * The <em>useStepper</em> parameter specifies whether or not the stepper
 * position affects which commands to execute. This should only be used
 * for the display commands, <strong>not</strong> the initialization ones.
 *
 * The <em>origStateAsIdentity</em> indicates that whenever a glLoadIdentity()
 * call is made, then the actual matrix that is to be loaded is the original
 * one, before any commands are / were executed, for either the model view
 * or project matrices.<br>
 * This is useful in the case of a world viewer, where the whole scene of
 * commands are to be transfromed with respect to a camera. In this case,
 * when a glLoadIdentity() call is made, we still want the camera to affect
 * the scene, so we must use the post camera-transformed matrices as our
 * idenities.
 *
 * The parameter <em>disableProjectionMods</em>, it allows you to prevent
 * modifications to the projection matrix. This is useful for implementing a
 * viewer that DRAWS the projection, rather that actually executing it.
 *
 * As for the parameter <em>disableViewports</em>, it allows you to prevent
 * viewports to be created. Again, this is useful for implementing a viewer
 * that does not support them, such as a global world viewer.
 *
 * <strong>Note:</strong> If <em>cmdInstances</em> = 0, then this method will
 * return without doing anything.
 *
 * <strong>Note:</strong> To fulfill the requirements of this second parameter
 * (when it is enabled), each command instance is checked to see if its name is
 * an exact match to the string GL_VIEWPORT_NAME ("glViewport").
 * Obviously if it is, then the command is ignored.
 *
 * <strong>Note:</strong> This method PURPOSELY does not save and restore the
 * current matrix, nor OpenGL attributes. This allows information about the
 * final state to be gathered by callers. Hence, when calling this method, you
 * must look after the state YOURSELF.
 *
 * See also GLScene::setAutoGLEndEnabled
 */
void GLScene::execCmdIntanceList( QPtrList<CmdInstance>* cmdInstances,
	bool useStepper, bool origStateAsIdentity,
	bool disableProjectionMods, bool disableViewports ) const
{
	GLint matrixMode;
	GLdouble origProjection[16];
	GLdouble origModelView[16];
	uint i;

	glGetDoublev( GL_PROJECTION_MATRIX, origProjection );
	glGetDoublev( GL_MODELVIEW_MATRIX, origModelView );

	// Ensure that the address of the command instance list has been specified
	if ( cmdInstances == NULL ) {
		return;
	}


	// Execute the OpenGL command instances
	uint end = ( ( !useStepper  || !stepMode
	           || cmdInstances->count() <= (stepModePos+1) )
	             ? cmdInstances->count()
				 : stepModePos+1 );
	for ( i=0; i < end; ++i )
	{
		if ( cmdInstances->at(i) == NULL || cmdInstances->at(i)->cmd() == NULL ) {
			continue; // SKIP command instance, preventing potential `seg-fault'
		}

		QString cmdName = cmdInstances->at(i)->cmd()->name();

		if ( disableViewports && cmdName == GL_VIEWPORT_NAME ) {
			continue; // SKIP command instance, move on to the next
		}

		
		glGetIntegerv( GL_MATRIX_MODE, &matrixMode );

		// If the original state is to be used for the load identity calls,
		// then do so.
		if ( origStateAsIdentity && cmdInstances->at(i)->isEnabled()
			&& cmdName == GL_LOAD_IDENTITY_NAME )
		{
			if ( matrixMode == GL_PROJECTION ) {
				glLoadMatrixd( origProjection );
			} else if ( matrixMode == GL_MODELVIEW ) {
				glLoadMatrixd( origModelView );
			}
		}
		else {
			// Execute the command, using the instance's current parameter values
			cmdInstances->at(i)->exec();
		}

		if ( disableProjectionMods )
		{
			/* If the current matrix mode is GL_PROJECTION, then ANY of the
			 * OpenGL commands within could potentially modify the projection
			 * matrix, hence we simply revert to the projection matrix that
			 * we started this method with.
			 * NOTE: We don't even bother checking if the projection matrix
			 *       ACTUALLY has been modified first, as it would take about
			 *       the same time to check it as simply overwriting it.
			 */
			glGetIntegerv( GL_MATRIX_MODE, &matrixMode );
			if ( matrixMode == GL_PROJECTION ) {
				glLoadMatrixd( origProjection );
			}
		}
	}

	/*
	 * W A R N I N G ! ! !
	 * The following is needed to keep OpenGL working in the case where glBegin
	 * is called but has not been followed through with a glEnd call.
	 * This happens for example, when in stepper mode and you have stepped past
	 * a glBegin but not yet to its glEnd call yet.
	 * It fixes the display and ALSO the ability to get the matrix values.
	 */
	if ( autoGLEnd ) {
		glEnd();
	}
}


/*!
 * The automatic glEnd() OpenGL call is needed to keep OpenGL working in the case
 * where glBegin is called but has not been followed through with a glEnd call.
 * This happens for example, when in stepper mode and you have stepped past a
 * glBegin but not yet to its glEnd call yet.
 *
 * It fixes the display and ALSO the ability to get the matrix values.
 *
 * <strong>Note:</strong> This does NOT call GLScene::exec(). That is to be done
 * by the viewer.
 */
void GLScene::setAutoGLEndEnabled( bool enable )
{
	autoGLEnd = enable;
}
