/****************************************************************************

 File:    supertable.cpp
 Created: by Aidan Lane, December 16, 2003
 Updated: by Aidan Lane, February 19, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include <qpainter.h>
#include <qapplication.h>

#include "supertable.h"


/*!
 * Constructor. This simply wrapps QTable's constructor that has the same
 * prototype.
 */
SuperTable::SuperTable( QWidget* parent, const char* name )
	: QTable( parent, name )
{}


/*!
 * Constructor. This simply wrapps QTable's constructor that has the same
 * prototype.
 */
SuperTable::SuperTable( int numRows, int numCols,
                        QWidget* parent, const char* name )
	: QTable( numRows, numCols, parent, name )
{}


/*!
 * Disable the Enter and Return keys, as to prevent moving the current row
 * selection to the next one when either of them are pressed.
 *
 * <strong>Note:</strong> We still call for focus to be transfered to the table.
 */
void SuperTable::keyPressEvent( QKeyEvent* event )
{
	if ( event->key() == Qt::Key_Enter || event->key() == Qt::Key_Return ) {
		setFocus();
		return;
	}

	QTable::keyPressEvent( event );
}


/*!
 * Change the background colour for a given row.
 *
 * If value row is out of bournds, then this method returns without doing anything.
 */
void SuperTable::setRowBackground( uint row, bool filled, QColor c )
{
	bool updateRow = false;

	// Catch out of bounds case
	// Note: it can't be -ve, as we require an unsigned integer
	if ( row > (uint)numRows()-1 ) {
		return;
	}

	if ( !filled && rowBkgrndColourMap.contains(row) ) {
		rowBkgrndColourMap.remove( row ); // clear existing setting for this row
		updateRow = true;
	}
	else if ( filled ) {
		rowBkgrndColourMap[row] = c;      // create an enry for this row
		updateRow = true;
	}

	if ( updateRow )
	{
		int i;
		for ( i=0; i < numCols(); ++i ) {
			updateCell( row, i );
		}
	}
}


/*!
 * This method allows you to set whether or not (individual) rows should LOOK
 * disabled. (Note the emphasis on LOOK, as the row won't actually be disabled
 * and hence you will still be able to click on it to select it!)
 */
void SuperTable::setRowDisabledLook( uint row, bool disabledLook )
{
	bool updateRow = false;

	// Catch out of bounds case
	// Note: it can't be -ve, as we require an unsigned integer
	if ( row > (uint)numRows()-1 ) {
		return;
	}

	if ( !disabledLook && rowDisabledLookMap.contains(row) ) {
		rowDisabledLookMap.remove( row ); // clear existing setting for this row
		updateRow = true;
	}
	else if ( disabledLook ) {
		rowDisabledLookMap[row] = true;   // create an entry for this row
		updateRow = true;
	}

	if ( updateRow )
	{
		int i;
		for ( i=0; i < numCols(); ++i ) {
			updateCell( row, i );
		}
	}
}


/*!
 * If a coloured row has been removed, then destroy alternative row background
 * colour.
 */
void SuperTable::setNumRows( int rows )
{
	int i;

	for ( i = rows-1; i < numRows(); i++ )
	{
		if ( rowBkgrndColourMap.contains(i) ) {
			rowBkgrndColourMap.remove( i );
		}

		if ( rowDisabledLookMap.contains(i) ) {
			rowDisabledLookMap.remove( i );
		}
	}

	QTable::setNumRows( rows );
}


/*!
 * Destroy alternative row background colour, as it (the row) is to be removed.
 */
void SuperTable::removeRow( int row )
{
	if ( rowBkgrndColourMap.contains(row) ) {
		rowBkgrndColourMap.remove( row );
	}

	if ( rowDisabledLookMap.contains(row) ) {
		rowDisabledLookMap.remove( row );
	}

	QTable::removeRow( row );
}


/*!
 * Same as the above, but can remove multiple rows at once.
 */
void SuperTable::removeRows( const QMemArray<int>& rows )
{
	QMemArray<int>::ConstIterator it;

	for ( it = rows.begin(); it != rows.end(); ++it )
	{
		if ( rowBkgrndColourMap.contains((*it)) ) {
			rowBkgrndColourMap.remove( (*it) );
		}

		if ( rowDisabledLookMap.contains((*it)) ) {
			rowDisabledLookMap.remove( (*it) );
		}
	}

	QTable::removeRows( rows );
}



/*!
 * (Re)Paint the table's cells with the correct background colours.
 */
void SuperTable::paintCell( QPainter* p, int row, int col, const QRect& cr,
                            bool selected, const QColorGroup& cg )
{
	QColorGroup* mod_cg = NULL;

	// Note: this MUST be peformed before the background is changed
	// (if necessary), as it would otherwise overwrite the background change.
	if ( rowDisabledLookMap.contains(row) ) {
		mod_cg = new QColorGroup( palette().disabled() );
	}

	if ( rowBkgrndColourMap.contains(row) ) {
		if ( mod_cg == NULL ) {
			mod_cg = new QColorGroup( cg );
		}
		mod_cg->setColor( QColorGroup::Base, rowBkgrndColourMap[row] );
	}

	// ONLY apply the modified colour group IF IT HAS ACTUALLY been modified
	if ( mod_cg != NULL ) {
		QTable::paintCell( p, row, col, cr, selected, *mod_cg );
	} else {
		QTable::paintCell( p, row, col, cr, selected, cg );
	}
}
