/****************************************************************************

 File:    varinstance.cpp
 Created: by Aidan Lane, January  08, 2004
 Updated: by Aidan Lane, February 26, 2004
 
 This file is part of Glitch
 Copyright (C) 2003-2004  Monash University, Clayton Campus, Australia
 Created by Aidan Lane, under the supervision of Jon McCormack.
 
 This program was developed to aid the students studying the CSE3313
 Computer Graphics course at Monash University.
 
 This software may contain portions that are copyright (C) 1993,
 Silicon Graphics, Inc. All Rights Reserved.
 
 Glitch is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, version 2.
 
 Glitch is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*****************************************************************************/

#include "varinstance.h"

#include <qapplication.h>
#include <qtooltip.h>
#include <qcombobox.h>

#include <typeinfo>


/*!
 * Constructor. Allows you to create a variable instance that can set the
 * element values to their defaults.
 */
VarInstance::VarInstance( const QString& name, CmdParam* cmdParam,
	bool autoInitElements )
{
	sharedInit();
	myVarName = name;

	setCmdParam( cmdParam, autoInitElements );
}


/*!
 * Constructor. Allows you to create a variable instance, where the element
 * values are set as specified by the elementValues parameter.
 *
 * <strong>Note:</strong> if |elementValues| != |elements|, then the
 * <em>elementValues</em> parameter will be ignored and the element values will
 * be set to their defaults.
 */
VarInstance::VarInstance( const QString& name, CmdParam* cmdParam,
	const QStringList& elementValues )
{
	sharedInit();
	myVarName = name;

	// DON'T GO ANY FURTHER IF THE COMMAND PARAM == NULL
	// THIS IS THE CUT OFF POINT
	if ( cmdParam == NULL ) {
		return;
	}

	setCmdParam( cmdParam, false );

	// Initialize the element values
	if ( elementValues.count() != (uint)cmdParam->numExtVarElements() ) {
		initElementValues();
	} else {
		// Ensure that the correct string representation is used for the values
		setElementValues( elementValues );
	}
}


/*!
 * Shared initializer for the class. This method contains code that must be
 * performed by all constructors.
 */
void VarInstance::sharedInit()
{
	varEditorBox = NULL;
	varFrontLabel = NULL;
	varEndLabel = NULL;
}


/*!
 * Set the command parameter that this variable instance is being used for.
 *
 * <strong>Note:</strong> This will return straight away if <em>cmd</em> = 0.
 */
void VarInstance::setCmdParam( CmdParam* cmdParam, bool autoInitElements )
{

	// DON'T GO ANY FURTHER IF THE COMMAND PARAM == NULL
	// THIS IS THE CUT OFF POINT
	if ( cmdParam == NULL
		|| cmdParam->numExtVarElements() <= 0 ) {
		return;
	}

	myCmdParam = cmdParam;

	// Initialize the element values to their defaults if required
	if ( autoInitElements ) {
		initElementValues();
	}

	// Get the type name of the elements
	myElementTypeName = myCmdParam->extVarElements()->first()->typeName();
}


/*!
 * Set the value(s) of the element(s) in this variable instance.
 *
 * This ensures that the correct string representation is used for the values.
 *
 * If <em>myCmdParam</em> == NULL, then this will return without doing anything.
 */
void VarInstance::setElementValues( const QStringList& values )
{
	// DON'T GO ANY FURTHER IF THE COMMAND PARAM == NULL
	if ( myCmdParam == NULL ) {
		return;
	}

	myElementValues.clear();

	QPtrList<CmdParam>* extVarElements = myCmdParam->extVarElements();
	if ( extVarElements != NULL )
	{
		// Initialize the variable instance's element values from the formal paremeter defaults
		uint i;
		for ( i=0; i < extVarElements->count(); ++i )
		{
			// NOTE: We do NOT dupport arrays of arrays!	
			myElementValues.append(
				extVarElements->at(i)->getStringRep(
					values[i] ) );
		}
	}
}


/*!
 * Initialize the elements to the defaults that are specified by the command
 * parameter.
 *
 * If myCmdParam is NULL, then this will NOT do anything.
 */
void VarInstance::initElementValues()
{
	if ( myCmdParam == NULL
		|| myCmdParam->numExtVarElements() <= 0 ) {
		return;	// no parameter specified, therefore we can't do anything
	}

	// Clear the current element string list, ready for new ones to be appended
	myElementValues.clear();

	// Build the string representation of the elements, IF THERE ARE ANY elements
	QPtrList<CmdParam>* extVarElements = myCmdParam->extVarElements();
	if ( extVarElements != NULL )
	{
		// Initialize the variable instance's element values from the formal paremeter defaults
		uint i;
		for ( i=0; i < extVarElements->count(); ++i )
		{
			// NOTE: We do NOT dupport arrays of arrays!	
			myElementValues.append(
				extVarElements->at(i)->getStringRep(
					extVarElements->at(i)->defaultValue() ) );
		}
	}
}


/*!
 * Produces a string/text representation of the variable instance.
 */
QString VarInstance::toString() const
{
	if ( myCmdParam->numExtVarElements() <= 0 ) {
		return QString::null; // no param specified, therefore we can't do anything
	}
	
	// NOTE! We just use the type name of the first element, as all of the
	// elements are meant to be of the same type (this IS listed already as as
	// assumption). The condition above enforces that atleast 1 element exists.
	QString str = QString( "%1 %2[] = { %3 };" )
	                  .arg( myElementTypeName )
					  .arg( myVarName )
					  .arg( myElementValues.join(", ") );
	return str;
}


/*!
 * Create an editor widget (which is a QHBox), which can be used to edit the
 * elements of the variable instance.
 */
QHBox* VarInstance::createEditorWidget( QWidget* parent )
{
	// Destroy any existing editor. (cmdEditorBox==NULL if not alive)
	delete varEditorBox;

	// We can't proceed if no parameter has been selected!
	if ( myCmdParam == NULL ) {
		return NULL;
	}

	varEditorBox = new QHBox( parent, "varEditorBox" );
	varFrontLabel = new QLabel(
	QString( "%1 %2[] = {" )
		.arg( myElementTypeName )
		.arg( myVarName ), varEditorBox, "varFrontLabel" );

	// Create the element editor/input widgets
	varElementWidgets.clear();
	QPtrList<CmdParam>* extVarElements = myCmdParam->extVarElements();
	if ( extVarElements != NULL )
	{
		uint i;

		// Create a suitable size policy that is to be used by the widgets
		QSizePolicy widgetSizePolicy( QSizePolicy::Expanding, 
		                              QSizePolicy::Expanding );
		widgetSizePolicy.setHorStretch( 3 );

		for ( i=0; i < extVarElements->count(); ++i )
		{
			CmdParam* element = extVarElements->at( i );

			QWidget* w = element->createWidget(
							varEditorBox, element->formalVarName(),
							this, SLOT(elementValueChanged()) );

			// Apply the custom size policy to the widget
			w->setSizePolicy( widgetSizePolicy );

			element->setWidgetValue( w, myElementValues[i] );
			varElementWidgets.append( w );
			// The formalVarName in this case should be an integer 0..|elements-1|
			QToolTip::add( w, 
				QString( "%1[%2]" )
					.arg(myVarName)
					.arg(element->formalVarName()) );
			w->setMaximumWidth( element->maxWidgetWidth() );
			w->show();
		}
	}

	varEndLabel = new QLabel( "<b>};</b>", varEditorBox, "varEndLabel" );

	// Ensure that varFrontLabel and varEndLabel both use the same font as the
	// application's default, except for being BOLD
	QFont f = QApplication::font();
	f.setBold( true );
	varFrontLabel->setFont( f );
	varEndLabel->setFont( f );

	// Ensure that varFrontLabel and varEndLabel does not try to `word-wrap'
	// their text and that they stay vertically centred
	varFrontLabel->setAlignment( Qt::SingleLine | Qt::AlignAuto | Qt::AlignVCenter );
	varEndLabel->setAlignment( Qt::SingleLine | Qt::AlignAuto | Qt::AlignVCenter );

	// Apply a suitable custom size policy to varEndLabel
	QSizePolicy sp( QSizePolicy::Expanding, QSizePolicy::Expanding );
	sp.setHorStretch( 1 );
	varEndLabel->setSizePolicy( sp );

	connect( varEditorBox, SIGNAL( destroyed() ),
	                       SLOT( editorWidgetCleanup() ) );

	return varEditorBox;
}


/*!
 * This method cleans up this command instance when it editor is destroyed.
 * This is called by cmdEditorBox's destroyed signal.
 */
void VarInstance::editorWidgetCleanup()
{
	// Delete all existing element widget ENTRIES.
	// Note that the actual elements are deleted with cmdEditorBox, so autoDelete
	// was NOT enabled, as the each element's contents is already gone.
	varElementWidgets.clear();

	if ( varEditorBox != NULL ) {
		varEditorBox->disconnect();     // disconnect editorWidgetCleanup slot
		//cmdSelectButton->disconnect();  // disconnect cmdSelectClicked slot
	}

	// Delete any existing editor, varEditorBox.
	// Note: this will automatically delete varFrontLabel and varEndLabel.
	varEditorBox = NULL;
}


/*!
 * This method is very useful, as it allows an update in the name of the
 * instance to be refected by the editor... if it exists.
 *
 * If varEditorBox == NULL, then this method returns without doing anything.
 *
 * <strong>Note:</strong> Currently, only the element type name and variable
 * name are actualy updated by this method. The reason for this is that they
 * are the only two that NEEDED to be updated (by an external object) in Glitch.
 */
void VarInstance::updateEditorWidget()
{
	if ( varEditorBox == NULL ) {
		return;
	}

	//FUTURE: update more properties!

	varFrontLabel->setText(
		QString( "%1 %2[] = {" )
			.arg( myElementTypeName )
			.arg( myVarName ) );
}



/*!
 * Called when one of the element input widget's value is changed.
 * It also emits the VarInstance::varElementChanged() signal just before it
 * returns.
 */
void VarInstance::elementValueChanged()
{
	// Extract values from ALL of the element widgets, as this method can
	// be called by ANY of these widgets on update.

	QPtrList<CmdParam>* extVarElements = NULL;

	if ( myCmdParam == NULL ) {
		return; // error
	}

	if ( (extVarElements = myCmdParam->extVarElements()) == NULL ) {
		return;	// error
	}

	if ( varElementWidgets.count() != extVarElements->count() ) {
		return; // error OR simply the widgets are still being created and their
		        // text is being set, hence calling this prematurely.
				// either way, we don't want to continue
	}

	// Get the new element values. Clear the existing ones first.
	myElementValues.clear();
	uint i;
	for ( i=0; i < varElementWidgets.count(); ++i )
	{
		// Get value from the element input widget
		if ( varElementWidgets.at(i) != NULL )
		{
			CmdParam* element = extVarElements->at(i);
			
			myElementValues.append(
				element->getStringRep(
					element->getWidgetValue(varElementWidgets.at(i)) ) );
		}
	}

	emit varElementChanged();
}


/*!
 * Set the background colour of the editor, that is if it is active.
 *
 * <strong>Note:</strong> There is a QtMac (Q_OS_MACX) specific section in this
 * method. It is used to set colours appropiately for specific widgets.
 */
void VarInstance::setEditorBackgroundColour( const QColor& colour )
{
	if ( varEditorBox == NULL ) {
		return;
	}
	
	varEditorBox->setPaletteBackgroundColor( colour );
	varFrontLabel->setPaletteBackgroundColor( colour );
	varEndLabel->setPaletteBackgroundColor( colour );

	// Make combo-boxes on the Mac look better, give them the background colour
	#if defined(Q_OS_MACX)
	uint i;
	QWidget* w;
	for ( i=0; i < varElementWidgets.count(); ++i )
	{
		w = varElementWidgets.at(i);
		if ( w != NULL
			&& typeid(*w) == typeid(QComboBox) )
		{
			w->setPaletteBackgroundColor( colour );
		}
	}
	#endif // defined(Q_OS_MACX)
}


